# @param container [String] The namespace for the service. This is the string used in the path to conf files for the service, and should not contain "pe-"
# @param service_name [String] The name of the service
# @param database_host [String] The hostname of where the database for this service lives
# @param database_name [String] The name of the PostgreSQL database
# @param database_user [String] The user used by the service to talk to the database (should have write permissions)
# @param database_migration_user [String] The user used by the service to perform migrations on service startup
# @param database_password [Optional[String]] If the database is using password authentication (not used by default), the password used by both the regular and migration users to access the database
# @param database_port [Variant[String, Integer]] The port to talk to the PostgreSQL instance. This is a variant since some users have specified the default as a string.
# @param database_properties [String] Additional properties to add on to the "database.subname" value
# @param group [String] The group to use for ownership of resources for this service
# @param user [String] The user to use for ownership of resources for this service
# @param ssl_listen_address [String] The network interface address used for SSL connections to the service.
# @param ssl_listen_port [Integer] The port used for SSL connections to the service.
# @param service_stop_retries Number of seconds to wait during shutdown after the TERM signal is sent before sending a KILL signal to the process.
# @param service_start_timeout Number of seconds to wait during startup for the service to finish booting before timing out.
# @param java_args [Hash] Java args to use for the service. Defaults to setting the heap size.
# @param flight_recorder_enabled [Boolean] Enable/disable Java Flight Recorder for all Java services
# @param flight_recorder_settings [String] Settings value for JFR. The 'default' setting is low overhead, 'profile' gives more data.
# @param flight_recorder_stackdepth [Integer] Number of stack frames to capture. Default is 128 to account for Clojure/JRuby's deeply nested function calls.
# @param flight_recorder_maxage [String] Maximum amount of time to keep recorded data for each service. Can be specified by s (seconds), m (minutes), h (hours), or d (days) suffixes.
# @param flight_recorder_maxsize [String] Maximum size of recorded data files for each service. Can be specified with k (kilobytes), m (megabytes) and g (gigabytes) suffixes.
# @param flight_recorder_repository [String] Directory to save JFR chunks while recording.
# @param flight_recorder_filename [String] File to save completed recording.
# @param disable_string_deduplication [Boolean] When using the default G1 garbage collector, this disables
#        the use of string deduplication for optimization.
# @param enable_gc_logging [Boolean] Enable the garbage collector log.
# @param host_data_log_path [String] Path to save host action log files to.
# @param purge_frequency_ms [Integer] Time in milliseconds between runs that purge old host data.
# @param max_host_data_age_months [Integer] Any host data older than this will be purged when purge is run.
# @param max_event_data_age_days [Integer] Events are collected into host data entries. Any events older than this will be purged when purge is run.
# @param ssl_dir [String] Path to directory the service will use to store the cert and key used by the service.
# @param disable_service [Boolean] Whether the service should be running or not. Should only run on the primary, not the replica
# @param replication_mode [Enum] One of "none", "source", or "replica", which indicates to the service
# @param puppetdb_host Array[String] The hostname running puppetdb.
# @param puppetdb_port Array[Integer] The port that puppetdb is listening on.

# what the app should expect when interacting with its database. Value must be the same as that provided to
# the database profile or app behavior will be incorrect.
class puppet_enterprise::master::host_action_collector(
  String $certname                                      = $facts['clientcert'],
  String $container                                     = 'host-action-collector',
  String $service_name                                  = 'pe-host-action-collector',
  String $database_host                                 = $puppet_enterprise::database_host,
  String $database_name                                 = $puppet_enterprise::host_action_collector_database_name,
  String $database_user                                 = $puppet_enterprise::host_action_collector_service_regular_db_user,
  String $database_migration_user                       = $puppet_enterprise::host_action_collector_service_migration_db_user,
  Optional[String] $database_password                   = $puppet_enterprise::hac_database_password,
  Variant[String, Integer] $database_port               = $puppet_enterprise::database_port,
  String $database_properties                           = $puppet_enterprise::database_properties,
  String $group                                         = 'pe-host-action-collector',
  String $user                                          = 'pe-host-action-collector',
  String $ssl_listen_address                            = $puppet_enterprise::params::ssl_address,
  Integer $ssl_listen_port                              = $puppet_enterprise::host_action_collector_port,
  Integer $service_stop_retries                         = 60,
  Integer $service_start_timeout                        = 300,
  Hash $java_args                                       = $puppet_enterprise::params::host_action_collector_java_args,
  Boolean $flight_recorder_enabled                      = $puppet_enterprise::flight_recorder_enabled,
  String $flight_recorder_settings                      = $puppet_enterprise::flight_recorder_settings,
  Integer $flight_recorder_stackdepth                   = $puppet_enterprise::flight_recorder_stackdepth,
  String $flight_recorder_maxage                        = $puppet_enterprise::flight_recorder_maxage,
  String $flight_recorder_maxsize                       = $puppet_enterprise::flight_recorder_maxsize,
  String $flight_recorder_repository                    = '/var/log/puppetlabs/host-action-collector/jfr',
  String $flight_recorder_filename                      = '/var/log/puppetlabs/host-action-collector/last_recording.jfr',
  Boolean $disable_string_deduplication                 = $puppet_enterprise::disable_string_deduplication,
  Boolean $enable_gc_logging                            = true,
  String $host_data_log_path                            = '/var/log/puppetlabs/host-action-collector/data',
  Integer $purge_frequency_ms                           = 1 * 24 * 60 * 60 * 1000,
  Integer $max_host_data_age_months                     = 2,
  Integer $max_event_data_age_days                      = 30,
  String $ssl_dir                                       = "${puppet_enterprise::server_data_dir}/host-action-collector/ssl",
  Boolean $disable_service                              = false,
  String $suite_license_path                            = '/etc/puppetlabs/suite-license.lic',
  String $license_key_path                              = $puppet_enterprise::params::dest_license_key_path,
  Puppet_enterprise::Replication_mode $replication_mode = 'none',
  Array[String]  $puppetdb_host                         = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_port                         = $puppet_enterprise::puppetdb_ports_array,
  String $inventory_host                                = $puppet_enterprise::console_host,
  Integer $inventory_port                               = $puppet_enterprise::orchestrator_port,
  String $inventory_url_prefix                          = $puppet_enterprise::inventory_url_prefix,
) inherits puppet_enterprise {
  include puppet_enterprise::packages

  $rootdir = "/etc/puppetlabs/${container}"
  $confdir  = "/etc/puppetlabs/${container}/conf.d"
  $webserver_conf = "${confdir}/webserver.conf"
  $service_conf = "${confdir}/pe-host-action-collector.conf"
  $license_service_conf = "${confdir}/license-service.conf"
  $local_ssl_cert = "${ssl_dir}/${certname}.cert.pem"
  $local_ssl_key = "${ssl_dir}/${certname}.private_key.pem"

  Pe_hocon_setting {
    ensure => present,
    notify => Service[$service_name],
    require => Package['pe-host-action-collector'],
  }

  File {
    ensure => present,
    owner  => $user,
    group  => $group,
    mode   => '0640',
    require => Package['pe-host-action-collector'],
  }

  file { $host_data_log_path:
    ensure  => directory,
    group   => $group,
    owner   => $user,
    mode    => '0750',
    require => Package['pe-host-action-collector'],
  }

  ### Make copy of cert/key in service ssl directory ###
  puppet_enterprise::certs { 'pe-host-action-collector::ssl':
    certname      => $certname,
    container     => $container,
    cert_dir      => $ssl_dir,
    append_ca     => false,
    make_pk8_cert => true,
    notify        => Service[$service_name],
    require => Package['pe-host-action-collector'],
  }

  ### global config ###
  puppet_enterprise::trapperkeeper::global_settings { $container :
    replication_mode => $replication_mode,
    require => Package['pe-host-action-collector'],
    certname    => $certname,
    ssl_cert    => $local_ssl_cert,
    ssl_key     => $local_ssl_key,
    localcacert => $puppet_enterprise::params::localcacert,
  }

  file { $webserver_conf :
    ensure => present,
  }

  ### Web router service setting ###
  pe_hocon_setting { 'web-router-service."puppetlabs.pe-host-action-collector.service/pe-host-action-collector-service"':
    path   => $webserver_conf,
    type   => 'hash',
    value  => { route => '/', server => 'pe-host-action-collector' },
  }

  pe_hocon_setting { 'web-router-service."puppetlabs.pe-host-action-collector.license-service/license-retrieval-service"':
    path   => $webserver_conf,
    type   => 'hash',
    value  => { route => '/license', server => 'pe-host-action-collector' },
  }

  pe_hocon_setting { 'host-action-collector: web-router-service."puppetlabs.trapperkeeper.services.metrics.metrics-service/metrics-webservice"':
    path    => $webserver_conf,
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.metrics.metrics-service/metrics-webservice"',
    type    => 'hash',
    value   => { route => '/metrics', server => 'pe-host-action-collector' },
  }

  pe_hocon_setting { 'host-action-collector: web-router-service."puppetlabs.trapperkeeper.services.status.status-service/status-service"':
    path => $webserver_conf,
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.status.status-service/status-service"',
    type => 'hash',
    value => { route => '/status', server => 'pe-host-action-collector' },
  }

  ### analytics client settings ###
  puppet_enterprise::trapperkeeper::analytics_client_settings { $container:
    container => $container,
    owner     => $user,
    notify    => Service[$service_name],
    require => Package['pe-host-action-collector'],
  }

  file { $service_conf:
    ensure => present,
  }

  ### rbac-consumer settings ###
  pe_hocon_setting { "${container}.rbac-consumer.api-url":
    path => $service_conf,
    setting => 'rbac-consumer.api-url',
    value => $puppet_enterprise::rbac_url,
  }

  ### pe-host-action-collector settings ###
  pe_hocon_setting{ 'pe-host-action-collector.day-entry':
    path => $service_conf,
    type => 'hash',
    value => {
      'host-day-purge-run-frequency-ms' => $purge_frequency_ms,
      'log-location' => $host_data_log_path,
      'max-host-day-age-in-months' => $max_host_data_age_months,
      'max-event-age-in-days' => $max_event_data_age_days,
    }
  }

  pe_hocon_setting{ 'pe-host-action-collector.inventory-url':
    path => $service_conf,
    value => "https://${inventory_host}:${inventory_port}${inventory_url_prefix}"
  }
  $first_puppetdb_host = $puppetdb_host[0]
  $first_puppetdb_port = pe_any2array($puppetdb_port)[0]
  pe_hocon_setting{ 'pe-host-action-collector.puppetdb-url':
    path => $service_conf,
    value => "https://${first_puppetdb_host}:${first_puppetdb_port}"
  }

  file { $license_service_conf:
    ensure => present,
  } 
  pe_hocon_setting{ 'license-retrieval-service.license-file-path':
    path => $license_service_conf,
    value => $suite_license_path,
  }

  pe_hocon_setting{ 'license-retrieval-service.yaml-license-file-path':
    path => $license_service_conf,
    value => $license_key_path,
  }

  ### Database settings ###
  if $puppet_enterprise::database_ssl and $puppet_enterprise::database_cert_auth {
    $client_pk8_key  = "${ssl_dir}/${certname}.private_key.pk8"
    $client_cert     = "${ssl_dir}/${certname}.cert.pem"
    $ssl_database_properties = "${database_properties}&sslkey=${client_pk8_key}&sslcert=${client_cert}"
  } else {
    $ssl_database_properties = $database_properties
  }

  puppet_enterprise::trapperkeeper::database_settings { $service_name:
    container           => $container,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    database_port       => Integer($database_port),
    database_properties => $ssl_database_properties,
    database_user       => $database_user,
    migration_user      => $database_migration_user,
    migration_password  => $database_password,
    group               => $group,
    user                => $user,
    notify              => Service[$service_name],
    require             => Package['pe-host-action-collector'],
  }

  ### Webserver settings in webserver.conf ###
  puppet_enterprise::trapperkeeper::webserver_settings { $service_name:
    container          => $container,
    ssl_listen_address => $ssl_listen_address,
    ssl_listen_port    => $ssl_listen_port,
    ssl_cert           => $local_ssl_cert,
    ssl_key            => $local_ssl_key,
    notify             => Service[$service_name],
    require            => Package['pe-host-action-collector'],
    access_log_config  => "/etc/puppetlabs/host-action-collector/request-logging.xml",
    client_auth        => "want",
  }

  ### Service definition ###
  puppet_enterprise::trapperkeeper::init_defaults { $container:
    user                 => $user,
    group                => $group,
    service_stop_retries => $service_stop_retries,
    start_timeout        => $service_start_timeout,
    notify               => Service[$service_name],
    require              => Package['pe-host-action-collector'],
  }

  $_service_ensure = $disable_service ? { false => 'running', true => 'stopped' }
  puppet_enterprise::trapperkeeper::pe_service { $container:
    ensure  => $_service_ensure,
    enable  => !$disable_service,
    require => Package['pe-host-action-collector'],
  }

  ### Bootstrap config ###
  # service => namespace
  $trapperkeeper_services = {
    'remote-activity-reporter'         => 'puppetlabs.rbac-client.services.activity',
    'remote-rbac-consumer-service'     => 'puppetlabs.rbac-client.services.rbac',
    'jetty10-service'                  => 'puppetlabs.trapperkeeper.services.webserver.jetty10-service',
    'webrouting-service'               => 'puppetlabs.trapperkeeper.services.webrouting.webrouting-service',
    'metrics-service'                  => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'metrics-webservice'               => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'scheduler-service'                => 'puppetlabs.trapperkeeper.services.scheduler.scheduler-service',
    'status-service'                   => 'puppetlabs.trapperkeeper.services.status.status-service',
    'pe-host-action-collector-service' => 'puppetlabs.pe-host-action-collector.service',
    'license-retrieval-service'        => 'puppetlabs.pe-host-action-collector.license-service',
  }

  $trapperkeeper_services.each |$service, $namespace| {
    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}: ${service}":
      container => $container,
      namespace => $namespace,
      service   => $service,
      notify    => Service[$service_name],
      require   => Package['pe-host-action-collector'],
    }
  }

  ### Java args ###
  if $flight_recorder_enabled {
    $_jfr_java_args = puppet_enterprise::jfr_java_args(
      $flight_recorder_repository,
      $flight_recorder_filename,
      $flight_recorder_settings,
      $flight_recorder_stackdepth,
      $flight_recorder_maxage,
      $flight_recorder_maxsize
    )
  } else {
    $_jfr_java_args = {}
  }
  $_java_args = pe_merge($java_args, $_jfr_java_args)
  puppet_enterprise::trapperkeeper::java_args { $container:
    java_args                    => $_java_args,
    disable_string_deduplication => $disable_string_deduplication,
    enable_gc_logging            => $enable_gc_logging,
    notify                       => Service[$service_name],
    require                      => Package['pe-host-action-collector'],
  }
}
