require 'puppet/indirector/face'
require 'puppet/util/colors'
require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/configure/postgres'
require 'puppet_x/puppetlabs/meep/configure/psql'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/apply'
require 'puppet_x/util/classification'
require 'puppet_x/util/classifier'
require 'puppet_x/util/interview'
require 'puppet_x/util/service_status'
require 'puppet_x/util/shell'
require 'puppet_x/util/stringformatter'

ReinitializeStatusTimeoutSeconds = 5

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend Puppet::Util::Colors
  extend PuppetX::Puppetlabs::Meep::Util
  extend PuppetX::Puppetlabs::Meep::Configure::Postgres

  action :reinitialize do
    summary _('Reinitialize replication')
    description _("Clear and re-synchronize replicated databases on the replica")

    arguments "<role>"

    option('-y', '--yes') do
      summary _("Answer yes to all confirmations.")
      default_to {nil}
    end

    option '--db DATABASE' do
      summary 'Reinitialize only this database'
      default_to { nil }
    end

    when_invoked do |role, options|
      if role != 'replica'
        msg = <<-HEREDOC
Usage:
  puppet infrastructure reinitialize replica
        HEREDOC
        puts(colorize(:red, msg))
        exit(1)
      end

      capture_cli_logs('reinitialize_replica')

      replica_certname = Puppet[:certname]

      config = PuppetX::Util::ServiceStatus.load_services_config()
      nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config()
      primary_master_node = nodes_config.find { |node| node[:role] == 'primary_master' }
      nc_service = config.find { |svc| svc[:type] == 'classifier' && svc[:node_certname] == primary_master_node[:certname] }

      # Bail if nc_service is nil
      if nc_service.nil?
        msg = PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
_("Could not find configuration for the Classifier terminus."),
_("Has this node been provisioned as a replica?"),
_("This command must only be run on a replica!")])
        Puppet.err(msg)
        exit(1)
      end

      nc = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      all_groups = nc.get_groups()

      # Is the current node even a replica?
      if !PuppetX::Util::Classifier.node_is_pinned_as_replica?(all_groups, replica_certname)
        Puppet.err(_("This command must only be run on a replica!"))
        exit(1)
      end

      # Warn the user if the primary is down
      primary = PuppetX::Util::Classification.find_primary_master(all_groups)
      primary_status = PuppetX::Util::ServiceStatus.services_for(primary, nil, config, ReinitializeStatusTimeoutSeconds, nodes_config: nodes_config)
      unless PuppetX::Util::ServiceStatus.all_services_running?(primary_status)
        not_running = PuppetX::Util::ServiceStatus.services_not_running(primary_status)
                        .map{|service| service[:display_name]}
                        .join(", ")
        services_warning = _("The following services on the primary are not running: %{services}") % {services: not_running}
        Puppet.warning(services_warning)
        if options[:yes].nil?
          unless PuppetX::Util::Interview.confirm(_("Are you sure you want to proceed?") + " [Y/n]:")
          end
        else
          Puppet.warning(_("%{warning}, continuing anyways due to `--yes` flag") % { warning: services_warning })
        end
      end

      # Get confirmation
      if options[:yes].nil?
        unless PuppetX::Util::Interview.confirm(PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
_("Reinitializing is a destructive operation: the existing databases will be dropped and re-created."),
_("Are you sure you want to proceed?") + " [y/N]:"]), false)
          exit(1)
        end
      end

      pe_infrastructure_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Infrastructure')
      puppet_enterprise = pe_infrastructure_group['classes']['puppet_enterprise']

      if options[:db]
        # Is the target_database_name specified, and is it the puppetdb_database_name?
        if (options[:db] == puppet_enterprise['puppetdb_database_name'])
          Puppet.err(_("The %{database} database cannot be reinitialized via this command.") % {database: target_database_name})
          exit(1)
        end
        databases = [options[:db]]
      else
        databases = [
          puppet_enterprise['classifier_database_name'] || 'pe-classifier',
          puppet_enterprise['rbac_database_name'] || 'pe-rbac',
          puppet_enterprise['activity_database_name'] || 'pe-activity',
          puppet_enterprise['orchestrator_database_name'] || 'pe-orchestrator',
          puppet_enterprise['inventory_database_name'] || 'pe-inventory',
          puppet_enterprise['host_action_collector_database_name'] || 'pe-hac',
        ]
      end

      # Detect if databases exist. A previous reinitialize cut short might have already
      # dropped the database, so we need to prevent the manifest from trying to act
      # on that database if that's true.
      #
      # Since we are doing the query on the local postgres on the replica, we don't
      # need most of the regular psql options.
      psql_opts = {:uid => 'pe-postgres'}
      valid_databases = []
      databases.each do |database|
        psql = PuppetX::Puppetlabs::Meep::Configure::PSQL.new(psql_opts)
        result = psql.run("SELECT 1 FROM pg_database WHERE datname='#{database}'")
        valid_databases << database if result == '1'
      end
      # We aren't in enterprise_tasks here, but this will allow us to not clobber another
      # PE infra management plan run
      enterprise_tasks_agent_lockfile = '/opt/puppetlabs/enterprise_tasks_agent_run.lock'
      # Run the reinitialize_replication.pp manifest
      reinitialization_status = with_agent_disabled('puppet infrastructure reinitialize replica in progress') do
        unless valid_databases.empty?
          puts
          Puppet.notice(_("Destroying replicated databases on %{certname}...") % {certname: replica_certname})
          puts

          manifest = <<-EOS
          class { 'puppet_enterprise':
            puppet_master_host => '#{replica_certname}',
          }
          class { 'pe_manager::reinitialize_replication':
            databases          => #{valid_databases},
          }
          EOS

          # Run the reinitialize_replication.pp manifest
          PuppetX::Util::Apply.run_puppet_apply(manifest)
          puts
          Puppet.info(_("Replicated databases have been destroyed"))
          puts
        end

        Puppet.notice(_("Running puppet to recreate replicated databases on %{certname}...") % {certname: replica_certname})
        puts

        # Run puppet normally
        PuppetX::Util::Shell.run_and_log("/opt/puppetlabs/bin/puppet agent -t --agent_disabled_lockfile #{enterprise_tasks_agent_lockfile}")
      end

      puts
      ## we expect Puppet to exit with its code for a successful run with changes
      if [0, 2].include?(reinitialization_status.exitstatus)
        Puppet.info(_("Replica successfully reinitialized"))
        exit 0
      else
        Puppet.err _("Replica reinitialization failed")
        exit(reinitialization_status.exitstatus)
      end
    end
  end
end
