require 'puppet_x/puppetlabs/meep/provision/base.rb'
require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/enable/action'
require 'puppet_x/puppetlabs/meep/enable/interview'
require 'puppet_x/util/service_status'
require 'pp'

module PuppetX
module Puppetlabs
module Meep
module Provision

  # Improved provising workflow relying on pg_basebackup and
  # orchestration of tasks and plans to speed up initial sync.
  class Streaming < PuppetX::Puppetlabs::Meep::Provision::Base
    include PuppetX::Puppetlabs::Meep::Infra::Lookup

    def self.pdb_replication_timeout
      600
    end

    # The hash containing infrastructure information taken either
    # from flags or from the interview
    attr_accessor :infra_conf

    def classify_replication(state)
      case state
      when :on
        enabled = true
        action  = 'Enabled'
      when :off
        enabled = false
        action  = 'Disabled'
      else raise(_("classify_replication() unable to classify to state: '%{state}'" % { state: state }))
      end

      all_groups = PuppetX::Util::Classifier.get_groups(classifier)
      pe_infra_group = PuppetX::Util::Classification.find_and_check_node_group(all_groups, 'PE Infrastructure', required_classes: ['puppet_enterprise'])
      classifier.update_group(
        {
          :id => pe_infra_group['id'],
          :classes => {
            'puppet_enterprise' => {
              'replicating' => enabled
            }
          }
        }
      )
      Puppet.notice(_("%{action} replication" % { action: action }))
    end

    def interview
      if options[:enable]
        self.infra_conf = PuppetX::Puppetlabs::Meep::Enable::Interview.enable_interview(replica_certname, primary_certname, classifier.get_groups, options)
      end
    end

    def pg_basebackup
      # Cleanup cert backups created by other tasks/plans
      pg_ver = platform_current_postgres_version
      Dir.glob("/opt/puppetlabs/server/data/postgresql/#{pg_ver}/data/certs_bak_*").each { |f| FileUtils.rm_rf f, :secure => true  }

      _invoke_task(
        'enterprise_tasks::pg_basebackup',
        nodes: [replica_certname],
        params: {
          'primary': primary_certname,
          'postgresql_version': pg_ver,
        }
      )
    end

    def reinitialize_replica
      _invoke_task('enterprise_tasks::reinitialize_replica', nodes: [replica_certname])
    end

    # After a successful provision, inform the user and provide
    # instructions for subsequently enabling.
    def closing_text
      Puppet.info(_("A replica has been provisioned."))
      Puppet.notice(
        _("Since the puppetdb database was streamed to the replica using the pg_basebackup utility, services should now be synced.")
      )
      Puppet.notice(
        _("To verify that synchronization is complete, enter the following command:\n") +
        "    puppet infrastructure status --host %{certname}" % { certname: replica_certname }
      )
      Puppet.notice(
        _("Once this is verified, enter the following command on the primary so that the replica will be ready to handle a failover from the primary.\n") +
        "    puppet infrastructure enable replica %{certname}" % { certname: replica_certname }
      )
      Puppet.notice(_("If you run this command before the replica has fully synchronized, the replica will not have all the necessary information to replace the primary during a failover."))
    end

    # After the replica is provisioned, services.conf changes to include
    # replica services, so we need to reload these. nodes_config does too,
    # but we aren't using it after provisioning, so saving time and not
    # reloading it here.
    def reload_config
      self.config = nil
      self.services = nil
      prep_config
    end

    def wait_for_replication
      Puppet.notice(_('Checking status of services on replica'))
      check_replication = ['classifier', 'rbac', 'activity']
      statuses = {}
      retries = 0
      max_retries = 5
      loop do
        if retries >= max_retries
          Puppet.err(_("Some services aren't running. Check 'puppet infrastructure status' for details. When all services are running, enable the replica with 'puppet infrastructure enable replica %{certname}'.") % {certname: replica_certname})
          exit(11)
        end
        statuses = PuppetX::Util::ServiceStatus.services_for(replica_certname, nil, config, 10, nodes_config: nodes_config)
        statuses.each { |status| Puppet.debug(status.pretty_inspect) }
        states = statuses.map { |status| status[:state] }
        break if states.all? { |state| state == :running }

        errors = statuses.select { |status| ! [:running, :starting].include?(status[:state]) }
        errored_services = errors.map { |status| status[:display_name] }
        if !errors.empty?
          Puppet.err(_("These services are not running: #{errored_services}\nCheck 'puppet infrastructure status' for details. When all services are running, enable the replica with 'puppet infrastructure enable replica %{certname}'.") % {certname: replica_certname})
          exit(12)
        end
        retries += 1
        Puppet.notice((_("Some services are still starting up, waiting and trying again. (Retry %{retries}/%{max_retries})") % {retries: retries, max_retries: max_retries}))
        Puppet.debug(_("Status: %{statuses}") % {statuses: statuses})
        sleep(5)
      end

      replication_statuses = statuses.select { |status| check_replication.include?(status[:type]) }
      replication_statuses.each do |status|
        if status[:status].nil? ||
          status[:status]['replication']['mode'] != 'replica' ||
          status[:status]['replication']['status'] != 'running'
          Puppet.err(_("%{service} replication is inactive. Check the logs for this service in /var/log/puppetlabs. When 'puppet infrastructure status' shows all services running and replication active and streaming, enable the replica with 'puppet infrastructure enable replica %{certname}.") % {service: status[:display_name], certname: replica_certname})
          Puppet.debug(_("Status: %{status}") % {status: status})
          exit(12)
        end
      end

      # Because we've just used pg_basebackup to set up the replica,
      # there really should not be much, if anything, to sync here.
      # Nevertheless, we'll wait 10 minutes and bail if it doesn't finish.
      Puppet.notice(_('Waiting for PuppetDB to finish initial sync'))
      timeout = PuppetX::Puppetlabs::Meep::Provision::Streaming::pdb_replication_timeout
      starttime = Time.now
      synced = false
      pdb_status = statuses.find{ |status| status[:type] == 'puppetdb' }
      while Time.now < starttime + timeout
        sync_status = pdb_status[:status]['sync_status']
        Puppet.debug(_('Sync status:'))
        Puppet.debug(sync_status.pretty_inspect)
        if sync_status['state'] && sync_status['state'] == 'error'
          Puppet.err(_("PuppetDB sync is in an errored state. Check /var/log/puppetlabs/puppetdb/puppetdb.log on the primary and replica for details. When PuppetDB syncing is fixed, enable the replica with 'puppet infrastructure enable replica %{certname}'.") % {certname: replica_certname})
          exit(12)
        end
        synced = sync_status.keys.include?('last_successful_sync')
        break if synced
        sleep(5)
        pdb_status = PuppetX::Util::ServiceStatus.services_for(replica_certname, 'puppetdb', config, 5, nodes_config: nodes_config).first
      end
      if !synced
        Puppet.err(_("Timed out waiting for PuppetDB syncing. Check /var/log/puppetlabs/puppetdb/puppetdb.log on the primary and replica for details. When PuppetDB syncing has completed, enable the replica with 'puppet infrastructure enable replica %{certname}'.") % {certname: replica_certname})
        exit(11)
      end
    end

    def enable_replica
      Puppet.notice(_('Enabling replica'))
      enable = PuppetX::Puppetlabs::Meep::Enable::Action.new(replica_certname, log_context, options,
        config: config,
        nodes_config: nodes_config,
        services: services,
        all_groups: classifier.get_groups,
        rbac_token: rbac_token,
        infra_conf: infra_conf,
        classifier_object: classifier
      )
      enable.run(skip_setup: true)
    end

    def run
      prep_config
      validate
      interview
      _with_agent_disabled_on([primary_certname, replica_certname]) do
        pin_to_pe_infra_agent
        classify
        classify_replication(:off)
        run_puppet
        pg_basebackup
        classify_replication(:on)
        run_puppet_on(primary_certname, description: 'third_primary_run_resync')
        reinitialize_replica
        unpin_from_pe_infra_agent
        if options[:enable]
          reload_config
          wait_for_replication
          enable_replica
        end
      end
      closing_text if !options[:enable]
      return nil
    end
  end

end
end
end
end
