require 'puppet'
require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/configure/validator'
require 'puppet_x/util/bolt'
require 'puppet_x/util/rbac'
require 'puppet_x/util/classification'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/service_status'
require 'puppet/util/pe_node_groups'

module PuppetX
  module Util
    module HA

      class AgentFactError < RuntimeError; end
      class PDBSyncError < RuntimeError; end

      def validate_and_get_info(host, log_context, options)
        state = {}
        rules = []

        state[:replica_host] = host
        state[:pe_postgresql_info] = pe_postgresql_info

        orch_service = PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator')
        rbac_token = PuppetX::Util::RBAC.load_token(options[:token_file])

        replica_host_info = get_postgres_info(host, orch_service, rbac_token, log_context, options)
        replica_postgresql_facts = replica_host_info['pe_postgresql_info']
        replica_pe_build = replica_host_info['pe_build_version']

        # installed_server_version is `nil` when postgres isn't installed on a given host
        if replica_postgresql_facts['installed_server_version']
          state[:replica_pg_installed] = true
          state[:replica_pg_version] = replica_postgresql_facts['installed_server_version']
        else
          state[:replica_pg_installed] = false
          state[:replica_pg_version] = nil
        end

        # Skip node validation when --force is used
        unless options[:force]
          rules << PuppetX::Puppetlabs::Meep::Configure::ReplicaUpgrade
          validator = PuppetX::Puppetlabs::Meep::Configure::Validator.new(state)
          validator.register(rules)

          if !validator.valid?
            validator.log_and_fail!
          end
        end
        return replica_pe_build, replica_postgresql_facts
      end

      def get_postgres_info(host, orch_service, rbac_token, log_context, options)
        env = options[:pe_environment]
        task = 'pe_install::get_postgresql_info'
        result_array = PuppetX::Util::Orchestrator.run_task(
            orch_service,
            scope: { 'nodes': [host] },
            rbac_token: rbac_token,
            task: task,
            params: {},
            environment: env,
            log_context: log_context,
          )
        result = result_array[0]

        if result['state'] != 'finished'
          raise AgentFactError, _("Failed to gather PostgreSQL info from host %{host}. Please make sure %{host} is reachable from the primary via the Orchestrator, has Puppet installed, and is a replica.") % {host: host}
        end

        result['result']
      end

      def wait_for_pdb_sync(replica_certname)
        services_config = PuppetX::Util::ServiceStatus.load_services_config
        nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config
        done = false
        max_wait = 950
        start_time = Time.now
        Puppet.notice _("Checking puppetdb sync status on replica node")
        while !done && Time.now - start_time < max_wait
          pdb_status = PuppetX::Util::ServiceStatus.services_for(replica_certname, 'puppetdb', services_config, 10, nodes_config: nodes_config).first
          raise PDBSyncError, _("PuppetDB in error state") if pdb_status[:state] == :error
          sync_status = pdb_status[:status]['sync_status']
          Puppet.debug(_('Sync status:'))
          Puppet.debug(sync_status.pretty_inspect)
          done = sync_status.keys.include?('last_successful_sync')
          unless done
            sleep(1)
          end
        end
        done
      end

      def self.get_host_agent_version(host, orch_service, rbac_token, environment, log_context)
        replica_agent_version_result = PuppetX::Util::Orchestrator.run_task(
          orch_service,
          scope: { 'nodes': [host] },
          rbac_token: rbac_token,
          task: 'enterprise_tasks::get_agent_version',
          params: {},
          environment: environment,
          log_context: log_context,
        ).first()['result']

        if replica_agent_version_result.has_key?('version') && replica_agent_version_result['version'] != 'null'
          return replica_agent_version_result['version']
        else
          raise AgentFactError, _("Failed to gather facts from host %{host}. Please make sure %{host} is reachable from the primary and has Puppet installed.") % {host: host}
        end
      end
    end
  end
end
