plan enterprise_tasks::upgrade_and_migrate_replica(
  TargetSpec $replica,
  String $old_postgres_version,
  String $new_postgres_version,
  Integer $pg_upgrade_timeout   = 0,
  Optional[TargetSpec] $primary = 'localhost',
  Optional[Boolean] $force      = false,
) {
  $constants = constants()
  enterprise_tasks::test_connection([$replica, $primary])
  $status_hash = run_plan(enterprise_tasks::get_service_status, target => $replica,
    service    => 'puppet',
  )

  enterprise_tasks::verify_node($primary, 'primary', $force)
  enterprise_tasks::verify_node($replica, 'replica', $force)

  $result_or_error = catch_errors() || {
    enterprise_tasks::with_agent_disabled([$replica]) || {
      $replica_target = get_targets($replica)[0]
      $primary_target = get_targets($primary)[0]

      enterprise_tasks::message('upgrade_and_migrate_replica', 'Establish primary and replica certnames.')
      $primary_certname = enterprise_tasks::get_certname($primary)
      $replica_certname = enterprise_tasks::get_certname($replica)

      enterprise_tasks::message('upgrade_and_migrate_replica', 'Looking up facts.')
      enterprise_tasks::get_facts_on($replica_target)
      enterprise_tasks::get_facts_on($primary_target)
      $pe_ver = $primary_target.facts['pe_build']
      $original_replica_pe_ver = $replica_target.facts['pe_build']
      $original_agent_ver = $replica_target.facts['aio_agent_version']

      $lookup_results = run_task('enterprise_tasks::get_enterprise_data', $primary_target, 'parse' => true)
      $enterprise_data = $lookup_results.first()['enterprise_data']
      $user_data_conf = enterprise_tasks::first_defined(
        $enterprise_data["/etc/puppetlabs/enterprise/conf.d/nodes/${primary_certname}.conf"],
        $enterprise_data['/etc/puppetlabs/enterprise/conf.d/user_data.conf'],
        {}
      )

      $pglogical_databases = [
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::activity_database_name'], 'pe-activity'),
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::classifier_database_name'], 'pe-classifier'),
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::inventory_database_name'], 'pe-inventory'),
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::orchestrator_database_name'], 'pe-orchestrator'),
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::rbac_database_name'], 'pe-rbac'),
        enterprise_tasks::first_defined($user_data_conf['puppet_enterprise::host_action_collector_database_name'], 'pe-hac'),
      ]

      enterprise_tasks::message('upgrade_and_migrate_replica', 'Obtaining list of replica databases.')
      $database_list_output = run_command('su -s /bin/bash -c "/opt/puppetlabs/server/bin/psql -qt -c \'select datname from pg_database;\'" pe-postgres', $replica_target).first['stdout']
      $existing_databases = $database_list_output.split("\n").map |$i| { $i.strip }
      $existing_pglogical_databases = $pglogical_databases.filter |$d| { $d in $existing_databases }
      debug("pglogical_databases: ${pglogical_databases}")
      debug("existing_databases: ${existing_databases}")
      debug("existing_pglogical_databases: ${existing_pglogical_databases}")

      ##########################################
      # Stop services
      #
      # Services need to be stopped except for pe-postgresql, since we need it up
      # to do the migration.
      $services_list = $constants['pe_services'] + ['puppet'] + $constants['db_timers'] + $constants['db_services']
      $services_list.each |$service| {
        $ensure = $service == 'pe-postgresql' ? {
          true => 'running',
          default => 'stopped',
        }
        run_command("${constants['puppet_bin']} resource service ${service} ensure=${ensure}", $replica)
      }

      ##########################################
      # Import GPG key
      #
      # When upgrading from older PE versions that did not include the 
      # GPG-KEY-puppet-2025-04-06 key, we need to import the key from
      # the primary before updating the repo that is now signed with
      # this newer key. For RHEL-based platforms, this is already 
      # specified in the repo config and downloaded by yum/dnf
      # automatically. I believe Zypper picks it up from the repo
      # itself. For Ubuntu, we need to do this manually.
      if $replica_target.facts['os']['family'].downcase == 'debian' {
        enterprise_tasks::message('upgrade_and_migrate_replica', 'Importing GPG-KEY-puppet')
        $temp_file = run_command('mktemp', $replica_target).first['stdout'].chomp
        apply($replica_target) {
          file { $temp_file:
            source => 'puppet:///modules/pe_repo/GPG-KEY-puppet',
          }
          -> exec { 'import key':
            command => "/usr/bin/env apt-key add ${temp_file}",
          }
        }
        apply($replica_target) {
          file { $temp_file:
            ensure => absent,
            force  => true,
          }
        }
      }

      ##########################################
      # Update puppet_enterprise repo
      #
      # We need to update the puppet_enterprise package repo to point
      # to the new version on the primary so that we can upgrade
      # the agent.
      enterprise_tasks::message('upgrade_and_migrate_replica', 'Updating puppet_enterprise repo configuration')
      $repo_update_result = apply($replica_target, '_catch_errors' => true) {
        class { 'puppet_enterprise':
          puppet_master_host => $primary_certname,
        }

        class { 'puppet_enterprise::repo':
          pe_ver   => $pe_ver,
          manage   => true,
          certname => $replica_certname,
        }
      }.first()
      enterprise_tasks::apply_report('upgrade_and_migrate_replica', $repo_update_result)
      if !$repo_update_result.ok {
        fail_plan($repo_update_result.error())
      }

      ##########################################
      # Drop the pglogical replicated databases.
      #
      # Instead of migrating them with pg_upgrade, we drop them
      # and allow replication to regenerate them. This should be sufficiently fast
      # for these databases which are smaller than pe-puppetdb. We do want to use
      # pg_upgrade to migrate puppetdb, however, as this should be faster than
      # letting puppetdb sync as if we were reprovisioning from scratch.
      #
      # The reason we can't migrate the rest of the databases using pg_upgrade comes
      # back to how schema migrations are handled for the databases pglogical is handling.
      # The migrations are forwarded by the provider db as each tk service uses
      # [jdbc-utils](https://github.com/puppetlabs/jdbc-util/blob/master/src/puppetlabs/jdbc_util/pglogical.clj#L27)
      # which applies the individual ddl changes via pglogical.replicate_ddl_command().
      #
      # This means that when the primary services upgrade their schema, normally
      # the replica's schema is upgraded at the same time. (This has its own
      # problem in that the services on the replica have not yet been upgraded
      # and may not be able to handle the migration changes, but that's a
      # different issue).
      #
      # However, pg_upgrade run on the primary does not migrate pg_replication_slots,
      # which breaks replication between the primary and the replica so schema does
      # not end up transferring.
      #
      # If we then migrate the pglogical databases on the replica with pg_upgrade,
      # they never get the missing migrations because the replica services hang
      # waiting for schema changes from the provider rather than applying the migrations
      # themselves. Consequently the services don't end up initializing properly if
      # there were any schema migrations pending.
      #
      # Because pe_manager is not available to the orchestrator in the base modulepath,
      # we recreate some of pe_manager::reinitialize_replication here.
      #
      # We drop pg_repack here rather than in the pe_install::upgrade::postgres class,
      # since we are also dropping most of the databases in this section,
      # and to avoid the other pieces that section of the upgrade::postgres
      # class depends on (namely, stop_services). We only need to drop pg_repack on pe-puppetdb
      # since that's all that will be left.

      enterprise_tasks::message('upgrade_and_migrate_replica', 'Dropping the pglogical replicated databases on the replica, and pg_repack on pe-puppetdb.')
      $drop_replicas_result = apply($replica_target, '_catch_errors' => true) {
        class { 'puppet_enterprise':
          puppet_master_host => $primary_certname,
        }

        class { 'pe_postgresql::globals':
          user      => 'pe-postgres',
          group     => 'pe-postgres',
          psql_path => "${puppet_enterprise::server_bin_dir}/psql",
        }

        include pe_postgresql::params

        puppet_enterprise::psql { 'Drop pg_repack on pe-puppetdb prior to migration':
          db      => 'pe-puppetdb',
          command => 'DROP EXTENSION IF EXISTS "pg_repack" CASCADE',
        }

        $subscription_name = sprintf('s%.14s', sha1($primary_certname))
        $existing_pglogical_databases.map |$database_name| {
          # remove pglogical subscriptions and then pglogical itself
          puppet_enterprise::pg::pglogical::subscription { "${database_name}/${subscription_name}":
            ensure            => absent,
            subscription_name => $subscription_name,
            database          => $database_name,
          }

          -> puppet_enterprise::psql { "drop pglogical extension for ${database_name} sql":
            db      => $database_name,
            command => 'drop extension pglogical cascade',
          }

          # kill potentially leftover connections
          -> puppet_enterprise::psql { "kill pglogical connections for ${database_name} sql":
            db      => 'pe-postgres',
            command => "SELECT pg_terminate_backend(pg_stat_activity.pid)
                              FROM pg_stat_activity
                              WHERE pg_stat_activity.datname = '${database_name}'",
          }

          # drop the database
          -> puppet_enterprise::psql { "drop ${database_name} sql":
            db      => 'pe-postgres',
            command => "DROP DATABASE IF EXISTS \"${database_name}\"",
          }
        }
      }.first()
      enterprise_tasks::apply_report('upgrade_and_migrate_replica', $drop_replicas_result)
      if !$drop_replicas_result.ok {
        fail_plan($drop_replicas_result.error())
      }

      ##########################
      # Migrate database cluster
      #
      # This will also upgrade the agent
      enterprise_tasks::message('upgrade_and_migrate_replica', "Migrating the pe-postgresql database cluster from ${old_postgres_version} to ${new_postgres_version}.")
      enterprise_tasks::message('upgrade_and_migrate_replica', 'NOTE: It may take considerable time for pg_upgrade to complete depending on the size of the pe-puppetdb database.')

      $migrate_pg_result = apply($replica_target, '_catch_errors' => true) {
        class { 'puppet_enterprise':
          puppet_master_host => $primary_certname,
        }
        include 'puppet_enterprise::params'

        class { 'puppet_enterprise::packages':
          pe_ver => $pe_ver,
        }
        contain 'puppet_enterprise::packages'

        $install_options = $facts['os']['family'].downcase == 'debian' ? {
          true  => ['-o', 'Dpkg::Options::=--force-overwrite'],
          false => undef,
        }

        package { 'puppet-agent':
          ensure          => latest,
          install_options => $install_options,
          before          => Class['pe_install::upgrade::postgres'],
        }

        # Older versions of pe-installer put parts of enterprise_tasks in the base modulepath,
        # which is now handled by pe-modules. To avoid conflicts, we need to update pe-installer
        # before pe-modules. We can remove this when we no longer need to upgrade from 2019.8.
        #
        # This same logic is in the puppet_enterprise::profile::master class, but since we're
        # updating packages before that gets applied in order to migrate to the new postgres,
        # we need to include it here.
        Package['pe-installer'] -> Package['pe-modules']

        # Migrate the database.
        #
        # Essentially, migrate pe-puppetdb. The rest of the databases will be initialized
        # from scratch, and then will replicate.
        class { 'pe_install::upgrade::postgres':
          old_postgres_version => $old_postgres_version,
          new_postgres_version => $new_postgres_version,
          pg_upgrade_timeout   => $pg_upgrade_timeout,
          replica_migration    => true,
          require              => Class['puppet_enterprise::repo'],
        }

        # Refresh the database profile
        #
        # Completes the rest of the datbase configuration, and sets up
        # replication for the databases replicated via pglogical.
        #
        # They will then replicate asynchronously, so we will need to
        # poll for their status later.
        class { 'puppet_enterprise::profile::database':
          certname                    => $replica_certname,
          replication_source_hostname => $primary_certname,
          replication_mode            => 'replica',
        }
      }.first()
      enterprise_tasks::apply_report('upgrade_and_migrate_replica', $migrate_pg_result, false)

      if !$migrate_pg_result.ok() {
        # rollback to a state that we can re-run the migration from
        enterprise_tasks::message('upgrade_and_migrate_replica', '!!! -----------------------')
        enterprise_tasks::message('upgrade_and_migrate_replica', "Migration Failed! Rolling back to pe-postgresql ${old_postgres_version}")
        $new_package = "pe-postgresql${new_postgres_version}"
        $new_data_dir = "/opt/puppetlabs/server/data/postgresql/${new_postgres_version}/data"
        $new_tablespaces = "/opt/puppetlabs/server/data/postgresql/{activity,classifier,inventory,orchestrator,puppetdb,rbac}/PG_${new_postgres_version}*"

        enterprise_tasks::message('upgrade_and_migrate_replica', "Removing ${new_package} packages, data, and tablespaces, and rolling back the puppet_enterprise repo and puppet agent")
        $rollback_result = apply($replica_target, '_catch_errors' => true) {
          class { 'puppet_enterprise':
            puppet_master_host => $primary_certname,
          }

          class { 'puppet_enterprise::repo' :
            pe_ver   => $original_replica_pe_ver,
            manage   => true,
            certname => $replica_certname,
            before   => Package['puppet-agent'],
          }
          contain 'puppet_enterprise::repo'

          $install_options = $facts['os']['family'].downcase == 'suse' ? {
            true  => ['--oldpackage'],
            false => undef,
          }
          $rollback_agent_ver = $facts['os']['family'].downcase == 'debian' ? {
            true  => "${original_agent_ver}*",
            false => $original_agent_ver,
          }
          package { 'puppet-agent':
            ensure          => $rollback_agent_ver,
            install_options => $install_options,
          }

          package { ["${new_package}-pglogical","${new_package}-pgrepack"]:
            ensure => absent,
          }
          -> package { ["${new_package}-contrib","${new_package}-server"]:
            ensure  => absent,
          }
          -> package { $new_package:
            ensure => absent,
          }
          -> exec { "Remove ${new_package} datadir at ${new_data_dir}":
            command => "rm -rf ${new_data_dir}",
            path    => '/bin:/sbin:/usr/sbin',
          }
          -> exec { "Remove ${new_package} tablespaces at ${new_tablespaces}":
            command => "rm -rf ${new_tablespaces}",
            path    => '/bin:/sbin:/usr/sbin',
          }
          -> service { 'pe-postgresql':
            ensure => running,
          }
        }.first()
        # Report on rollback attempt
        enterprise_tasks::apply_report('upgrade_and_migrate_replica', $rollback_result, false)
        if $rollback_result.ok() {
          enterprise_tasks::message('upgrade_and_migrate_replica', 'Rollback succeeded.')
        } else {
          enterprise_tasks::message('upgrade_and_migrate_replica', "${rollback_result.error()}")
          enterprise_tasks::message('upgrade_and_migrate_replica', " *** Rollback failed! Note that if the agent failed to roll back, any puppet runs with the newer agent can cause problems. Roll the agent back to ${original_agent_ver} manually.")
        }
        enterprise_tasks::message('upgrade_and_migrate_replica', '!!! -----------------------')

        # But fail with the migration error.
        fail_plan($migrate_pg_result.error())
      } else {
        # We upgraded the agent, so restart pxp-agent in case it's needed
        run_task(service, $replica, action => 'restart', name => 'pxp-agent')
        wait_until_available($replica)

        # Upgrade everything else
        enterprise_tasks::message('upgrade_and_migrate_replica', 'Running puppet to complete upgrade')
        run_task('enterprise_tasks::run_puppet', $replica,
          env_vars => { 'FACTER_pe_build' => $pe_ver },
        )

        # Make sure packages upgraded
        $primary_server_pe_modules_version = run_task(enterprise_tasks::get_package_version, $primary, package => 'pe-modules').first().value()['version']
        $secondary_pe_modules_version = run_task(enterprise_tasks::get_package_version, $replica, package => 'pe-modules').first().value()['version']
        if $primary_server_pe_modules_version != $secondary_pe_modules_version {
          enterprise_tasks::message('upgrade_and_migrate_replica',"ERROR: The target did not have the expected pe-modules version, had ${secondary_pe_modules_version} expected to match the primary server's version of ${primary_server_pe_modules_version}" )
          fail_plan("Failed on ${replica}", 'Target did not have an upgraded pe-modules package')
        }
      }
    }
  }

  enterprise_tasks::message('upgrade_and_migrate_replica','Applying original puppet service state...')
  run_command("${constants['puppet_bin']} resource service puppet ensure=${status_hash[status]} enable=${status_hash[enabled]}", $replica)
  if $result_or_error =~ Error {
    fail_plan($result_or_error)
  }
}
