#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require_relative '../lib/enterprise_tasks/puppet_helper.rb'
require 'open3'
require 'time'

class DisableAgent < EnterpriseTaskHelper
  include EnterpriseTasks::PuppetHelper

  def task(agent_lock_timeout: 300, **_kwargs)
    certname_cmd = [puppet_bin, 'config', 'print', 'certname']
    certname = Open3.capture2e(*certname_cmd)[0].strip
    run_lockfile_cmd = [puppet_bin, 'config', 'print', 'agent_catalog_run_lockfile']
    run_lockfile = Open3.capture2e(*run_lockfile_cmd)[0].strip
    disabled_lockfile_cmd = [puppet_bin, 'config', 'print', 'agent_disabled_lockfile']
    disabled_lockfile = Open3.capture2e(*disabled_lockfile_cmd)[0].strip

    # Disable the agent first so that we don't get a puppet run popping in after we
    # wait for the run lock to clear.
    already_disabled = File.exist?(disabled_lockfile)
    unless already_disabled
      disable_cmd = [puppet_bin, 'agent', '--disable', 'Puppet_infrastructure_management_plan_running']
      output, status = Open3.capture2e(*disable_cmd)
      raise EnterpriseTaskHelper::Error.new("Could not disable puppet agent on #{certname}", 'puppetlabs.disable-agent/disable-agent-failure', 'output' => output) unless status.exitstatus.zero?
    end

    # Wait for any existing run lock to clear.
    starttime = Time.now
    while File.exist?(run_lockfile) && Time.now < starttime + agent_lock_timeout
      $stderr.puts "#{run_lockfile} exists. Waiting for agent lock to clear."
      sleep(5)
    end
    raise EnterpriseTaskHelper::Error.new("Agent lock did not clear before timeout of #{agent_lock_timeout} seconds on #{certname}.", 'puppetlabs.disable-agent/agent-lock-timeout') if File.exist?(run_lockfile)

    { already_disabled: already_disabled }
  end
end

DisableAgent.run if __FILE__ == $PROGRAM_NAME
