require 'puppet/indirector/face'
require 'puppet_x/puppetlabs/meep/infra/feature_flags'
require 'puppet_x/puppetlabs/meep/infra/plan_executor'
require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/docs'
require 'puppet_x/util/ha'
require 'puppet_x/util/lei'
require 'puppet_x/util/infrastructure_error'
require 'puppet_x/util/service_status'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend PuppetX::Util::HA
  extend PuppetX::Util::LEI
  extend PuppetX::Puppetlabs::Meep::Infra::FeatureFlags
  extend PuppetX::Puppetlabs::Meep::Infra::Lookup
  extend PuppetX::Puppetlabs::Meep::Util

  action :upgrade do
    summary _('Upgrade a PE secondary infrastructure node.')
    description(
      _("Upgrades a secondary infrastructure node to the primary\'s current version.\n") +
      _("Allows for the upgrading of multiple compilers with comma separated certname values.\n") +
      _("Known roles:\n") +
      _("replica\n") +
      _("compiler\n")
    )

    arguments "<role> <node_certname>"

    option('--all') do
      summary _('Targets all nodes of role for upgrade. Only works for compiler role.')
    end

    option('--pe-environment ENVIRONMENT') do
      summary _('The PE Environment to be used when upgrading')
      default_to { 'production' }
    end

    option("--token-file PATH") do
      summary _("Path to a token file")
      description <<-EOT
        Path to an RBAC token to use for authentication when upgrading a secondary infrastructure node.
      EOT
    end

    option("--force") do
      summary _("Skip node type verification in underlying plans.")
      default_to {false}
      description <<-EOT
      Skip node type verification when upgrading a replica or compiler.
      Ensure the node you are providing to the command is the correct
      type before using this flag.
      EOT
    end

    option("--only-recreate-databases") do
      summary _("Recreate databases on replica after a pg_basebackup failure during upgrade. CAUTION: Only use if a previous failed replica upgrade directs you to do so.")
      default_to {false}
      description <<-EOT
      Perform a pg_basebackup of all databases from primary to replica.
      CAUTION: Only use this flag if a previous failed replica upgrade directs you to do so.
      EOT
    end

    when_invoked do |role, *args|
      options = args.pop
      if !args.empty?
        certname = args[0]
      elsif !options[:all]
        Puppet.err(_("Missing target node. Please provide a target node to be upgraded"))
        exit(1)
      end

      known_roles = ['replica', 'compiler']
      if !known_roles.include?(role)
        raise PuppetX::Util::InfrastructureUnknownRoleError.new(role, known_roles) # rubocop:disable GetText/DecorateFunctionMessage
      end

      log_context = capture_cli_logs("upgrade_#{role}")

      case role
      when 'replica'
        if options[:all]
          Puppet.err(_("--all is not a usable option for upgrading replicas."))
          exit(1)
        end

        _, replica_postgres_info = validate_and_get_info(certname, log_context, options)
        primary_pg_ver = platform_current_postgres_version
        replica_pg_ver = replica_postgres_info['installed_server_version']

        # Before upgrade, cleanup old certs backup dirs generated by cert regen plans.
        # Incorrect permissions on these backup dirs will cause pg_basebackup failure
        delete_certs_backup(primary_pg_ver)

        if options[:only_recreate_databases]
          # TODO: Fix this to just be able to run enterprise_tasks::reprovision_replica
          upgrade_and_reprovision_replica(role, certname, log_context, options)
        elsif replica_pg_ver != primary_pg_ver
          upgrade_and_migrate_replica(certname, replica_pg_ver, primary_pg_ver, log_context, options)
        else
          upgrade_secondary(role, certname, log_context, options)
        end
        
        config = PuppetX::Util::ServiceStatus.load_services_config()
        nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config()
        replica_status = PuppetX::Util::ServiceStatus.services_for(certname, nil, config, ReinitializeStatusTimeoutSeconds, nodes_config: nodes_config)
        unless PuppetX::Util::ServiceStatus.all_services_running?(replica_status)
          Puppet.notice(_("Reinitializing replica..."))
          rbac_token = PuppetX::Util::RBAC.load_token(options[:token_file]) # nil will use default path
          orch_service = PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator')
          result_array = PuppetX::Util::Orchestrator.run_task(
            orch_service,
            scope: { 'nodes': [certname] },
            rbac_token: rbac_token,
            task: 'enterprise_tasks::reinitialize_replica',
            params: { },
            environment: options[:pe_environment],
            log_context: log_context,
          )
          result_state = result_array[0]['state']
          if result_state != 'finished'
            Puppet.err(_("We had trouble automatically reinitializing the replica after upgrade. If 'puppet infrastructure status' shows errors, run 'puppet infrastructure reinitialize replica' on %{certname}.") % { certname: certname })
          end
        end
      when 'compiler'
        if options[:all]
          compilers = get_nodes_with_role('compile_master') + get_nodes_with_role('pe_compiler')
          certname = compilers.join(',')
          if certname.empty?
            Puppet.err(_("No known compilers found."))
            exit(1)
          end
        end
        targets = certname.split(',')
        targets.each do |target|
          exit(1) unless validate_compiler(target, options)
        end

        # upgrade all compilers at once, then output warnings for legacy compilers if necessary
        upgrade_secondary(role, targets, log_context, options)
        legacy_compilers = targets.select { |target| get_compiler_type(target) == 'compiler' }
        if !legacy_compilers.empty?
          convert_compiler_link = PuppetX::Util::Docs.link(:CONVERT_LEGACY_COMPILER_LINK)
          if legacy_compilers.length == 1
            message = "%{host} is a legacy compiler. In most cases, converting to a PE Compiler with the PuppetDB service improves scalability. In geo-diverse installations, you might experience better PuppetDB performance with a legacy compiler." % { host: legacy_compilers[0] }
          else
            compiler_string = legacy_compilers.join(',')
            message = "%{hosts} are legacy compilers. In most cases, converting to PE Compilers with the PuppetDB service improves scalability. In geo-diverse installations, you might experience better PuppetDB performance with a legacy compiler." % { hosts: compiler_string }
          end
          Puppet.notice(_("%{message} Please see %{convert_compiler_link} for more details.") % { message: message, convert_compiler_link: convert_compiler_link })
        end
      end

      # Explicit return of nil prevents accidental Puppet::Application::FaceBase.render().
      return
    end
  end

  def delete_certs_backup(pg_ver = platform_current_postgres_version)
    Dir.glob("/opt/puppetlabs/server/data/postgresql/#{pg_ver}/data/certs_bak_*").each { |f| FileUtils.rm_rf f, :secure => true  }
  end

  def upgrade_and_reprovision_replica(role, certname, log_context, options)
    params = {
      'replica' => certname,
      'primary' => Puppet[:certname],
      'force'   => options[:force].to_s,
      'only_recreate_databases' => options[:only_recreate_databases].to_s,
    }

    Puppet.notice(_("Starting %{role} upgrade on %{certname}. This may take a while.") % { role: role, certname: certname})
    status, result = PuppetX::Puppetlabs::Meep::Infra::PlanExecutor.run(
      'enterprise_tasks::upgrade_and_reprovision_replica',
      params: params,
      options: options,
      engine: 'orchestrator',
      log_context: log_context
    )
    if status.exitstatus != 0
      Puppet.err(result)
      exit status.exitstatus
    else
      Puppet.notice(_("%{role} upgrade complete.") % {role: role.capitalize})
    end
  end

  def upgrade_and_migrate_replica(certname, old_postgres_version, new_postgres_version, log_context, options)
    params = {
      'replica'              => certname,
      'primary'              => Puppet[:certname],
      'old_postgres_version' => old_postgres_version,
      'new_postgres_version' => new_postgres_version,
      'force'                => options[:force].to_s,
    }
    Puppet.notice(_("Starting replica upgrade on %{certname}. This involves a PE-PostgreSQL migration from version %{old_postgres_version} to %{new_postgres_version}. This may take a while.") % { certname: certname, old_postgres_version: old_postgres_version, new_postgres_version: new_postgres_version })
    status, result = PuppetX::Puppetlabs::Meep::Infra::PlanExecutor.run(
      'enterprise_tasks::upgrade_and_migrate_replica',
      params: params,
      options: options,
      engine: 'orchestrator',
      log_context: log_context
    )
    if status.exitstatus != 0
      Puppet.err(result)
      exit status.exitstatus
    else
      Puppet.notice(_("Replica upgrade complete."))
    end
  end

  def upgrade_secondary(role, certname, log_context, options)
    params = {
      'target' => certname,
      'force'  => options[:force].to_s,
    }

    display_certname = certname.kind_of?(Array) ? certname.join(',') : certname

    Puppet.notice(_("Starting %{role} upgrade on %{display_certname}. This may take a while.") % { role: role, display_certname: display_certname})
    status, result = PuppetX::Puppetlabs::Meep::Infra::PlanExecutor.run(
      'enterprise_tasks::upgrade_secondary',
      params: params,
      options: options,
      engine: 'orchestrator',
      log_context: log_context
    )
    if status.exitstatus != 0
      Puppet.err(result)
      exit status.exitstatus
    else
      Puppet.notice(_("%{role} upgrade complete.") % {role: role.capitalize})
    end
  end
end
