require 'puppet_x/puppetlabs/meep/caching'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/puppetlabs/meep/configure/psql'
require 'puppet_x/puppetlabs/meep/infra/defaults'
require 'puppet_x/puppetlabs/meep/infra/layout'
require 'puppet_x/puppetlabs/meep/infra/lookup'

module PuppetX
module Puppetlabs
module Meep
module Configure

  # Subroutines for assisting with migrations of pe-postgresql across major versions.
  module Postgres
    include PuppetX::Puppetlabs::Meep::Caching
    include PuppetX::Puppetlabs::Meep::Util
    include PuppetX::Puppetlabs::Meep::Infra::Defaults
    include PuppetX::Puppetlabs::Meep::Infra::Layout
    include PuppetX::Puppetlabs::Meep::Infra::Lookup

    # Notifies the user about some of the details of a successful migration.
    def inform_on_successful_postgres_migration(options)
      if options[:postgres_migration]
        current_postgres_version = requested_postgres_version
        old_datastore = pe_postgresql_info['versions'][previous_postgres_version] || {}
        old_data_dir = old_datastore['data_dir']
        old_tablespaces = old_datastore['tablespaces']

        directories_to_remove = [old_data_dir, old_tablespaces].flatten.compact

        message = [
          _("Your Postgresql datastore has been migrated from %{previous_postgres_version} to %{current_postgres_version}.") % { previous_postgres_version: previous_postgres_version, current_postgres_version: current_postgres_version },
          _("The %{previous_postgres_version} datastore has been left on disk as a backup.") % { previous_postgres_version: previous_postgres_version },
          _("When you have verified that the %{current_postgres_version} datastore is up to date, you may wish to remove the following %{previous_postgres_version} directories to free up space: %{directories_to_remove}") % { current_postgres_version: current_postgres_version, previous_postgres_version: previous_postgres_version, directories_to_remove: directories_to_remove.join(", ") }
        ]
        Puppet.notice(message.join("\n"))
      end
    end

    # Pushes postgresql locale and encoding information to pe.conf.
    # Used prior to upgrade to ensure that the new database ends up with the same
    # encoding and locale as the previous.
    def update_pe_conf_with_database_encoding_if_migrating(options)
      if options[:postgres_migration]
        psql_opts = {}
        psql_opts[:uid] = pe_postgres_uid

        psql = PuppetX::Puppetlabs::Meep::Configure::PSQL.new(psql_opts)
        encoding = nil
        begin
          encoding = psql.encoding_data
        rescue Puppet::ExecutionFailure => e
          message = [
            _("Failed to obtain current Postgresql server encoding, lc_type and lc_collate."),
            _("Will proceed with defaults (UTF8, en_US.UTF8, en_US.UTF8)."),
            _("Original error: %{error}") % { error: e.message },
          ]
          Puppet.warning(message.join("\n"))
          return false
        end

        enterprise_conf_dir = infra_default(:ENTERPRISE_CONF_DIR)
        pe_conf = PuppetX::Puppetlabs::Meep::Modify.new(enterprise_conf_dir)
        [
          ["server_encoding","encoding"],
          # The Postgresql initdb command takes a --locale flag which sets all of
          # the lc_* settings for Postgres which are in turn stored in the
          # database. We can only query out the lc_* values.  The only settings
          # we can pass to the postgresql module for initdb are locale, ctype and
          # collate, so we have to fudge and use ctype for locale.
          # https://github.com/puppetlabs/puppetlabs-pe_postgresql/blob/2016.5.0/manifests/server/initdb.pp#L45-L47
          ["lc_ctype", "locale"],
          ["lc_ctype", "ctype"],
          ["lc_collate", "collate"],
        ].each do |postgres_key,module_parameter|
          pe_conf.set_in_pe_conf("puppet_enterprise::profile::database::#{module_parameter}", encoding[postgres_key])
        end

        return true
      end
    end

    # @return [Hash] A set of postgres options that should allow a DB connection
    def get_psql_options()
      database_host = database_host_parameter
      database_port = puppet_lookup('puppet_enterprise::database_port')
      default_pe_user = 'pe-postgres'

      master_host = puppet_lookup('puppet_enterprise::puppet_master_host')
      orch_ssldir = '/etc/puppetlabs/orchestration-services/ssl'
      orch_creds = {
        cert: "#{orch_ssldir}/#{master_host}.cert.pem",
        key: "#{orch_ssldir}/#{master_host}.private_key.pem",
        user: puppet_lookup('puppet_enterprise::orchestrator_database_super_user') || puppet_lookup('puppet_enterprise::orchestrator_service_migration_db_user') || 'pe-orchestrator',
        database: puppet_lookup('puppet_enterprise::orchestrator_database_name') || 'pe-orchestrator',
        password: puppet_lookup('puppet_enterprise::orchestrator_database_password')
      }

      console_host = puppet_lookup('puppet_enterprise::console_host')
      console_ssldir = '/opt/puppetlabs/server/data/console-services/certs'
      console_creds = {
        cert: "#{console_ssldir}/#{console_host}.cert.pem",
        key: "#{console_ssldir}/#{console_host}.private_key.pem",
        user: puppet_lookup('puppet_enterprise::classifier_database_super_user') || puppet_lookup('puppet_enterprise::classifier_service_migration_db_user') || 'pe-classifier',
        database: puppet_lookup('puppet_enterprise::classifier_database_name') || 'pe-classifier',
        password: puppet_lookup('puppet_enterprise::classifier_database_password')
      }

      puppetdb_host = puppet_lookup('puppet_enterprise::puppetdb_host')
      puppetdb_ssldir = '/etc/puppetlabs/puppetdb/ssl'
      puppetdb_creds = {
        cert: "#{puppetdb_ssldir}/#{puppetdb_host}.cert.pem",
        key: "#{puppetdb_ssldir}/#{puppetdb_host}.private_key.pem",
        user: puppet_lookup('puppet_enterprise::puppetdb_database_user') || 'pe-puppetdb',
        database: puppet_lookup('puppet_enterprise::puppetdb_database_name') || 'pe-puppetdb',
        password: puppet_lookup('puppet_enterprise::puppetdb_database_password')
      }

      ca = puppet_lookup('puppet_enterprise::database_ca_cert') || '/etc/puppetlabs/puppet/ssl/certs/ca.pem'

      options = {
        :host => database_host,
      }
      options[:port] = database_port if !database_port.nil?

      available_ssl = [orch_creds, puppetdb_creds, console_creds].select { |creds| File.readable?(creds[:cert]) && File.readable?(creds[:key]) }
      found_ssl = available_ssl[0] if available_ssl.any?
      # assume cert auth unless overridden
      puppet_enterprise_database_ssl = puppet_lookup('puppet_enterprise::database_ssl')
      database_ssl = !puppet_enterprise_database_ssl.nil? ? puppet_enterprise_database_ssl : true
      puppet_enterprise_database_cert_auth = puppet_lookup('puppet_enterprise::database_cert_auth')
      database_cert_auth = !puppet_enterprise_database_cert_auth.nil? ? puppet_enterprise_database_cert_auth : true

      if found_ssl && options[:host]
        options[:user] = found_ssl[:user]
        options[:database] = found_ssl[:database]
        # User has cert auth disabled, but the SSL certs exist
        if !database_cert_auth
          options[:password] = found_ssl[:password]
        end
      # SSL certs aren't available and local postgres user exists
      elsif system("id -u #{default_pe_user}", :out => File::NULL, :err => File::NULL)
        options[:uid] = default_pe_user
        options[:user] = default_pe_user
        options[:database] = 'postgres'
        options[:host] = nil
      else
        options[:user] = puppetdb_creds[:user]
        options[:database] = puppetdb_creds[:database]
        # Not sure the condition is necessary, since no certs were found, we can assume the user has to use password auth?
        if !database_cert_auth
          options[:password] = puppetdb_creds[:password]
        end
      end

      if database_ssl && options[:host]
        options[:sslmode] = 'verify-full' if database_ssl
        if File.readable?(ca)
          options[:sslcacert] = ca
        else
          Puppet.warning(_("We found that database SSL is enabled, but no CA certificate could be found at %{ca}" % {ca: ca}))
        end
      end

      if database_cert_auth && options[:host]
        if found_ssl
          options[:sslcert] = found_ssl[:cert]
          options[:sslkey] = found_ssl[:key]
        else
          Puppet.warning(_("We found that database SSL certificate-based authentication is enabled, but no certificates could be found."))
        end
      end

      return options
    end

    # @return [String] the major.minor of the external postgresql database,
    #   or nil if we manage postgresql on primary or puppetdb node.
    def external_database_version
      return nil if !separate_database_node?

      cached(:external_database_version) do
        psql_opts = get_psql_options()
        psql = PuppetX::Puppetlabs::Meep::Configure::PSQL.new(psql_opts)
        version = nil
        begin
          version = psql.server_version
        rescue Puppet::ExecutionFailure => e
          messages = [
            _("It appears you have an externally managed Postgresql server '%{host}'.") % psql_opts,
            _("However, we were unable to look up the version of the Postgresql server on '%{host}'.") % psql_opts,
            _("Puppet::ExecutionFailure: %{error}") % { error: e.message },
          ]
          Puppet.warning(messages.join("\n"))
        end

        version
      end
    end

    def local_postgresql_client_is_behind?
      installed_packages = pe_postgresql_info['installed_packages'] || {}
      version_entry = installed_packages.find { |p| p[0] =~ /\Ape-postgresql\d*\Z/ }
      postgresql_version = version_entry.nil? ? nil : version_entry.last 
      postgresql_version.nil? ?
        false :
        version_cmp(postgresql_version, platform_current_postgres_version) == -1
    end
  end
end
end
end
end
