require 'puppet/agent'
require 'time'
module PuppetX
module Puppetlabs
module Meep
  # Utility methods.
  module Util
    include Puppet::Agent::Disabler
    include Puppet::Agent::Locker
    # Expects dot separated version strings such as "2017.3.0"
    #
    # Short circuits returning nil if either argument is nil or an empty string.
    #
    # @return -1, 0 or 1 based on <=> comparison of the Gem::Version
    #   where -1 indicates a < b
    #          0 indicates a = b
    #          1 indicates a > b
    def version_cmp(a, b)
      return nil if a.nil? || b.nil? || a.empty? || b.empty?
      v1 = Gem::Version.create(a)
      v2 = Gem::Version.create(b)
      v1 <=> v2
    end

    # This is similar to what we do in the enterprise_tasks with_agent_disabled function,
    # but only works on the local node this is running from.
    # Note: the enable method of Puppet::Agent::Disabler interferes with
    # puppet infra enable, so it is recreated in the ensure block.
    def with_agent_disabled(message, timeout=300, &block)
      already_disabled = disabled?
      disable(message) unless already_disabled
      start = Time.now
      if lockfile.locked?
        Puppet.debug(_("Puppet run in progress. Waiting up to #{timeout} seconds for lock to clear."))
        while lockfile.locked? && Time.now - start < timeout
          sleep(1)
        end
        raise Puppet::LockError.new(_("Agent lock did not clear in #{timeout} seconds.")) if lockfile.locked?
      end
      yield
    ensure
      unless already_disabled
        Puppet.notice _("Enabling Puppet.")
        disable_lockfile.unlock
      end
    end

    def self.time_stamp(fmt='%Y-%m-%dT%H.%M.%S%z')
      Time.now.strftime(fmt)
    end

    def self.default_logfile_path
      '/var/log/puppetlabs/installer'
    end

    # Keeps state about the overall log file context for a particular
    # puppet-infra action execution.
    class InfraLogContext
      attr_accessor :action, :timestamp, :description, :path, :ext

      def initialize(
        action:,
        description: nil,
        timestamp: PuppetX::Puppetlabs::Meep::Util.time_stamp,
        path: PuppetX::Puppetlabs::Meep::Util.default_logfile_path,
        ext: 'log'
      )
        self.action = action
        self.timestamp = timestamp
        self.description = description
        self.path = path
        self.ext = ext
      end

      # For each step in an action that needs a logfile, supply the description.
      #
      # @return [InfraLogContext] new instance with description set.
      def step(step_description)
        InfraLogContext.new(
          action: action,
          description: step_description,
          timestamp: timestamp,
          path: path,
          ext: ext
        )
      end

      # Construct log file path based on context state.
      #
      # @param certname [String] optional certname qualifier for filename.
      # @return [String] path to log file.
      def logfile(certname = nil)
        file = "#{path}/"
        filename_elements = [action, timestamp, description, certname].compact
        file += filename_elements.join('_')
        file += ".#{ext}" if ext
        file.tr(':', '_')
      end
    end

    def capture_cli_logs(action, log_level: 'info', capture_cli: true)
      log_context = PuppetX::Puppetlabs::Meep::Util::InfraLogContext.new(
        action: action,
      )
      Puppet[:log_level] = log_level if Puppet[:log_level] != 'debug'
      Puppet::Util::Log.newdestination(:infra_console)
      Puppet::Util::Log.close(:console)
      if capture_cli
        cli_log = log_context.step('cli-output').logfile
        Puppet::Util::Log.newdestination(cli_log)
        Puppet.notice(_('Logging cli output to: %{cli_log}') % { cli_log: cli_log })
      end
      log_context
    end
  end
end
end
end
