# Ruby Support Script Implementation


## Overview

The Ruby implementation of the support script can be found at:

  [../lib/puppet_x/puppetlabs/support_script/v3/puppet-enterprise-support.rb](../lib/puppet_x/puppetlabs/support_script/v3/puppet-enterprise-support.rb)

This script is the third major revision of the Support Script and is implemented
in Ruby. This document covers the layout of the Ruby Support Script along
with guidance for implementing and testing new diagnostics.


## Architecture

The Ruby Support Script is implemented as a single source file. This makes
the script easy to deploy to remote systems as there is no need to copy
and then unpack an archive. The script is organized as a set of Ruby
classes inside of a `PuppetX::Puppetlabs::SupportScript` module and
followed by a CLI entrypoint that is used if `puppet-enterprise-support.rb`
is invoked directly instead of being executed through the
`puppet enterprise support` command.

The following diagram shows how the major components of the support
script interact:

```
    ┌──────────────────────────────────────────────────┐
    │SupportScript::Settings                           │
    │                                                  │
 ┌ ─│Singleton instance containing setting values and  │
    │run-time state.                                   │
 │  └──────────────────────────────────────────────────┘
                              │
 │                            └─┐
                                ▼
 │      ┌──────────────────────────────────────────────┐
        │SupportScript::LogManager                     │
 │      │                                              │
        │Directs log output to stderr and files.       │
 │      └──────────────────────────────────────────────┘

 │  ┌──────────────────────────────────────────────────┐
    │SupportScript::Configable                         │
 │  │                                                  │
  ─▷│A mix-in that allows objects to access settings   │
    │and state, and send messages to the log.          │
    └──────────────────────────────────────────────────┘
                              │
 ┌ ─ ─ ─Included by─ ─ ─ ─ ─ ─

 │  ┌──────────────────────────────────────────────────┐
 │─▷│SupportScript::Runner                             │
    │                                                  │
 │  │Implements execution workflow for a support script│
    │invocation: valdation of settings, creation of    │
 │  │output directory, invocation of diagnostic scopes,│
    │cleanup, output.                                  │
 │  │                                                  │
    │                                                  │
 │  │   ┌──────────────────────────────────────────────┤
 │─ ┼ ─▷│SupportScript::Scope::Base                    │
    │   │                                              │
 │  │   │Top-level group containing other scopes and   │
    │   │checks.                                       │
 │  │   └──────────────────────────────────────────────┤
    │                           │                      │
 │  │    ┌──────────────────────┴─┐                    │
    │    │                        ▼                    │     ┌─────────────┐
 │  │    │  ┌──────────────────────────────────────────┤  ┌─▶│exec commands│
 │─ ┼ ─ ─│─▷│SupportScript::Confinable                 │  │  └─────────────┘
    │    │  │                                          │  │  ┌─────────────┐
 │  │    │  │Collection of constraints that determines │──┼─▶│inspect files│
    │    │  │if the associated Scope or Check will run.│  │  └─────────────┘
 │  │    │  └──────────────────────────────────────────┤  │  ┌─────────────┐
    │    │                                             │  └─▶│ check facts │
 │  │    │                                             │     └─────────────┘
    │    │                                             │
 │  │    │                                             │     ┌─────────────┐
    │    │  ┌──────────────────────────────────────────┤  ┌─▶│exec commands│
 ├ ─│─ ─ ┼ ▷│SupportScript::Check                      │  │  └─────────────┘
    │    │  │                                          │  │  ┌─────────────┐
 │  │    ├─▶│Diagnostic that collects and adds data to │──┼─▶│ copy files  │
    │    │  │the runner's output directory.            │  │  └─────────────┘
 │  │    │  └──────────────────────────────────────────┤  │  ┌─────────────┐
    │    │                        │                    │  └─▶│  call APIs  │
 │  │    │                        └─┐                  │     └─────────────┘
    │    │                          ▼                  │
 │  │    │      ┌──────────────────────────────────────┤
 │─ ┼ ─ ─│─ ─ ─▷│SupportScript::Confinable             │
    │    │      └──────────────────────────────────────┤
 │  │    │                                             │
    │    │  ┌──────────────────────────────────────────┤
 └ ─│─ ─ ┼ ▷│Further SupportScript::Check and          │
    │    └─▶│SupportScript::Scope objects.             │
    │       └──────────────────────────────────────────┤
    └────┬─────────────────────────────────────────────┘
         │
         │  ┌──────────────────────────────┐
         ├─▶│Create output archive with tar│
         │  └──────────────────────────────┘
         │  ┌──────────────────────────────────────────┐
         ├─▶│(optional) Encrypt output archive with gpg│
         │  └──────────────────────────────────────────┘
         │  ┌──────────────────────────────────────────┐
         └─▶│(optional) Upload output archive with sftp│
            └──────────────────────────────────────────┘
```


## Adding new diagnostics

The task of collecting and saving data is carried out by
descendants of the `PuppetX::Puppetlabs::SupportScript::Check`
class. Each `Check` class must implement one method, `run`,
that contains the logic for gathering diagnostics.

The most basic form a `Check` can take is:


```ruby
class Check::MyAwesomeCheck < Check
  def run
    # Runtime logic goes here.

    # Functions from the Configable module provide access to
    # logging, configuration, and cached state data.
    log.info('hello')

    # Functions from the DiagnosticHelpers module are available
    # and can be used accomplish common tasks.
    exec_drop('echo "hello world!"', state[:drop_directory], 'hello.txt')
  end
end
```

In order to be executed, the new `Check` class must be added
to a `PuppetX::Puppetlabs::SupportScript::Scope` class
and given a `name`. The `name` is used by the `--enable`,
`--disable` and `--only` flags to toggle checks on and off.
Scopes group checks together and can also contain other scopes.
This grouping organizes the support script into a tree structure
with the scopes forming branches and the checks forming leaves.

The root of the tree is `Scope::Base`:

```ruby
class Scope::Base < Scope
  # ...

  self.add_child(Check::BaseStatus, name: 'base-status')
  self.add_child(Check::MyAwesomeCheck, name: 'my-awesome-check')
end
```

However, there are other scopes under `Scope::Base` that
may be more suitable for adding a particular check:

  - `Scope::System`: Diagnostics for the operating system.
  - `Scope::Puppet`: Diagnostics for the Puppet agent components.
  - `Scope::PuppetServer`: Diagnostics for the Puppet Server service.
  - `Scope::PuppetDB`: Diagnostics for the PuppetDB service.
  - `Scope::Pe`: Diagnostics for PE installation and upgrades.
  - `Scope::Pe::Console`: Diagnostics for PE Console service.
  - `Scope::Pe::Orchestrator`: Diagnostics for the PE Orchestrator service.
  - `Scope::Pe::Postgres`: Diagnostics for the PE PostgreSQL service.

Each scope listed above has up to four "standard" check classes
with the following `name` values:

  - `config`: Collects configuration files and other static settings.
  - `logs`: Collects log files.
  - `metrics`: Collects metrics snapshots.
  - `status`: Collects diagnostics from running services.

Adding diagnostics to the `run` method of one of the existing
`config`, `logs`, `metrics`, or `status` checks is preferable
to creating a new `Check` class.

A set of diagnostics that the user must opt-in to or that should
be confined by specific logic will require the creation of a
new `Check`. These checks make use of the `setup` method and are
described in the following sections.

The `setup` method is used to initialize the check and has the
following signature:

```
def setup(**options)

end
```

The `options` is a hash of all arguments passed to the
`add_child` method call and can be used to initialize
the same check class with different parameters.
The `setup` method therefore serves a role similar to
`initialize`. Take care to keep expensive computations
or diagnostics out of the `setup` method. These should
be located in `run` so that they may be skipped if
the check is disabled or does not satisfy its confines.


### Opt-in diagnostics

An opt-in diagnostic uses the `setup` method to set its enabled
flag to `false`:

```ruby
class Check::MyAwesomeOptInCheck < Check
  def setup(**options)
    self.enabled = false
  end

  def run
    # Runtime logic goes here.
  end
end
```


### Confined diagnostics

The `setup` method may also be used to confine the execution
of the logic in `run` if specific conditions are met.
This is done by adding one or more `confine` statements to
`setup`. These statements have the same behavior as the
`confine` statement in Facter facts (the implementation
was borrowed from Facter 2):

```ruby
class Check::MyAwesomeConfinedCheck < Check
  def setup(**options)
    # Only execute run if the kernel fact resolves to "linux"
    confine kernel: 'linux'

    # Only execute run if the block of logic evaluates to a
    # truthy value.
    confine do
      # Arbitrary logic resulting in a true or false value.
    end
  end

  def run
    # Runtime logic goes here.
  end
end
```

All `confine` statements must be satisfied in order for the
`run` method to be executed.
