#!/opt/puppetlabs/puppet/bin/ruby

require 'date'
require 'fileutils'
require 'json'
require 'pathname'
require 'tempfile'
require 'timeout'
require 'forwardable'
require 'logger'
require 'rubygems'

# Test nodes do not meet the minimum system requirements for tune to optimize.
if ENV['BEAKER_TESTING']
  ENV['TEST_CPU'] = '8'
  ENV['TEST_RAM'] = '16384'
end


module PuppetX
module Puppetlabs
# Diagnostic tools for Puppet Enterprise
#
# This module contains components for running diagnostic checks on Puppet
# Enterprise installations.
module SupportScript
  VERSION = '3.8.0'.freeze
  DOC_URL = 'https://www.puppet.com/docs/pe/2021.7/getting_support_for_pe.html#info-collected-support-scripts'.freeze

  PGP_RECIPIENT = '49E44EC4025DBB4BE3B67B1BB0AE480298FD85B0'.freeze
  PGP_KEY = (<<-'EOS').freeze
-----BEGIN PGP PUBLIC KEY BLOCK-----
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=
=q5SY
-----END PGP PUBLIC KEY BLOCK-----
EOS

  SFTP_HOST = 'customer-support.puppetlabs.net'.freeze
  SFTP_USER = 'puppet.enterprise.support'.freeze
  SFTP_KEY = (<<-'EOS').freeze
-----BEGIN RSA PRIVATE KEY-----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-----END RSA PRIVATE KEY-----
EOS
  SFTP_KNOWN_HOSTS = (<<-'EOS').freeze
# Primary
customer-support.puppetlabs.net ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQCrLB9mWc9pxVjUin3LtIRj3vMmqgv8oUKa/JAfXkRVoKgF7EYmmsjCU55pg+ZFBUD87hJ9JNKVM8TGEQ89sjnPBN6lCdKn0sc4wfVHqbh70VvX7LhQPM79eUUkvdfHcRep1VsgWrxJlKZH42X+ermWrnzE+1vz2OB/edDOjG4Ku/gh7YHFTS1VyPzf+R0q5Nl0VQvo0RHXaeVVNMLlMy5BuRQCU1+WPKKHtH+ZvzfE6/rc/CR8L4PKzcHuQN5n1bcl13hlsYr+IHMkESJyZWIHeZiKUSa7hu464Nl0LNGhDLN25bAZrqiFwiyNEhz1+v1BOhhgkFJ0vWSoKPlsqS55
customer-support.puppetlabs.net ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBOwBtY6ojejMa6tl9QSAWDi2pSpTYBKldD3r6kIOJDTd2b7x99WQPFhJgWdJ76ANIolvEWI5lAkvFwMJ5SMG5Ak=
customer-support.puppetlabs.net ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIO8GiFNutya82Ya+xeI8LWEbA2EmwVQF5gtvjsJ6s+W0
# Asia-Pacific
customer-support-syd.puppetlabs.net ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQCw9P+9D/QFSveyYQUEIkB0Ii9OPZpna32x05RKgMslFZWXyctXfhoFQvtE/df9TfcYA8dFZuibJZamQKwQ6VPjkbk7YdMpWbho5X9j78B7Dr74iQQKzZzLUYf4Nqrjpo+S6lHGLTA2Oxt8Hi6a7FqYqzVDR8umuetncLsPMSpjlU+veAcMIhPa5Lvw7m8dOoeiBfLs3TL+HgLMr/IUJ31QLUDIRDnB6nVBwoUU3OW+an9JksIeGyoB0kqT86nW22jFaZpzJ5YeRWvtmrlZkPjpjayPb91rKLd8ZLQGTR3Y55yArok9Q55+C74LsouNyFMKKdoa4dOh7ikhJ5wE1dU1
customer-support-syd.puppetlabs.net ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBFUTiJ8+OWy3QIF2ajlOaiE7k10Ae1TP9eh4ClgNMKvrGXojaJ/qztQHGQbhsDLQT0BduJ24ow58bXebziz5JCs=
customer-support-syd.puppetlabs.net ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIJ+v91GesXhPY9+hpOTqPIFlyFkMT8CrKDVNL7vFycP2
EOS

  # Manages one or more Logger instances
  #
  # Instances of this class wrap one or more instances of Logger and direct
  # logged messages to each as appropriate. Instances of this class provide
  # a method for each level in the stdlib `Logger::Severity` module, usually:
  #
  #   - `debug`
  #   - `info`
  #   - `warn`
  #   - `error`
  #   - `fatal`
  #   - `unknown`
  class LogManager
    # Create a Logger instance sending messages to stderr
    #
    # This method creates an instance of Ruby's standard Logger class that
    # sends messages to stderr. Formatting is kept simple and is intended
    # for consumption by humans.
    #
    # @return [Logger]
    def self.console_logger
      logger = ::Logger.new($stderr)
      # TODO: Should be configurable.
      logger.level = ::Logger::WARN
      logger.formatter = proc do |severity, datetime, progname, msg|
                           "%s: %s\n" % [severity, msg]
                         end

      logger
    end

    # Create a Logger instance sending messages to a file
    #
    # This method creates an instance of Ruby's standard Logger class that
    # sends messages to a file. DEBUG level is used to capture maximum
    # detail. Messages are formatted as JSON objects, one per line.
    #
    # @return [Logger]
    def self.file_logger(path)
      logger = ::Logger.new(path)
      # TODO: Level should be configurable.
      logger.level = ::Logger::DEBUG

      logger.formatter = proc do |severity, datetime, progname, msg|
                           {time: datetime.iso8601(3),
                            level: severity,
                            msg: msg}.to_json + "\n".freeze
                         end

      logger
    end

    def initialize
      @loggers = []
    end

    # Add a Logger instance to this manager
    #
    # @param logger [Logger] The logger to add.
    # @return [void]
    def add_logger(logger)
      unless logger.is_a?(::Logger)
        raise ArgumentError, 'An instance of Logger must be passed. Got a value of type %{class}.' %
          {class: logger.class}
      end

      @loggers.push(logger)
    end

    # Remove a Logger instance from this manager
    #
    # @param logger [Logger] The logger to add.
    # @return [void]
    def remove_logger(logger)
      @loggers.delete(logger)
    end

    ::Logger::Severity.constants.each do |name|
      method_name = name.to_s.downcase.to_sym
      level = ::Logger::Severity.const_get(name)

      define_method(method_name) do |message = nil, &block|
        # If a block was passed, ignore the message.
        message = nil unless block.nil?

        @loggers.each do |logger|
          next unless logger.level <= level
          message ||= block.call unless block.nil?

          logger.send(method_name, message)
        end
      end
    end
  end

  # Holds configuration and state shared by other objects
  #
  # Classes that need access to state managed by the Settings class should
  # include the {Configable} module, which provides access to a singleton
  # instance shared by all objects.
  class Settings
    # Access the configured {LogManager}.
    #
    # This object can be used to log messages at `debug`, `info`, `warn`, and
    # `error` levels.
    #
    # @return [LogManager]
    attr_reader :log
    # Access the configured settings.
    #
    # Values in this hash should only be read and never written.
    #
    # @return[Hash]
    attr_accessor :settings
    # Access runtime state.
    #
    # Values can be stored or updated in this hash. Use carefully.
    #
    # @return[Hash]
    attr_accessor :state

    def self.instance
      @instance ||= new
    end

    def initialize
      @log = LogManager.new
      @settings = {dir: File.directory?('/var/tmp') ? '/var/tmp' : '/tmp',
                   log_age: 7,
                   noop: false,
                   encrypt: false,
                   ticket: '',
                   upload: false,
                   upload_disable_host_key_check: false,
                   z_do_not_delete_drop_directory: false,

                   list: false,
                   enable: [],
                   disable: [],
                   only: [],

                   # TODO: Take this out of settings.
                   version: VERSION,

                   # TODO: Deprecate and replace these with Scope classes.
                   scope: %w[enterprise etc log networking resources system].product([true]).to_h,
                   # TODO: Deprecate and replace these with Check classes
                   #       that default to disabled.
                   classifier: false,
                   filesync: false}
      @state = {exit_code: 0}
    end

    # Update configuration of the settings object
    #
    # @param options [Hash] a hash of options to merge into the existing
    #   configuration.
    #
    # @raise [ArgumentError] if a configuration option is invalid.
    #
    # @return [void]
    def configure(**options)
      options.each do |key, value|
        v = case key
            when :enable, :disable, :only
              unless value.is_a?(Array)
                raise ArgumentError, 'The %{key} option must be set to an Array value. Got a value of type %{class}.' %
                  {key: key,
                   class: value.class}
              end
            when :noop, :encrypt, :upload, :upload_disable_host_key_check, :list, :z_do_not_delete_drop_directory
              unless [true, false].include?(value)
                raise ArgumentError, 'The %{key} option must be set to true or false. Got a value of type %{class}.' %
                  {key: key,
                   class: value.class}
              end
            when :log_age
              unless value.to_s.match(%r{\A\d+|all\Z})
                raise ArgumentError, 'The log_age option must be a number, or the string "all". Got %{value}' %
                  {value: value}
              end

              (value.to_s == 'all') ? 999 : value.to_i
            when :scope
              (value.to_s == '') ? {}  : Hash[value.split(',').product([true])]
            when :ticket
              unless value.match(%r{\A[\d\w\-]+\Z})
                raise ArgumentError, 'The ticket option may contain only numbers, letters, underscores, and dashes. Got %{value}' %
                  {value: value}
              end
            end

        @settings[key] = v || value
      end
    end

    # Validate runtime configuration
    #
    # The validate method performs runtime verification of settings. This
    # method is used to ensure that file-based settings point to accssable
    # locations and that appropriate combinations of values have been provided
    # for settings that depend on each other.
    #
    # @raise [RuntimeError] raised if an invalid setting is found.
    #
    # @return [void]
    def validate
      if File.symlink?(@settings[:dir])
        raise 'The dir option cannot be a symlink: %{dir}' %
              {dir: @settings[:dir]}
      elsif (! (File.directory?(@settings[:dir]) && File.writable?(@settings[:dir])))
        raise 'The dir option is not set to a writable directory: %{dir}' %
              {dir: @settings[:dir]}
      end

      if ( @settings[:upload] && (@settings[:ticket].nil? || settings[:ticket].empty?) )
        raise 'The upload option requires a value to be specified for the ticket setting.'
      end

      if ( @settings[:upload] &&
           @settings.key?(:upload_key) && (! File.readable?(@settings[:upload_key])) )
        raise 'The upload_key option is not readable or does not exist: %{key}' %
              {key: @settings[:upload_key]}
      end
    end
  end

  # Execute external commands
  module Exec
    # Command Result
    #
    # @param stdout [String] A string containing the standard output
    #   written by the command.
    # @param stderr [String] A string containing the standard error
    #   written by the command.
    # @param status [Integer] An integer representing the exit code of the
    #   command.
    # @param error [String, nil] A string holding an error message if
    #   command execution was not successful.
    Result = Struct.new(:stdout, :stderr, :status, :error)

    # Exception class for failed commands
    class ExecError < StandardError; end

    # Execute a command and return a Result
    #
    # This is basically `Open3.popen3`, but with added logic to time the
    # executed command out if it runs for too long.
    #
    # @param cmd [Array<String>] Command and arguments to execute. Commands
    #  consisting of a single String will cause `Process.spawn` to wrap the
    #  execution in a system shell such as `/bin/sh -c`.
    # @param env [Hash] A hash of environment variables to set
    #   when the command is executed.
    # @param stdin_data [String] A string of standard input to pass
    #   to the executed command. Sould be smaller than 4096 characters.
    # @param timeout [Integer] Number of seconds to allow for command
    #   execution to complete. A value of 0 will disable the timeout.
    #
    # @return [Result] A `Result` object containing the output and status
    #   of the command.
    def self.exec_cmd(*cmd, env: {}, stdin_data: nil, timeout: 300)
      out_r, out_w = IO.pipe
      err_r, err_w = IO.pipe
      _env = {'LC_ALL' => 'C', 'LANG' => 'C'}.merge(env)

      input = if stdin_data.nil?
                Gem.win_platform? ? 'NUL' : '/dev/null'
              else
                # NOTE: Pipe capacity is limited. Probably at least 4096 bytes.
                #       65536 bytes at most.
                in_r, in_w = IO.pipe
                in_w.binmode
                in_w.sync = true

                in_w.write(stdin_data)
                in_w.close

                in_r
              end

      opts = {in: input,
              out: out_w,
              err: err_w}

      pid = Process.spawn(_env, *cmd, opts)

      [out_w, err_w].each(&:close)
      stdout_reader = Thread.new do
        stdout = out_r.read
        out_r.close
        stdout
      end
      stderr_reader = Thread.new do
        stderr = err_r.read
        err_r.close
        stderr
      end

      deadline = (Process.clock_gettime(Process::CLOCK_MONOTONIC, :float_second) + timeout)
      status = nil

      loop do
        _, status = Process.waitpid2(pid, Process::WNOHANG)
        break if status
        unless timeout.zero?
          raise Timeout::Error if (deadline < Process.clock_gettime(Process::CLOCK_MONOTONIC, :float_second))
        end
        # Sleep for a bit so that we don't spin in a tight loop burning
        # CPU on waitpid() syscalls.
        sleep(0.01)
      end

      Result.new(stdout_reader.value, stderr_reader.value, status.to_i, nil)
    rescue Timeout::Error
      Process.kill(:TERM, pid)
      Process.detach(pid)

      Result.new('', '', -1, 'command failed to complete after %{timeout} seconds' %
                  {timeout: timeout})
    rescue StandardError => e
      # File not found. Permission denied. Etc.
      Result.new('', '', -1, '%{class}: %{message}' %
                 {class: e.class,
                  message: e.message})
    end
  end

  # Mix-in module for accessing shared settings and state
  #
  # Including the `Configable` module in a class provides access to a shared
  # state object returned by {Settings.instance}. Including the `Configable`
  # module also defines helper methods to access various components of the
  # `Settings` instance as if they were local instance variables.
  module Configable
    extend Forwardable

    # @!method log
    #   @see Settings#log
    def_delegators :@config, :log
    # @!method settings
    #   @see Settings#settings
    def_delegators :@config, :settings
    # @!method state
    #   @see Settings#state
    def_delegators :@config, :state

    def initialize_configable
      @config = Settings.instance
    end

    def noop?
      @config.settings[:noop]
    end
  end

  # A restricting tag for diagnostics
  #
  # A confine instance  may be initialized with with a logical check and
  # resolves the check on demand to a `true` or `false` value.
  class Confine
    include Configable

    attr_accessor :fact, :values

    # Create a new confine instance
    #
    # @param fact [Symbol] Name of the fact
    # @param values [Array] One or more values to match against. They can be
    #   any type that provides a `===` method.
    # @param block [Proc] Alternatively a block can be supplied as a check.
    #   The fact value will be passed as the argument to the block. If the
    #   block returns true then the fact will be enabled, otherwise it will
    #   be disabled.
    def initialize(fact = nil, *values, &block)
      initialize_configable

      raise ArgumentError, "The fact name must be provided" unless fact or block_given?
      if values.empty? and not block_given?
        raise ArgumentError, "One or more values or a block must be provided"
      end

      @fact = fact
      @values = values
      @block = block
    end

    def to_s
      return @block.to_s if @block
      return "'%s' '%s'" % [@fact, @values.join(",")]
    end

    # Convert a value to a canonical form
    #
    # This method is used by {true?} to normalize strings and symbol values
    # prior to comparing them via `===`.
    def normalize(value)
      value = value.to_s if value.is_a?(Symbol)
      value = value.downcase if value.is_a?(String)
      value
    end

    # Evaluate the fact, returning true or false.
    def true?
      if @block and not @fact then
        begin
          return !! @block.call
        rescue StandardError => e
          log.error("%{exception_class} raised during Confine: %{message}\n\t%{backtrace}" %
                    {exception_class: e.class,
                     message: e.message,
                     backtrace: e.backtrace.join("\n\t")})
          return false
        end
      end

      unless fact = Facter[@fact]
        log.warn('Confine requested undefined fact named: %{fact}' %
                 {fact: @fact})
        return false
      end
      value = normalize(fact.value)

      return false if value.nil?

      if @block then
        begin
          return !! @block.call(value)
        rescue StandardError => e
          log.error("%{exception_class} raised during Confine: %{message}\n\t%{backtrace}" %
                    {exception_class: e.class,
                     message: e.message,
                     backtrace: e.backtrace.join("\n\t")})
          return false
        end
      end

      @values.any? { |v| normalize(v) === value }
    end
  end

  # Mix-in module for declaring and evaluating confines
  #
  # Including this module in another class allows instances of that class
  # to declare {Confine} instances which are then evaluated with a call to
  # the {suitable?} method. The module also provides a simple enable/disable
  # switch that can be set by calling {#enabled=} and checked by calling
  # {#enabled?}
  module Confinable
    # Initalizes object state used by the Confinable module
    #
    # This method should be called from the `initialize` method of any class
    # that includes the `Confinable` module.
    #
    # @return [void]
    def initialize_confinable
      @confines = []
      @enabled = true
      @sensitive = false
    end

    # Sets the conditions for this instance to be used.
    #
    # This method accepts multiple forms of arguments. Each call to this method
    # adds a new {Confine} instance that must pass in order for {suitable?} to
    # return `true`.
    #
    # @return [void]
    #
    # @overload confine(confines)
    #   Confine a fact to a specific fact value or values. This form takes a
    #   hash of fact names and values. Every fact must match the values given
    #   for that fact, otherwise this resolution will not be considered
    #   suitable. The values given for a fact can be an array, in which case
    #   the value of the fact must be in the array for it to match.
    #
    #   @param [Hash{String,Symbol=>String,Array<String>}] confines set of facts
    #     identified by the hash keys whose fact value must match the
    #     argument value.
    #
    #   @example Confining to a single value
    #       confine :kernel => 'Linux'
    #
    #   @example Confining to multiple values
    #       confine :osfamily => ['RedHat', 'SuSE']
    #
    # @overload confine(confines, &block)
    #   Confine to logic in a block with the value of a specified fact yielded
    #   to the block.
    #
    #   @param [String,Symbol] confines the fact name whose value should be
    #     yielded to the block
    #   @param [Proc] block determines suitability. If the block evaluates to
    #     `false` or `nil` then the confined object will not be evaluated.
    #
    #   @yield [value] the value of the fact identified by `confines`
    #
    #   @example Confine to a host with an ipaddress in a specific subnet
    #       confine :ipaddress do |addr|
    #         require 'ipaddr'
    #         IPAddr.new('192.168.0.0/16').include? addr
    #       end
    #
    # @overload confine(&block)
    #   Confine to a block. The object will be evaluated only if the block
    #   evaluates to something other than `false` or `nil`.
    #
    #   @param [Proc] block determines suitability. If the block
    #     evaluates to `false` or `nil` then the confined object will not be
    #     evaluated.
    #
    #   @example Confine to systems with a specific file
    #       confine { File.exist? '/bin/foo' }
    def confine(confines = nil, &block)
      case confines
      when Hash
        confines.each do |fact, values|
          @confines.push Confine.new(fact, *values)
        end
      else
        if block
          if confines
            @confines.push Confine.new(confines, &block)
          else
            @confines.push Confine.new(&block)
          end
        else
        end
      end
    end

    # Check all conditions defined for the confined object
    #
    # Checks each condition defined through a call to {#confine}.
    #
    # @return [false] if any condition evaluates to `false`.
    # @return [true] all conditions evalute to `true` or no conditions
    #   were defined.
    def suitable?
      @confines.all? { |confine| confine.true? }
    end

    # Toggle the enabled status of the confined object
    #
    # @param value [true, false]
    # @return [void]
    def enabled=(value)
      unless value.is_a?(TrueClass) || value.is_a?(FalseClass)
        raise ArgumentError, 'The value of enabled must be set to true or false. Got a value of type %{class}.' %
          {class: value.class}
      end

      @enabled=value
    end

    # Check the enabled status of the confined object
    #
    # @return [true, false]
    def enabled?
      @enabled
    end

    # Toggle the sensitive status of the confined object
    #
    # @param value [true, false]
    # @return [void]
    def sensitive=(value)
      unless value.is_a?(TrueClass) || value.is_a?(FalseClass)
        raise ArgumentError, 'The value of enabled must be set to true or false. Got a value of type %{class}.' %
          {class: value.class}
      end

      @sensitive=value
    end
    #
    # Check the enabled status of the sensitive object
    #
    # @return [true, false]
    def sensitive?
      @sensitive
    end

  end

  # Helper functions for gnerating diagnostic output
  #
  # The methods in this module provide an API for executing commands, returing
  # results, copying files, and otherwise generating diagnostic output. This
  # module should be included along with the {Configable} module and depends
  # on state initialized by {Runner#setup}
  module DiagnosticHelpers
    PUP_PATHS = {puppetlabs_bin: '/opt/puppetlabs/bin',
                 puppet_bin:     '/opt/puppetlabs/puppet/bin',
                 server_bin:     '/opt/puppetlabs/server/bin',
                 server_data:    '/opt/puppetlabs/server/data'}.freeze

    #===========================================================================
    # Utilities
    #===========================================================================

    # Display a message.
    def display(info = '')
      $stdout.puts(info)
    end

    # Display an error message.
    def display_warning(info = '')
      log.warn(info)
    end

    # Display an error message, and exit.
    def fail_and_exit(datum)
      log.error(datum)
      exit 1
    end

    # Execute a command line and return the result
    #
    # @param command_line [String] The command line to execute.
    # @param timeout [Integer] Amount of time, in sections, allowed for
    #   the command line to complete. A value of 0 will disable the timeout.
    #
    # @return [String] STDOUT from the command.
    # @return [String] An empty string, if an exception is raised while
    #   executing the command.
    def exec_return_result(command_line, timeout = 300)
      options = { timeout: timeout }
      result = Exec.exec_cmd(command_line, **options)

      if !result.error.nil?
        log.error('exec_return_result: command failed: %{command_line} with error: %{error}' %
                  {command_line: command_line,
                   error: result.error})
        ''
      elsif !result.status.zero?
        # Log at DEBUG level due to some diagnostics that are expected
        # to return non-zero codes.
        #
        # TODO: Add support for specifying expected non-zero codes and
        #       promote this log message to ERROR level.
        log.debug('exec_return_result: command executed: %{command_line} with non-zero exit code: %{status}%{stderr}' %
                  {command_line: command_line,
                   status: result.status,
                   stderr: result.stderr.empty? ? '' : "\n\t" + result.stderr.lines.join("\n\t")})
        result.stdout
      else
        result.stdout
      end
    end

    # Execute a command line and return true or false
    #
    # @param (see #exec_return_result)
    #
    # @return [true] If the command completes with an exit code of zero.
    # @return [false] If the command completes win a non-zero exit code or
    #   an exception is raised.
    def exec_return_status(command_line, timeout = 300)
      options = { timeout: timeout }
      result = Exec.exec_cmd(command_line, **options)

      if result.error.nil?
        result.status.zero?
      else
        log.error('exec_return_status: command failed: %{command_line} with error: %{error}' %
                  {command_line: command_line,
                   error: result.error})
        false
      end
    end

    # Execute a command line or raise an error
    #
    # @param (see #exec_return_result)
    #
    # @return [void]
    # @raise [Exec::ExecError] If the command exits with a non-zero code or
    #   an exception is raised during execution.
    def exec_or_fail(command_line, timeout = 300)
      options = { timeout: timeout }
      result = Exec.exec_cmd(command_line, **options)

      if !result.error.nil?
        raise Exec::ExecError,
              'exec_or_fail: command failed: %{command_line} with error: %{error}' %
              {command_line: command_line,
               error: result.error}
      elsif !result.status.zero?
        raise Exec::ExecError,
              'exec_or_fail: command failed: %{command_line} with status: %{status}' %
              {command_line: command_line,
               status: result.status}
      end
    end

    # Test for command existance
    #
    # @param command [String] The name of an executable
    #
    # @return [String] Expanded path to the executable if it exists and is
    #   executable.
    # @return [nil] If no executable matching `command` can be found in the
    #   `PATH`.
    def executable?(command)
      Facter::Core::Execution.which(command)
    end

    # Test whether command responds to --help
    #
    # @param command [String] The name of a command to test
    #
    # @return [true] If the command exits successfully when invoked with `--help`.
    # @return [false] If the command exits with a non-zero code when invoked
    #   with `--help`, or an exception is raised.
    def help_option?(command)
      command_line = "#{command} --help > /dev/null 2>&1"

      exec_return_status(command_line)
    end

    # Test for command option existence
    #
    # @param command [String] The name of a command to test
    # @param option [String] the name of an option to test
    #
    # @return [true] If the `option` can be found in the `--help` output
    #   or manpage for the command.
    # @return [false] If the `option` is not found, or an exception
    #   is raised.
    def documented_option?(command, option)
      if help_option?(command)
        command_line = "#{command} --help | grep -q -- '#{option}' > /dev/null 2>&1"
      else
        command_line = "man #{command}    | grep -q -- '#{option}' > /dev/null 2>&1"
      end

      exec_return_status(command_line)
    end

    # Pretty Format JSON
    #
    # Parses a string of JSON, optionally removes blacklisted top-level keys,
    # and returns the output as a pretty-printed string.
    #
    # @param text [String] A string of JSON data.
    # @param blacklist [Array<String>] A list of keys to remove from the
    #   output.
    #
    # @return [String] Pretty printed JSON.
    # @return [String] An empty string, if parsing or generation fails.
    def pretty_json(text, blacklist = [])
      return text if text == ''
      begin
        json = JSON.parse(text)
      rescue JSON::ParserError
        log.error('pretty_json: unable to parse json')
        return ''
      end
      blacklist.each do |blacklist_key|
        if json.is_a?(Array)
          json.each do |item|
            if item.is_a?(Hash)
              item.delete(blacklist_key) if item.key?(blacklist_key)
            end
          end
        end
        if json.is_a?(Hash)
          json.delete(blacklist_key) if json.key?(blacklist_key)
        end
      end
      begin
        JSON.pretty_generate(json)
      rescue JSON::GeneratorError
        log.error('pretty_json: unable to generate json')
        return ''
      end
    end

    # Return the value of a Puppet setting
    #
    # Values are stored in a cache after being read. Subsequent calls
    # will return the cached value.
    #
    # @param setting [String] The setting to retrieve.
    def puppet_conf(setting, section = 'main')
      state['puppet_conf'] ||= {}
      state['puppet_conf'][section] ||= {}

      cached_value = state['puppet_conf'][section].fetch(setting, nil)

      if cached_value.nil?
        value = exec_return_result("#{PUP_PATHS[:puppet_bin]}/puppet config print --section '#{section}' '#{setting}'").strip
        state['puppet_conf'][section][setting] = value

        value
      else
        cached_value
      end
    end

    # Return the version of PE installed
    #
    # Value is cached after being read. Subsequent calls will
    # return the cached value.
    #
    # @return [Gem::Version] The version of PE as a Gem::Version object.
    # @return [nil] If there is an error retrieving the PE version, or
    #   the support script is run on an agent node.
    def pe_version
      unless state.key?(:pe_version)
        begin
          if File.readable?('/opt/puppetlabs/server/pe_version')
            state[:pe_version] = Gem::Version.new(File.read('/opt/puppetlabs/server/pe_version'))
          else
            state[:pe_version] = nil
          end
        rescue => e
          log.error('unable to read PE version: %{error}' %
                    {error: e.message})

          state[:pe_version] = nil
        end
      end

      state[:pe_version]
    end

    # Request a URL using curl
    #
    # @param url [String] The URL to request.
    # @param headers [Hash{String => String}] A hash of headers to add to
    #   the request where the key is the header name and the value is the
    #   header value.
    # @param options [Hash] A hash of options where the key is the name
    #   of a `curl` long flag and the value is the value to pass with the
    #   flag.
    #
    # @return [String] The body returned by the request.
    def curl_url(url, headers: {}, **options)
      networking_info = Facter.value('networking')
      local_hostnames = if networking_info.nil?
                          ['localhost', '127.0.0.1']
                        else
                          ['localhost', '127.0.0.1',
                           networking_info['fqdn'], networking_info['hostname'], networking_info['ip']].compact
                        end

      headers = headers.reduce('') {|m, (k,v)| m += " -H '#{k}: #{v}'"}
      opts = options.reduce("--insecure --silent --show-error --connect-timeout 5 --max-time 60 --noproxy #{local_hostnames.join(',')}") {|m, (k,v)| m += " --#{k} '#{v}'"}

      exec_return_result("#{PUP_PATHS[:puppet_bin]}/curl #{headers} #{opts} '#{url}'")
    end

    # Request a URL using the agent's certificate
    #
    # @see curl_url
    def curl_cert_auth(url, **options)
      cert = puppet_conf('hostcert')
      key = puppet_conf('hostprivkey')

      unless File.readable?(cert)
        log.error('unable to read agent certificate for curl: %{cert}' %
                  {cert: cert})
        return ''
      end

      unless File.readable?(key)
        log.error('unable to read agent private key for curl: %{key}' %
                  {key: key})
        return ''
      end

      options[:key] = key
      options[:cert] = cert

      curl_url(url, **options)
    end

    # Return package manager used by the OS executing the script
    #
    # @return [String] a string giving the name of the executable used
    #   to manage packages
    # @return [nil] for unknown operating systems
    def pkg_manager
      return state[:platform_packaging] if state.key?(:platform_packaging)

      os = Facter.value('os')
      pkg_manager = case os['family'].downcase
                    when 'redhat', 'suse'
                      'rpm'
                    when 'debian'
                      'dpkg'
                    else
                      log.error('Unknown packaging system for operating system "%{os_name}" and famliy "%{os_family}"' %
                                {os_name: os['name'],
                                 os_family: os['family']})
                      # Mark run as failed.
                      state[:exit_code] = 1
                      nil
                    end

      state[:platform_packaging] = pkg_manager
      pkg_manager
    end

    # Return a text report of installed packages that match a regex
    #
    # @return [String] a human-readble report of matching packages
    #   along with a list of any changes made to files managed by
    #   the package
    def query_packages_matching(regex)
      result = ''
      acsiibar = '=' * 80
      case pkg_manager
      when 'rpm'
        packages = exec_return_result(%(rpm --query --all | grep --extended-regexp '#{regex}'))
        result = packages
        packages.lines do |package|
          result << "\nPackage: #{package}\n"
          result << exec_return_result(%(rpm --verify #{package}))
          result << "\n#{acsiibar}\n"
        end
      when 'dpkg'
        packages = exec_return_result(%(dpkg-query --show --showformat '${Package}-${Version}\n' | grep --extended-regexp '#{regex}'))
        result = packages
        packages.lines do |package|
          result << "\nPackage: #{package}\n"
          result << exec_return_result(%(dpkg --verify #{package}))
          result << "\n#{acsiibar}\n"
        end
      else
        log.warn('query_packages_matching: unable to list packages: no package manager for this OS')
        result = 'no package manager for this OS'
      end
      result
    end

    # Query a package and report if it is installed
    #
    # @return [Boolean] a boolean value indicating whether the package is
    #   installed.
    def package_installed?(package)
      status = false
      state[:installed_packages] ||= {}
      return state[:installed_packages][package] if state[:installed_packages].key?(package)

      case pkg_manager
      when 'rpm'
        status = exec_return_result(%(rpm --query --info #{package})) =~ %r{Version}
      when 'dpkg'
        status = exec_return_status(%(dpkg-query  --show #{package}))
      else
        log.warn('package_installed: unable to query package for platform: no package manager for this OS')
      end

      state[:installed_packages][package] = status
      status
    end

    #===========================================================================
    # Output
    #===========================================================================

    # Execute a command and append the results to an output file
    #
    # @param command_line [String] Command line to execute.
    # @param dst [String] Destination directory for output.
    # @param file [String] File under `dst` where output should be appended.
    # @param options [Hash] A Hash of options.
    # @option options timeout [Integer] Optional number of seconds to allow for
    #   command execution. Defaults to 0 which disables the timeout.
    # @option options stderr [String, nil] An optional additional file to send
    #   stderr to. Stderr is merged into stdout if not provided.
    #
    # @return [true] If the command completes successfully.
    # @return [false] If the command cannot be found, exits with a non-zero code,
    #   or there is an error creating the output path.
    def exec_drop(command_line, dst, file, options = {})
      default_options = {
        'timeout' => 300,
        'stderr' => nil
      }
      options = default_options.merge(options)

      command = command_line.split(' ')[0]
      dst_file_path = File.join(dst, file)
      if options['stderr'].nil?
        stderr_dst = '2>&1'
      else
        stderr_dst = "2>> '#{File.join(dst, options['stderr'])}'"
      end
      command_line = %(#{command_line} >> '#{dst_file_path}' #{stderr_dst})
      unless executable?(command)
        log.debug('exec_drop: command not found: %{command} cannot execute: %{command_line}' %
                  {command: command,
                   command_line: command_line})
        return false
      end
      log.debug('exec_drop: appending output of: %{command_line} to: %{dst_file_path}' %
                {command_line: command_line,
                 dst_file_path: dst_file_path})

      if noop?
        display(' (noop) Collecting output of: %{command_line}' %
                {command_line: command_line})
        return
      else
        display(' ** Collecting output of: %{command_line}' %
                {command_line: command_line})
      end

      return false unless create_path(dst)

      exec_return_status(command_line, options['timeout'])
    end

    # Append data to an output file
    #
    # @param data [String] Data to append.
    # @param dst [String] Destination directory for output.
    # @param file [String] File under `dst` where data should be appended.
    #
    # @return [true] If data output succeeds.
    # @return [false] If there is an error creating the output path.
    def data_drop(data, dst, file)
      dst_file_path = File.join(dst, file)
      log.debug('data_drop: appending to: %{dst_file_path}' %
                {dst_file_path: dst_file_path})

      if noop?
        display(' (noop) Adding data to: %{dst_file_path}' %
                {dst_file_path: dst_file_path})
        return
      else
        display(' ** Adding data to: %{dst_file_path}' %
                {dst_file_path: dst_file_path})
      end

      return false unless create_path(dst)

      File.open(dst_file_path, 'a') { |file| file.puts(data) }
      true
    end

    # Compress file to a destination directory
    #
    # @param src [String] File to output.
    # @param dst [String] Destination directory for output.
    # @param options [Hash] A Hash of options.
    # @option options recreate_parent_path [Boolean] Whether to re-create parent
    #   directories of the `src` underneath `dst`. Defaults to `true`.
    #
    # @return [true] If the compress command succeeds.
    # @return [false] If the compress command exits with an error,
    #   or if there is an error creating the output path.
    def compress_drop(src, dst, options = {})
      default_options = { 'recreate_parent_path' => true }
      options = default_options.merge(options)

      log.debug('compress_drop: compressing: %{src} to: %{dst} with options: %{options}' %
                {src: src,
                 dst: dst,
                 options: options})

      unless File.readable?(src)
        log.debug('compress_drop: source not readable: %{src}' %
                  {src: src})
        return false
      end

      if noop?
        display(' (noop) Compressing: %{src}' %
                {src: src})
        return
      else
        display(' ** Compressing: %{src}' %
                {src: src})
      end

      if options['recreate_parent_path']
        dst_file = File.join(dst, "#{src}.gz")
        dst = File.dirname(dst_file)
      else
        dst_file = File.join(dst, "#{File.basename(src)}.gz")
      end
      command_line = %(gzip -c '#{src}' > '#{dst_file}' && touch -c -r '#{src}' '#{dst_file}')

      return false unless create_path(dst)

      exec_return_status(command_line)
    end

    # Copy directories or files to a destination directory
    #
    # @param src [String] Source directory for output.
    # @param dst [String] Destination directory for output.
    # @param options [Hash] A Hash of options.
    # @option options recreate_parent_path [Boolean] Whether to re-create parent
    #   directories of files in `src` underneath `dst`. Defaults to `true`.
    # @option options cwd [String, nil] Change to the directory given by `cwd`
    #   before copying `src`s as relative paths.
    # @option options age [Integer] Specifies maximum age, in days, to filter list
    #   of copied files.
    #
    # @return [true] If the copy command succeeds.
    # @return [false] If the copy command exits with an error,
    #   or if there is an error creating the output path.
    def copy_drop(src, dst, options = {})
      default_options = {
        'recreate_parent_path' => true,
        'cwd' => nil,
        'age' => nil
      }
      options = default_options.merge(options)

      log.debug('copy_drop: copying: %{src} to: %{dst} with options: %{options}' %
                {src: src,
                 dst: dst,
                 options: options})

      expanded_path = File.join(options['cwd'].to_s, src)
      unless File.readable?(expanded_path)
        log.debug('copy_drop: source not readable: %{src}' %
                  {src: expanded_path})
        return false
      end

      if noop?
        display(' (noop) Copying: %{src}' %
                {src: expanded_path})
        return
      else
        display(' ** Copying: %{src}' %
                {src: expanded_path})
      end

      parents_option = options['recreate_parent_path'] ? ' --parents' : ''
      cd_option = options['cwd'].nil? ? '' : "cd '#{options['cwd']}' && "

      if options['age'].nil?
        recursive_option = File.directory?(src) ? ' --recursive' : ''
        command_line = %(#{cd_option}cp --dereference --preserve #{parents_option} #{recursive_option} '#{src}' '#{dst}')
      else
        age_filter = (options['age'].is_a?(Integer) && (options['age'] > 0)) ? " -mtime -#{options['age']}" : ''
        command_line = %(#{cd_option}find '#{src}' -type f #{age_filter} -exec cp --dereference --preserve #{parents_option} --target-directory '#{dst}' {} +)
      end

      return false unless create_path(dst)

      exec_return_status(command_line)
    end

    # Recursively create a directory
    #
    # @param path [String] Path to the directory to create.
    # @param options [Hash] A Hash of FileUtils.mkdir_p options.
    #
    # @return [true] If directory exists or creation is successful.
    # @return [false] If directory creation fails.
    def create_path(path, options = {})
      default_options = { :noop => noop? }
      options = default_options.merge(options)
      FileUtils.mkdir_p(path, **options)
      true
    rescue => e
      log.error("%{exception_class} raised when creating directory: %{message}\n\t%{backtrace}" %
                {exception_class: e.class,
                 message: e.message,
                 backtrace: e.backtrace.join("\n\t")})
      false
    end
  end

  # Base class for diagnostic logic
  #
  # Instances of classes inheriting from `Check` represent diagnostics to
  # be executed. Subclasses may define a {#setup} method that can use
  # {Confinable#confine} to constrain when checks are executed. All subclasses
  # must define a {#run} method that executes the diagnostic.
  #
  # @abstract
  class Check
    include Configable
    include Confinable
    include DiagnosticHelpers

    # Initialize a new check
    #
    # @note This method should not be overriden by child classes. Override
    #   the #{setup} method instead.
    # @return [void]
    def initialize(parent = nil, **options)
      initialize_configable
      initialize_confinable
      @parent = parent
      @name = options[:name]

      setup(**options)

      if @name.nil?
        raise ArgumentError, '%{class} must be initialized with a name: parameter.' %
          {class: self.class.name}
      end
    end

    # Return a string representing the name of this check
    #
    # If initialized with a parent object, the return value of calling
    # `name` on the parent is pre-pended as a namespace.
    #
    # @return [String]
    def name
      return @resolved_name if defined?(@resolved_name)

      @resolved_name = if @parent.nil? || @parent.name.empty?
                         @name
                       else
                         [@parent.name, @name].join('.')
                       end

      @resolved_name.freeze
    end

    # Initialize variables and logic used by the check
    #
    # @param [Hash] options a hash of configuration options that can be used
    #   to initialize the check.
    # @return [void]
    def setup(**options)
    end

    # Execute the diagnostic represented by the check
    #
    # @return [void]
    def run
      raise NotImplementedError, 'A subclass of Check must provide a run method.'
    end
  end

  # Base class for grouping and managing diagnostics
  #
  # Instances of classes inheriting from `Scope` managage the configuration
  # and execution of a setion of children, which can be {Check} objects or
  # other `Scope` objects. Subclasses may define a {#setup} method that can use
  # {Confinable#confine} to constrain when the scope executes.
  class Scope
    include Configable
    include Confinable
    include DiagnosticHelpers

    attr_reader :children

    # Data for initializing children
    #
    # @return [Array<Array(Class, Hash)>]
    def self.child_specs
      @child_specs ||= []
    end

    # Add a child to be initialized by instances of this scope
    #
    # @param [Class] klass the class from which the child should be
    #   initialized.
    # @param [Hash] options a hash of options to pass when initializing
    #   the child.
    # @return [void]
    def self.add_child(klass, **options)
      child_specs.push([klass, options])
    end

    # Initialize a new scope
    #
    # @note This method should not be overriden by child classes. Override
    #   the #{setup} method instead.
    # @return [void]
    def initialize(parent = nil, **options)
      initialize_configable
      initialize_confinable
      @parent = parent
      @name = options[:name]

      setup(**options)
      if @name.nil?
        raise ArgumentError, '%{class} must be initialized with a name: parameter.' %
          {class: self.class.name}
      end

      initialize_children
    end

    # Return a string representing the name of this scope
    #
    # If initialized with a parent object, the return value of calling
    # `name` on the parent is pre-pended as a namespace.
    #
    # @return [String]
    def name
      return @resolved_name if defined?(@resolved_name)

      @resolved_name = if @parent.nil? || @parent.name.empty?
                         @name
                       else
                         [@parent.name, @name].join('.')
                       end

      @resolved_name.freeze
    end

    # Execute run logic for suitable children
    #
    # This method loops over all child instances and calls `run` on each
    # instance for which {Confinable#suitable?} returns `true`.
    #
    # @return [void]
    def run
      @children.each do |child|
        next unless child.enabled? && child.suitable?
        start_time = Process.clock_gettime(Process::CLOCK_MONOTONIC, :float_second)
        log.info('starting evaluation of: %{name}' %
                 {name: child.name})

        if child.class < Check
          display('Evaluating check: %{name}' %
                  {name: child.name})
        else
          display("\nEvaluating scope: %{name}" %
                  {name: child.name})
        end

        begin
          child.run
        rescue => e
          log.error("%{exception_class} raised during %{name}: %{message}\n\t%{backtrace}" %
                    {exception_class: e.class,
                     name: child.name,
                     message: e.message,
                     backtrace: e.backtrace.join("\n\t")})
        end

        end_time = Process.clock_gettime(Process::CLOCK_MONOTONIC, :float_second)
        log.debug('finished evaluation of %{name} in %<time>.3f seconds' %
                  {name: child.name,
                   time: (end_time - start_time)})
      end
    end

    # Recursively print a description of each child to stdout
    #
    # @return [void]
    def describe
      @children.each do |child|
        next unless child.suitable?

        if child.enabled?
          display child.name
        else
          display child.name + ' (opt-in with --enable)'
        end

        if child.is_a?(Scope)
          child.describe
        end
      end
    end

    # Initialize variables and logic used by the scope
    #
    # @param [Hash] options a hash of configuration options that can be used
    #   to initialize the scope.
    # @return [void]
    def setup(**options)
    end

    private

    def initialize_children
      # Add any globs to the enable_list
      enable_list = (settings[:only] + settings[:enable]).select {|e| e.start_with?(name) or e.include?('*')}

      @children = self.class.child_specs.map do |(klass, opts)|
        begin
          child = klass.new(self, **opts)
        rescue => e
          log.error("%{exception_class} raised when initializing %{klass} in scope '%{scope}': %{message}\n\t%{backtrace}" %
                    {exception_class: e.class,
                     klass: klass.name,
                     scope: self.name,
                     message: e.message,
                     backtrace: e.backtrace.join("\n\t")})
          next
        end

        child.enabled = false if (not self.enabled?)

        # Handle disabling things that do not appear in the `only` list
        if (not settings[:only].empty?) && child.enabled?
          # Disable children unless they are explicitly enabled, or one of
          # their parents is explicitly enabled.
          child.enabled = false unless enable_list.any? {|e| child.name.start_with?(e)}
        end

        # Handle enabling things that appear in the `enabled` or `only` lists
        if (klass < Scope)
          # Enable Scopes if they are explicitly enabled, or one of their
          # children is explicitly enabled, or one of their children matches
          # any globs from :only
          child.enabled = true if enable_list.any? {|e|
            e.start_with?(child.name) or File.fnmatch(e, child.name) or child.children.map(&:name).any? {|c| File.fnmatch(e, c) }
          }
        elsif (klass < Check)
          # Enable Checks if they are explicitly enabled or if the name
          # matches any globs from :only
          # Produce a warning if any sensitive checks would be enabled by the glob
          child.enabled = true if enable_list.include?(child.name) or enable_list.any? {|e|
            if child.sensitive? and File.fnmatch(e, child.name)
              log.warn("Check #{child.name} not enabled via wildcard. Please enable explicitly.")
            end

            File.fnmatch(e, child.name) and !child.sensitive?
          }
        end

        # Handle the `disable` list
        child.enabled = false if settings[:disable].any? {|d| child.name == d or File.fnmatch(d, child.name) }

        child
      end

      # Remove any children that failed to initialize
      @children.compact!
    end
  end

  # A generic check for collecting files
  #
  # This check gathers a list of files and directories, subject to limits on
  # age and disk space. The file list may include glob expressions which are
  # expanded.
  class Check::GatherFiles < Check
    def setup(**options)
      # TODO: assert that @files is a match for
      # Array[Struct[{from    => Optional[String[1]],
      #               copy    => Array[String[1], 1],
      #               to      => String[1],
      #               max_age => Optional[Integer]}]]
      @files = options[:files]

      # TODO: Solaris and AIX do `df` in their own very, very special ways.
      #       `df` and `find -ls` on non-Linux returns 512 byte blocks instead of 1024.
      #       The copy_drop* functions assume flags that are specific to
      #       GNU `cp`.
      confine(kernel: 'linux')
    end

    def run
      @files.map! do |batch|
        result = batch.dup
        result[:max_age] ||= settings[:log_age]
        # Resolve any globs in the copy array to a single list of files.
        result[:copy] = if batch[:from].nil?
                          Dir.glob(result[:copy])
                        else
                          base = Pathname.new(batch[:from])
                          absolute_paths = result[:copy].map {|p| base.join(p) }
                          Pathname.glob(absolute_paths).map {|p| p.relative_path_from(base).to_s }
                        end
        result[:to] = File.join(state[:drop_directory], result[:to])

        result
      end

      return unless disk_available?

      @files.each do |batch|
        batch[:copy].each do |src|
          copy_drop(src, batch[:to], { 'age' => batch[:max_age], 'cwd' => batch[:from] })
        end
      end
    end

    # Check there is enough disk space to copy the logs
    #
    # @return [true, false] A boolean value indicating whether enough
    #   space is available.
    def disk_available?
      return true if noop?

      df_output = exec_return_result("df '#{state[:drop_directory]}'|tail -n1|tr -s ' '|cut -d' ' -f4").chomp
      free = Integer(df_output) rescue nil

      if free.nil?
        log.error('Could not determine disk space available on %{drop_dir}, df returned: %{output}'%
                  {drop_dir: state[:drop_directory],
                   output: df_output})
        return false
      end

      required = 0
      @files.each do |batch|
        cd_option = batch[:from].nil? ? '' : "cd '#{batch[:from]}' && "
        age_filter = (batch[:max_age].is_a?(Integer) && (batch[:max_age] > 0)) ? " -mtime -#{batch[:max_age]}" : ''
        batch[:copy].each do |f|
          used = exec_return_result("#{cd_option}find '#{f}' -type f #{age_filter} -ls|awk '{total=total+$2}END{print total}'").chomp
          required += Integer(used) unless used.empty?
        end
      end

      # We require double the free space as we copy the file, then copy it
      # again into a compressed archive.
      if ((required * 2) > free)
        log.error("Not enough free disk space in %{output_dir} to gather %{name}.\nAvailable: %{available} MB, Required: %{required} MB" %
                  {output_dir: settings[:dir],
                   name: self.name,
                   # Convert 1024 byte blocks to MB
                   available: (free) / 1024,
                   required: (required * 2) / 1024})

        false
      else
        true
      end
    end
  end

  # Gather basic diagnostics
  #
  # This check produces:
  #
  #   - A metadata.json file that contains the version of the support
  #     script that is running, the Puppet ticket number, if supplied,
  #     and the time at which the script was run.
  class Check::BaseStatus < Check
    # The base check is always suitable
    def suitable?
      true
    end

    # The base check is always enabled
    def enabled?
      true
    end

    def run
      metadata = JSON.pretty_generate(version: PuppetX::Puppetlabs::SupportScript::VERSION,
                                      ticket: settings[:ticket],
                                      timestamp: state[:start_time].iso8601(3))
      metadata_file = File.join(state[:drop_directory], 'metadata.json')

      data_drop(metadata, state[:drop_directory], 'metadata.json')
    end
  end

  # Base scope which includes all other scopes
  class Scope::Base < Scope
    # The base scope is always suitable
    def suitable?
      true
    end

    # The base scope is always enabled
    def enabled?
      true
    end

    self.add_child(Check::BaseStatus, name: 'base-status')
  end

  # Gather operating system configuration
  #
  # This check gathers:
  #
  #   - A copy of /etc/hosts
  #   - A copy of /etc/nsswitch.conf
  #   - A copy of /etc/resolv.conf
  #   - Configuration for the apt, yum, and dnf package managers
  #   - Lock/pin info for apt, yum and zypper
  #   - The operating system version
  #   - The umask in effect
  #   - The status of SELinux
  #   - Kernel parameters from sysctl
  #   - A list of configured network interfaces
  #   - A list of configured firewall rules
  #   - A list of loaded firewall kernel modules
  class Check::SystemConfig < Check
    CONF_FILES = ['apt/apt.conf.d',
                  'apt/sources.list.d',
                  'apt/sources.list',
                  'apt/preferences',
                  'dnf/dnf.conf',
                  'hosts',
                  'nsswitch.conf',
                  'os-release',
                  'resolv.conf',
                  'yum.conf',
                  'yum/pluginconf.d/versionlock.conf',
                  'yum/pluginconf.d/versionlock.list',
                  'yum.repos.d',
                  'zypp/locks'].map { |f| File.join('/etc', f) }

    def run
      output_directory = File.join(state[:drop_directory], 'system')
      return false unless create_path(output_directory)

      exec_drop('lsb_release -a',       output_directory, 'lsb_release.txt')
      exec_drop('sestatus',             output_directory, 'selinux.txt')
      exec_drop('sysctl -a',            output_directory, 'sysctl.txt')
      exec_drop('umask',                output_directory, 'umask.txt')
      exec_drop('uname -a',             output_directory, 'uname.txt')

      CONF_FILES.each do |file|
        copy_drop(file, output_directory)
      end

      output_directory = File.join(state[:drop_directory], 'networking')
      return false unless create_path(output_directory)

      data_drop(Facter.value('networking')['fqdn'], output_directory, 'hostname_output.txt')
      exec_drop('ifconfig -a',        output_directory, 'ifconfig.txt')
      exec_drop('iptables -L -n',        output_directory, 'ip_tables.txt')
      exec_drop('ip6tables -L -n',       output_directory, 'ip_tables.txt')

      unless executable?('iptables')
        exec_drop('lsmod | grep ip', output_directory, 'ip_modules.txt')
      end

      # Create symlinks for compatibility with SOScleaner
      #   https://github.com/RedHatGov/soscleaner
      FileUtils.mkdir(File.join(state[:drop_directory], 'etc'), :noop => noop?)
      FileUtils.ln_s('networking/hostname_output.txt',
                      File.join(state[:drop_directory], 'hostname'), :noop => noop?)
      FileUtils.ln_s('../system/etc/hosts',
                      File.join(state[:drop_directory], 'etc/hosts'), :noop => noop?)
    end
  end

  # Gather operating system logs
  #
  # This check gathers:
  #
  #   - A copy of the system log
  #   - A copy of the kernel log
  class Check::SystemLogs < Check::GatherFiles
    def run
      output_directory = File.join(state[:drop_directory], 'logs')
      return false unless create_path(output_directory)

      compress_drop('/var/log/messages', output_directory, { 'recreate_parent_path' => false })
      compress_drop('/var/log/syslog', output_directory, { 'recreate_parent_path' => false })
      compress_drop('/var/log/system', output_directory, { 'recreate_parent_path' => false })

      if documented_option?('dmesg', '--ctime')
        if documented_option?('dmesg', '--time-format')
          exec_drop('dmesg --ctime --time-format iso', output_directory, 'dmesg.txt')
        else
          exec_drop('dmesg --ctime', output_directory, 'dmesg.txt')
        end
      else
        exec_drop('dmesg', output_directory, 'dmesg.txt')
      end

      if (! noop?) && executable?('journalctl')
        age_filter = (settings[:log_age].is_a?(Integer) && (settings[:log_age] > 0)) ? " --since '#{settings[:log_age]} days ago'" : ''

        exec_drop("journalctl --full --output=short-iso #{age_filter} | gzip -f9", output_directory, "journalctl.log.gz")
      end

      super
    end
  end

  # Gather operating system diagnostics
  #
  # This check gathers:
  #
  #   - A list of variables set in the environment
  #   - A list of running processes
  #   - A list of enabled services
  #   - A list of systemd timers
  #   - The uptime of the system
  #   - A list of established network connections
  #   - TCP statistics from nstat
  #   - NTP status
  #   - The IP address and hostname of the node according to DNS
  #   - Disk usage
  #   - RAM usage
  class Check::SystemStatus < Check
    def run
      output_directory = File.join(state[:drop_directory], 'system')
      return false unless create_path(output_directory)

      exec_drop('env',                  output_directory, 'env.txt')
      exec_drop('ps -aux',              output_directory, 'ps_aux.txt')
      exec_drop('ps -e f',               output_directory, 'ps_tree.txt')
      exec_drop('chkconfig --list',     output_directory, 'services.txt')
      exec_drop('svcs -a',              output_directory, 'services.txt')
      exec_drop('systemctl list-units', output_directory, 'services.txt')
      exec_drop('systemctl list-timers', output_directory, 'timers.txt')
      exec_drop('uptime',               output_directory, 'uptime.txt')

      output_directory = File.join(state[:drop_directory], 'networking')
      return false unless create_path(output_directory)

      exec_drop('netstat -anptuo',     output_directory, 'ports.txt')
      exec_drop('nstat -az',           output_directory, 'nstat_output.txt')
      exec_drop('ntpq -p',            output_directory, 'ntpq_output.txt')

      unless noop?
        command = %[ping -t1 -c1 '#{Facter.value('networking')['fqdn']}'|head -n1|tr -ds '()' ' '|cut -d ' ' -f3]
        ip_address = exec_return_result(command)

        unless ip_address.empty?
          data_drop(ip_address, output_directory, 'guessed_ip_address.txt')
          exec_drop("getent hosts '#{ip_address}'", output_directory, 'mapped_hostname_from_guessed_ip_address.txt')
        end
      end

      output_directory = File.join(state[:drop_directory], 'resources')
      return false unless create_path(output_directory)

      exec_drop('df -h',   output_directory, 'df_output.txt')
      exec_drop('df -i',   output_directory, 'df_output.txt')
      exec_drop('df -k',   output_directory, 'df_inodes_output.txt')
      exec_drop('free -h', output_directory, 'free_mem.txt')
    end
  end

  # Scope which collects diagnostics from the operating system
  #
  # @todo Should confine to *NIX and have a seperate scope for Windows.
  class Scope::System < Scope
    Scope::Base.add_child(self, name: 'system')

    self.add_child(Check::SystemConfig, name: 'config')
    self.add_child(Check::SystemLogs, name: 'logs',
                   files: [{from: '/var/log',
                            copy: ['dnf*.log',
                                   'apt/', 'dpkg.log',
                                   'zypp/', 'zypper.log'],
                            to: 'logs'}])

    self.add_child(Check::SystemStatus, name: 'status')
    self.add_child(Check::GatherFiles,
                   name: 'metrics',
                   files: [{from: '/opt/puppetlabs/puppet-metrics-collector',
                            copy: ['system_cpu/', 'system_memory/', 'system_processes/', 'vmware/'],
                            to: 'metrics'},
                           {from: '/var/log',
                            copy: ['sa/', 'sysstat/'],
                            to: 'metrics'}])

    def setup(**options)
      # TODO: Many diagnostics contained here are also applicable to Solaris
      #       AIX, and macOS, but should be reviewed and possibly split out
      #       to a separate set of checks or scope. Windows needs its own
      #       things.
      confine(kernel: 'linux')
    end
  end

  # A check for gathering log files and journalctl data
  class Check::ServiceLogs < Check::GatherFiles
    # This method takes either or both of two options: :services or :units
    # :services is an array of strings that get passed to journalctl --unit
    # :units is a array of [UNIT=<service>, SYSLOG_IDENTIFIER=<syslogid>] hashes used
    #   to construct a command like journalctl UNIT=<service>.service + SYSLOG_IDENTIFIER=<syslogid>
    #   Note that the '.service' extension is needed here
    # This is used to ensure we capture all relevent output, i.e. https://github.com/systemd/systemd/issues/2913
    # TODO: accept an array of syslogids in case we need to pair more than one with a given service
    def setup(**options)
      super

      @services = options[:services]
      @units = options[:units]
      # We can process both :services and :units as long as there are no name collisions
      # But we need at least one service from either of them
      if @services.nil? && @units.nil?
        raise ArgumentError, 'Check::ServiceLogs must be initialized with either or both of :services and :units'
      end
    end

    def run
      super

      if (! noop?) && executable?('journalctl')
        age_filter = (settings[:log_age].is_a?(Integer) && (settings[:log_age] > 0)) ? " --since '#{settings[:log_age]} days ago'" : ''

        unless @units.nil?
          @units.each do |unit|
            munged_name = unit['UNIT'].sub('pe-', '').sub('*', '').split('.').first
            # Turns a hash like {'UNIT' => 'pe-puppetserver.service', 'SYSLOG_IDENTIFIER' => 'puppetserver'}
            # into a string 'UNIT=pe-puppetserver.service + SYSLOG_IDENTIFIER=puppetserver'
            journal_args = unit.map{|unit,syslogid| "#{unit}=#{syslogid}"}.join(' + ')
            log_directory = File.join(state[:drop_directory], 'logs', munged_name)
            next unless create_path(log_directory)
            exec_drop("journalctl --full --output=short-iso #{journal_args} #{age_filter}", log_directory, "#{munged_name}-journalctl.log")
          end
        end

        unless @services.nil?
          @services.each do |service|
            munged_name = service.sub('pe-', '').sub('*', '').split('.').first
            log_directory = File.join(state[:drop_directory], 'logs', munged_name)
            next unless create_path(log_directory)

            exec_drop("journalctl --full --output=short-iso --unit #{service} #{age_filter}", log_directory, "#{munged_name}-journalctl.log")
          end
        end
      end
    end
  end

  # A check for gathering configuration files with redaction
  class Check::ConfigFiles < Check::GatherFiles
    def run
      super

      unless noop?
        @files.each do |batch|
          batch[:copy].each do |file|
            exec_return_result("cd '#{batch[:to]}' && find '#{file}' -type f -exec sed --in-place '/password/d' {} +")
          end
        end
      end
    end
  end

  # A check for gathering runtime information about a set of services
  class Check::ServiceStatus < Check
    def setup(**options)
      @services = options[:services]
      if @services.nil? || (! @services.is_a?(Array))
        raise ArgumentError, 'Check::ServiceStatus must be initialized with a list of strings for the services: parameter.'
      end

      @service_pids = {}

      # Everything here is specific to systemd or sysvinit
      confine(kernel: 'linux')
    end

    def run
      output_directory = File.join(state[:drop_directory], 'system')
      return false unless create_path(output_directory)

      if (! noop?) && executable?('systemctl')
        @services.each do |service|
          exec_drop("systemctl status '#{service}.service'", output_directory, 'systemctl-status.txt')
        end
      end

      unless noop?
        cgroup_v2 = File.exist?('/sys/fs/cgroup/cgroup.controllers')

        @services.each do |service|
          @service_pids[service] = []

          cgroup_pidfile = if cgroup_v2
                             "/sys/fs/cgroup/system.slice/#{service}.service/cgroup.procs"
                           else
                             "/sys/fs/cgroup/systemd/system.slice/#{service}.service/cgroup.procs"
                           end

          if File.file?(cgroup_pidfile)
            File.readlines(cgroup_pidfile).each do |line|
              pid = line.to_i
              # to_i returns 0 if String -> Integer conversion fails
              @service_pids[service] << pid if pid.positive?
            end
          end

          # The service_pids array will be empty if the service is stopped,
          # or the Linux OS is not using SystemD for some reason.
          next if @service_pids[service].empty?

          proc_directory = File.join(output_directory, 'proc', service)

          @service_pids[service].each do |pid|
            proc_pid_directory = File.join(output_directory, 'proc', service, pid.to_s)
            return false unless create_path(proc_pid_directory)
            ['cmdline','limits','environ', 'smaps'].each do |procfile|
              copy_drop("/proc/#{pid}/#{procfile}", proc_pid_directory, { 'recreate_parent_path' => false })
            end
          end

          if @service_pids[service].first
            # This readlink can fail for a variety of reasons, most often a
            # process ending between when PIDs were listed and when we go
            # to gather data.
            real_executable = File.readlink("/proc/#{@service_pids[service].first}/exe") rescue nil
            data_drop(real_executable, proc_directory, 'exe') unless real_executable.nil?
          end
          # Files copied from /proc are read-only by default. Add write
          # permissions to the copy so that files extracted from support
          # bundles can be easily removed.
          FileUtils.chmod_R('u+wX', proc_directory)

          # Grab CGroup settings for the service.
          cgroup_dirs = if cgroup_v2
                          ["/sys/fs/cgroup/system.slice/#{service}.service/"]
                        else
                          Dir.glob("/sys/fs/cgroup/{memory,cpu,blkio,devices,pids,systemd}/system.slice/#{service}.service")
                        end

          cgroup_dirs.each do |dir|
            copy_drop(dir, output_directory)
          end
          FileUtils.chmod_R('u+wX', "#{output_directory}/sys") if File.exist?("#{output_directory}/sys")
        end
      end
    end
  end


  # Check the status of components in the puppet-agent package
  #
  # This check gathers:
  #
  #   - Facter output and debug-level logs
  #   - A list of gems installed in the Puppet Ruby environment
  #   - Whether Puppet's configured server hostname responds to a ping
  #   - A copy of classes.txt, graphs/, last_run_summary.yaml, and
  #     resources.txt from Puppet's statedir.
  #   - A listing of files present in the following directories:
  #     * /etc/puppetlabs
  #     * /var/log/puppetlabs
  #     * /opt/puppetlabs
  #   - A listing of Puppet and PE packages installed on the system
  #     along with verification output for each.
  class Check::PuppetAgentStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')
      sys_directory = File.join(state[:drop_directory], 'system')
      net_directory = File.join(state[:drop_directory], 'networking')
      find_directory = File.join(state[:drop_directory], 'enterprise', 'find')

      exclude_directories = ['.snapshot',
                             '.snapshots',
                             '.backup'].join(' -o -name ')

      exec_drop("#{PUP_PATHS[:puppet_bin]}/facter --puppet --json --debug --timing", sys_directory, 'facter_output.json', stderr: 'facter_output.debug.log')
      exec_drop("#{PUP_PATHS[:puppet_bin]}/gem list --local", ent_directory, 'puppet_gems.txt')

      unless noop? || (puppet_server = puppet_conf('server', 'agent')).empty?
        exec_drop("ping -c 1 #{puppet_server}", net_directory, 'puppet_ping.txt')
      end

      unless (statedir = puppet_conf('statedir', 'agent')).empty?
        output_dir = File.join(state[:drop_directory], 'enterprise', 'state')

        ['classes.txt', 'graphs/', 'last_run_summary.yaml', 'resources.txt'].each do |file|
          copy_drop(file, output_dir, { 'age' => -1, 'cwd' => statedir })
        end
      end

      ['/etc/puppetlabs', '/var/log/puppetlabs', '/opt/puppetlabs'].each do |path|
        drop_name = path.gsub('/','_') + '.txt.gz'
        exec_drop("find '#{path}' \\( -name #{exclude_directories} \\) -prune -o -ls | gzip -f9",
                  find_directory,
                  drop_name,
                  # These directory trees can be deep. Allow extra time for traversal.
                  {'timeout' => 600})
      end

      data_drop(query_packages_matching('^pe-|^puppet'), ent_directory, 'puppet_packages.txt')
    end
  end

  # A Check that gathers Puppet agent SSL certificate to diagnosing SSL connection issues
  #
  # This check gathers:
  #
  #   - The puppet agent's certificate: puppet config print hostcert
  #   - The puppet agent's copy of the CA chain: puppet config print localcacert
  #   - The puppet agent's copy of the CRL chain: puppet config print hostcrl
  class Check::PuppetSSL < Check
    def run
      output_directory = File.join(state[:drop_directory], 'enterprise')

      copy_drop(puppet_conf("hostcert"), output_directory)
      copy_drop(puppet_conf("localcacert"), output_directory)
      copy_drop(puppet_conf("hostcrl"), output_directory)
    end
  end


  # Scope which collects diagnostics related to the Puppet-Agent service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for puppet,
  #     facter, and pxp-agent
  #   - Logs from /var/log/puppetlabs for puppet and pxp-agent
  class Scope::PuppetAgent < Scope
    def setup(**options)
      confine { package_installed?('puppet-agent') }
    end

    Scope::Base.add_child(self, name: 'puppet-agent')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['facter/facter.conf',
                                   'puppet/device.conf',
                                   'puppet/hiera.yaml',
                                   'puppet/puppet.conf',
                                   'pxp-agent/modules/',
                                   'pxp-agent/pxp-agent.conf'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['puppet', 'pxp-agent'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['puppet', 'pxp-agent'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['puppet/',
                                   'pxp-agent/'],
                            to: 'logs'}],
                   services: ['puppet', 'pxp-agent'])
    self.add_child(Check::PuppetAgentStatus,
                   name: 'status',
                   services: ['puppet', 'pxp-agent'])
    self.add_child(Check::PuppetSSL, name: 'ssl')
  end



  # Check the status of components related to Puppet Server
  #
  # This check gathers:
  #
  #   - A list of certificates issued by the Puppet CA
  #   - A list of gems installed for use by Puppet Server
  #   - Output from the `status/v1/services` API
  #   - Output from the `puppet/v3/environment_modules` API
  #   - Output from the `puppet/v3/environments` API
  #   - environment.conf and hiera.yaml from each environment
  #   - The disk space used by Code Manager cache, storage, client,
  #     and staging directories.
  #   - The output of `r10k deploy display`
  #   - The disk space used by the server's File Bucket
  class Check::PuppetServerStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')
      res_directory = File.join(state[:drop_directory], 'resources')

      if File.directory?(puppet_conf('cadir', 'master'))
        if SemanticPuppet::Version.parse(Puppet.version) >= SemanticPuppet::Version.parse('6.0.0')
          exec_drop("#{PUP_PATHS[:server_bin]}/puppetserver ca list --all", ent_directory, 'certs.txt')
        else
          exec_drop("#{PUP_PATHS[:puppet_bin]}/puppet cert list --all", ent_directory, 'certs.txt')
        end
      end

      exec_drop("#{PUP_PATHS[:server_bin]}/puppetserver gem list --local", ent_directory, 'puppetserver_gems.txt')

      data_drop(curl_cert_auth('https://127.0.0.1:8140/status/v1/services?level=debug'), ent_directory, 'puppetserver_status.json')
      data_drop(curl_cert_auth('https://127.0.0.1:8140/puppet/v3/environment_modules'), ent_directory, 'modules.json')
      data_drop(curl_cert_auth('https://127.0.0.1:8140/analytics/v1/collections/snapshots'), ent_directory, 'analytics_snapshot.json')

      # Collect data using environments from the puppet/v3/environments endpoint.
      # Equivalent to puppetserver_environments() in puppet-enterprise-support.sh
      puppetserver_environments_json = curl_cert_auth('https://127.0.0.1:8140/puppet/v3/environments')
      data_drop(puppetserver_environments_json, ent_directory, 'puppetserver_environments.json')
      puppetserver_environments = begin
                                    JSON.parse(puppetserver_environments_json)
                                  rescue JSON::ParserError
                                    log.error('PuppetServerStatus: unable to parse puppetserver_environments_json')
                                    {}
                                  end

      puppetserver_environments['environments'].keys.each do |environment|
        environment_manifests = puppetserver_environments['environments'][environment]['settings']['manifest']
        environment_directory = File.dirname(environment_manifests)
        environment_drop_directory = File.join(ent_directory, 'etc/puppetlabs/code/environments', environment)

        copy_drop('environment.conf', environment_drop_directory, { 'recreate_parent_path' => false, 'cwd' => environment_directory })
        copy_drop('hiera.yaml', environment_drop_directory, { 'recreate_parent_path' => false, 'cwd' => environment_directory })
      end unless puppetserver_environments.empty?

      r10k_config = '/opt/puppetlabs/server/data/code-manager/r10k.yaml'
      if File.exist?(r10k_config)
        # Code Manager and File Sync diagnostics
        code_staging_directory = '/etc/puppetlabs/code-staging'
        filesync_directory = '/opt/puppetlabs/server/data/puppetserver/filesync'
        code_manager_cache = '/opt/puppetlabs/server/data/code-manager'
        exec_drop("du -h --max-depth=1 #{code_staging_directory}", res_directory, 'code_staging_sizes_from_du.txt') if File.directory?(code_staging_directory)
        exec_drop("du -h --max-depth=1 #{filesync_directory}", res_directory, 'filesync_sizes_from_du.txt') if File.directory?(filesync_directory)
        exec_drop("du -h --max-depth=1 #{code_manager_cache}", res_directory, 'r10k_cache_sizes_from_du.txt') if File.directory?(code_manager_cache)
        exec_drop("#{PUP_PATHS[:puppet_bin]}/r10k deploy display -p --detail -c #{r10k_config}", ent_directory, 'r10k_deploy_display.txt')
      end

      # Collect Puppet Enterprise File Bucket diagnostics.
      filebucket_directory = '/opt/puppetlabs/server/data/puppetserver/bucket'
      exec_drop("du -sh #{filebucket_directory}", res_directory, 'filebucket_size_from_du.txt') if File.directory?(filebucket_directory)
    end
  end

  # Gathers event data from the PE Analytics Service
  class Check::PuppetServerEvents < Check::GatherFiles
    def setup(**options)
      # Disabled by default as this can end up collecting 10s of
      # 1000s of files which adds a lot of runtime to the support script.
      self.enabled = false

      super(**options)
    end
  end

  # Gather Puppet Server CA state
  #
  # This check gathers:
  #
  #   - The CA certificate bundle
  #   - The CA crl bundle
  #   - The CA inventory file of issued certificates and expiration dates
  #   - The CA serial number file
  class Check::PuppetServerSSL < Check
    def run
      output_directory = File.join(state[:drop_directory], 'enterprise')

      copy_drop(puppet_conf("cacert"), output_directory)
      copy_drop(puppet_conf("cacrl"), output_directory)
      copy_drop(puppet_conf("cert_inventory"), output_directory)
      copy_drop(puppet_conf("serial"), output_directory)
    end
  end

  # Scope which collects diagnostics related to the PuppetServer service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for puppet,
  #     puppetserver, and r10k
  #   - Logs from /var/log/puppetlabs for puppetserver and r10k
  #   - Metrics from /opt/puppetlabs/puppet-metrics-collector
  #     for puppetserver
  #   - Events, such as code deployments, from the primary PE server's analytics service
  class Scope::PuppetServer < Scope
    def setup(**options)
      confine { package_installed?('pe-puppetserver') }
    end

    Scope::Base.add_child(self, name: 'puppetserver')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['code/hiera.yaml',
                                   'puppet/auth.conf',
                                   'puppet/autosign.conf',
                                   'puppet/classfier.yaml',
                                   'puppet/fileserver.conf',
                                   'puppet/hiera.yaml',
                                   'puppet/puppet.conf',
                                   'puppet/puppetdb.conf',
                                   'puppet/routes.yaml',
                                   'puppetserver/bootstrap.cfg',
                                   'puppetserver/code-manager-request-logging.xml',
                                   'puppetserver/conf.d/',
                                   'puppetserver/logback.xml',
                                   'puppetserver/request-logging.xml',
                                   'r10k/r10k.yaml'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from: '/opt/puppetlabs/server/data/code-manager',
                            copy: ['r10k.yaml'],
                            to: 'enterprise/etc/puppetlabs/puppetserver',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-puppetserver'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-puppetserver'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['puppetserver/',
                                   'r10k/'],
                            to: 'logs'}],
                   units: [{'UNIT' => 'pe-puppetserver.service', 'SYSLOG_IDENTIFIER' => 'puppetserver'}])
    self.add_child(Check::GatherFiles,
                   name: 'metrics',
                   files: [{from: '/opt/puppetlabs/puppet-metrics-collector',
                            copy: ['puppetserver/'],
                            to: 'metrics'}])
    self.add_child(Check::PuppetServerStatus,
                   name: 'status',
                   services: ['pe-puppetserver'])

    self.add_child(Check::PuppetServerEvents,
                  name: 'events',
                  files: [{from: '/opt/puppetlabs/server/data/analytics/analytics',
                           copy: ['q/'],
                           to: 'enterprise/puppetserver_events'}])
    self.add_child(Check::PuppetServerSSL, name: 'ssl')
  end

  # Check the status of components related to PuppetDB
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  #   - Output from the `pdb/admin/v1/summary-stats` API
  #   - A list of certnames for nodes that PuppetDB considers to be active
  class Check::PuppetDBStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      # Out of all PE services, PuppetDB occasionally conflicts with other
      # installed software due to its use of ports 8080 and 8081.
      #
      # So, we check to see if an alternate port has been configured.
      puppetdb_port = exec_return_result(%(cat /etc/puppetlabs/puppetdb/conf.d/jetty.ini | tr -d ' ' | grep --extended-regexp '^port=[[:digit:]]+$' | cut -d= -f2)).strip
      puppetdb_port = '8080' if puppetdb_port.empty?

      data_drop(curl_url("http://127.0.0.1:#{puppetdb_port}/status/v1/services?level=debug"), ent_directory, 'puppetdb_status.json')
      data_drop(curl_url("http://127.0.0.1:#{puppetdb_port}/pdb/admin/v1/summary-stats"), ent_directory, 'puppetdb_summary_stats.json')
      data_drop(curl_url("http://127.0.0.1:#{puppetdb_port}/pdb/query/v4",
                         request: 'GET',
                         'data-urlencode': 'query=nodes[certname] {deactivated is null and expired is null}'),
                ent_directory, 'puppetdb_nodes.json')
    end
  end

  # Scope which collects diagnostics related to the PuppetDB service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for puppetdb
  #   - Logs from /var/log/puppetlabs for puppetdb
  #   - Metrics from /opt/puppetlabs/puppet-metrics-collector
  #     for puppetdb
  class Scope::PuppetDB < Scope
    def setup(**options)
      confine { package_installed?('pe-puppetdb') }
    end

    Scope::Base.add_child(self, name: 'puppetdb')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['puppetdb/bootstrap.cfg',
                                   'puppetdb/certificate-allowlist',
                                   'puppetdb/conf.d/',
                                   'puppetdb/logback.xml',
                                   'puppetdb/request-logging.xml'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-puppetdb'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-puppetdb'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['puppetdb/'],
                            to: 'logs'}],
                  units: [{'UNIT' => 'pe-puppetdb.service', 'SYSLOG_IDENTIFIER' => 'puppetdb'}])
    self.add_child(Check::GatherFiles,
                   name: 'metrics',
                   files: [{from: '/opt/puppetlabs/puppet-metrics-collector',
                            copy: ['puppetdb/'],
                            to: 'metrics'}])
    self.add_child(Check::PuppetDBStatus,
                   name: 'status',
                   services: ['pe-puppetdb'])
  end

  # A Check that gathers miscellaneous PE status info
  #
  # This check gathers:
  #
  #   - Status information for the entire PE install
  #   - Current tuning settings
  #   - Recommended tuning settings
  class Check::PeStatus < Check
    def run
      ent_directory = File.join(state[:drop_directory], 'enterprise')

      exec_drop("#{PUP_PATHS[:puppetlabs_bin]}/puppet-infrastructure status --format json", ent_directory, 'pe_infra_status.json')
      exec_drop("#{PUP_PATHS[:puppetlabs_bin]}/puppet-infrastructure tune",                 ent_directory, 'puppet_infra_tune.txt')
      exec_drop("#{PUP_PATHS[:puppetlabs_bin]}/puppet-infrastructure tune --current",       ent_directory, 'puppet_infra_tune_current.txt')
    end
  end

  # A Check that gathers files related to PE file sync state
  #
  # This check uses Check::GatherFiles to collect a copy of directories
  # related to File Sync. It is disabled by default due to the high
  # probablility that these directories contain sensitive data.
  class Check::PeFileSync < Check::GatherFiles
    def setup(**options)
      super(**options)

      self.enabled = false
      self.sensitive = true
      confine { package_installed?('pe-puppetserver') }
    end
  end

  # Scope which collects PE diagnostics
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for the PE installer
  #     and client tools
  #   - Logs from /var/log/puppetlabs for the PE installer and
  #     PE backup services
  class Scope::Pe < Scope
    def setup(**options)
      confine { package_installed?('pe-puppet-enterprise-release') }
    end

    Scope::Base.add_child(self, name: 'pe')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['client-tools/orchestrator.conf',
                                   'client-tools/puppet-access.conf',
                                   'client-tools/puppet-code.conf',
                                   'client-tools/puppetdb.conf',
                                   'client-tools/services.conf',
                                   'enterprise/conf.d/',
                                   'enterprise/hiera.yaml',
                                   'installer/answers.install'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1}])
    self.add_child(Check::GatherFiles,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['installer/',
                                   'pe-backup-tools/',
                                   'puppet_infra_recover_config_cron.log'],
                            to: 'logs'}])
    self.add_child(Check::PeStatus,
                   name: 'status')
    self.add_child(Check::PeFileSync,
                   name: 'file-sync',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['code-staging'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from: '/opt/puppetlabs/server/data/puppetserver',
                            copy: ['filesync'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1}])
  end

  # Check the status of components related to PE Console Services
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  #   - The Directory Service settings, with passwords removed
  class Check::PeConsoleStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_url('http://127.0.0.1:4432/status/v1/services?level=debug'), ent_directory, 'console_status.json')
      # FIXME: The v2/ds API is deprecated. Switch to v2/ldap for appropriate
      #        versions of 2023.x
      console_ds_settings = curl_cert_auth('https://127.0.0.1:4433/rbac-api/v2/ds')
      data_drop(pretty_json(console_ds_settings, %w[password ds_pw_obfuscated]), ent_directory, 'rbac_directory_settings.json')
    end
  end

  # Check the status of components related to PE Console Services
  #
  # This check gathers:
  #
  #   - A listing of classifier groups configured in the console
  #
  # This check is not enabled by default.
  class Check::PeConsoleGroups < Check
    def setup(**options)
      # Disabled by default as classifier groups configuration can contain
      # sensitive data.
      self.enabled = false
      self.sensitive = true
    end

    def run
      ent_directory = File.join(state[:drop_directory], 'enterprise')
      data_drop(curl_cert_auth('https://127.0.0.1:4433/classifier-api/v1/groups'), ent_directory, 'classifier.json')
    end
  end

  # Scope which collects diagnostics related to the PE Console service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for pe-console-services,
  #     and pe-nginx
  #   - Logs from /var/log/puppetlabs for pe-console-services and pe-nginx
  class Scope::Pe::Console < Scope
    def setup(**options)
      confine { package_installed?('pe-console-services') }
    end

    Scope::Pe.add_child(self, name: 'console')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['console-services/bootstrap.cfg',
                                   # NOTE: PE Console stores encryption keys in conf.d/secrets.
                                   #       Therefore, we explicitly list what to gather.
                                   'console-services/conf.d/activity.conf',
                                   'console-services/conf.d/activity-database.conf',
                                   'console-services/conf.d/analytics.conf',
                                   'console-services/conf.d/cache.conf',
                                   'console-services/conf.d/classifier.conf',
                                   'console-services/conf.d/classifier-database.conf',
                                   'console-services/conf.d/console.conf',
                                   'console-services/conf.d/global.conf',
                                   'console-services/conf.d/metrics.conf',
                                   'console-services/conf.d/rbac.conf',
                                   'console-services/conf.d/rbac-database.conf',
                                   'console-services/conf.d/value-report.conf',
                                   'console-services/conf.d/webserver.conf',
                                   'console-services/custom_pql_queries.json',
                                   'console-services/logback.xml',
                                   'console-services/rbac-certificate-allowlist',
                                   'console-services/request-logging-api.xml',
                                   'console-services/request-logging.xml',

                                   'nginx/conf.d/',
                                   'nginx/nginx.conf'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-console-services', 'pe-nginx'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-console-services', 'pe-nginx'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['console-services/',
                                   'nginx/'],
                            to: 'logs'}],
                   services: ['pe-nginx'],
                   units: [{'UNIT' => 'pe-console-services.service', 'SYSLOG_IDENTIFIER' => 'console-services'}])
    self.add_child(Check::PeConsoleStatus,
                   name: 'status',
                   services: ['pe-console-services', 'pe-nginx'])
    self.add_child(Check::PeConsoleGroups,
                   name: 'classifier-groups')
  end

  # Check the status of components related to PE Host Action Collector
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  class Check::PeHostActionCollector < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_url('https://127.0.0.1:8147/status/v1/services?level=debug'),
                ent_directory,
                'host_action_collector_status.json')
    end
  end

  # Scope which collects diagnostics related to the PE Host Action Collector service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for pe-host-action-collector
  #   - Logs from /var/log/puppetlabs for pe-host-action-collector
  #   - Service status from sysctl
  #   - Process status from /proc and /sys/fs/cgroup
  class Scope::Pe::HostActionCollector < Scope
    def setup(**options)
      confine { package_installed?('pe-host-action-collector') }
    end

    Scope::Pe.add_child(self, name: 'host-action-collector')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['host-action-collector/request-logging.xml',
                                   'host-action-collector/conf.d/',
                                   'host-action-collector/logback.xml',
                                   'host-action-collector/bootstrap.cfg'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-host-action-collector'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-host-action-collector'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])

    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['host-action-collector'],
                            to: 'logs'}],
                   units: [{'UNIT' => 'pe-host-action-collector.service', 'SYSLOG_IDENTIFIER' => 'host-action-collector'}])

    self.add_child(Check::PeHostActionCollector,
                   name: 'status',
                   services: ['pe-host-action-collector'])
  end

  # Check the status of components related to the PE Orchestration service
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  class Check::PeOrchestrationStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_cert_auth('https://127.0.0.1:8143/status/v1/services?level=debug'), ent_directory, 'orchestration_status.json')
    end
  end

  # Scope which collects diagnostics related to the PE Orchestration service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for ace-server, bolt-server,
  #     and pe-orchestration-services
  #   - Logs from /var/log/puppetlabs for ace-server, bolt-server,
  #     and pe-orchestration-services
  #   - Metrics from /opt/puppetlabs/puppet-metrics-collector for
  #     pe-orchestration-services
  class Scope::Pe::Orchestration < Scope
    def setup(**options)
      confine { package_installed?('pe-orchestration-services') }
    end

    Scope::Pe.add_child(self, name: 'orchestration')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['ace-server/conf.d/',
                                   'bolt-server/conf.d/',
                                   'orchestration-services/bootstrap.cfg',
                                   # NOTE: The PE Orchestrator stores encryption keys in its conf.d.
                                   #       Therefore, we explicitly list what to gather.
                                   'orchestration-services/conf.d/analytics.conf',
                                   'orchestration-services/conf.d/auth.conf',
                                   'orchestration-services/conf.d/file-sync.conf',
                                   'orchestration-services/conf.d/global.conf',
                                   'orchestration-services/conf.d/inventory.conf',
                                   'orchestration-services/conf.d/metrics.conf',
                                   'orchestration-services/conf.d/orchestrator.conf',
                                   'orchestration-services/conf.d/pcp-broker.conf',
                                   'orchestration-services/conf.d/web-routes.conf',
                                   'orchestration-services/conf.d/webserver.conf',
                                   'orchestration-services/logback.xml',
                                   'orchestration-services/request-logging.xml',
                                   'pe-plan-runner/conf.d/'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-ace-server', 'pe-bolt-server', 'pe-orchestration-services'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-ace-server', 'pe-bolt-server', 'pe-orchestration-services'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['ace-server/',
                                   'bolt-server/',
                                   'orchestration-services/',
                                   'plan-runner/'],
                            to: 'logs'},
                           # Copy all node activity logs without age limit.
                           {from: '/var/log/puppetlabs',
                            copy: ['orchestration-services/aggregate-node-count*.log*'],
                            to: 'logs',
                            max_age: -1}],
                   services: ['pe-ace-server.service', 'pe-bolt-server.service', 'pe-plan-runner.service'],
                   units: [{'UNIT' => 'pe-orchestration-services.service', 'SYSLOG_IDENTIFIER' => 'orchestration-services'}])
    self.add_child(Check::GatherFiles,
                   name: 'metrics',
                   files: [{from: '/opt/puppetlabs/puppet-metrics-collector',
                            copy: ['ace/', 'bolt/', 'orchestrator/'],
                            to: 'metrics'}])
    self.add_child(Check::PeOrchestrationStatus,
                   name: 'status',
                   services: ['pe-ace-server', 'pe-bolt-server', 'pe-orchestration-services', 'pe-plan-runner'])
  end

  # Check the status of components related to PE Patching Service
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  class Check::PePatchingService < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_url('https://127.0.0.1:8146/status/v1/services?level=debug'),
                ent_directory,
                'pe_patching_service_status.json')
    end
  end

  # Scope which collects diagnostics related to the PE Patching Service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for patching-service
  #   - Logs from /var/log/puppetlabs for patching-service
  #   - Service status from sysctl
  #   - Process status from /proc and /sys/fs/cgroup
  class Scope::Pe::PatchingService < Scope
    def setup(**options)
      confine { package_installed?('pe-patching-service') }
    end

    Scope::Pe.add_child(self, name: 'patching-service')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['patching-service/request-logging.xml',
                                   'patching-service/conf.d/',
                                   'patching-service/logback.xml',
                                   'patching-service/bootstrap.cfg'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-patching-service'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-patching-service'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])

    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['patching-service'],
                            to: 'logs'}],
                   units: [{'UNIT' => 'pe-patching-service.service', 'SYSLOG_IDENTIFIER' => 'patching-service'}])

    self.add_child(Check::PePatchingService,
                   name: 'status',
                   services: ['pe-patching-service'])
  end

  # Check the status of components related to PE Workflow Service
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API
  class Check::PeWorkflowService < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_url('https://127.0.0.1:8148/status/v1/services?level=debug'),
                ent_directory,
                'pe_workflow_service_status.json')
    end
  end

  # Scope which collects diagnostics related to the PE Workflow Service
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs for workflow-service
  #   - Logs from /var/log/puppetlabs for workflow-service
  #   - Service status from sysctl
  #   - Process status from /proc and /sys/fs/cgroup
  class Scope::Pe::WorkflowService < Scope
    def setup(**options)
      confine { package_installed?('pe-workflow-service') }
    end

    Scope::Pe.add_child(self, name: 'workflow-service')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['workflow-service/bootstrap.cfg',
                                   'workflow-service/logback.xml',
                                   'workflow-service/request-logging.xml',
                                   'workflow-service/conf.d/analytics.conf',
                                   'workflow-service/conf.d/global.conf',
                                   'workflow-service/conf.d/metrics.conf',
                                   'workflow-service/conf.d/pe-workflow-service.conf',
                                   'workflow-service/conf.d/pe-workflow-service-database.conf',
                                   'workflow-service/conf.d/webserver.conf'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-workflow-service'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-workflow-service'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])

    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['workflow-service'],
                            to: 'logs'}],
                   units: [{'UNIT' => 'pe-workflow-service.service', 'SYSLOG_IDENTIFIER' => 'workflow-service'}])

    self.add_child(Check::PeWorkflowService,
                   name: 'status',
                   services: ['pe-workflow-service'])
  end

  # Check the status of components related to PE Infrastructure Assistant
  #
  # This check gathers:
  #
  #   - Output from the `status/v1/services` API for port 8145
  class Check::PeInfraAssistant < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')

      data_drop(curl_url('https://127.0.0.1:8145/status/v1/services?level=debug'),
                ent_directory,
                'pe_infra_assistant_service_status.json')
    end
  end

  # Scope which collects diagnostics related to the PE Infrastructure Assistant
  #
  # This scope gathers:
  #
  #   - Configuration files from /etc/puppetlabs/infra-assistant
  #   - Logs from /var/log/puppetlabs/infra-assistant
  #   - Service status from sysctl
  #   - Process status from /proc and /sys/fs/cgroup
  class Scope::Pe::InfraAssistant < Scope
    def setup(**options)
      confine { package_installed?('pe-infra-assistant') }
    end

    Scope::Pe.add_child(self, name: 'infra-assistant')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/etc/puppetlabs',
                            copy: ['infra-assistant/logback.xml',
                                   'infra-assistant/bootstrap.cfg',
                                   'request-logging.xml',
                                   'request-logging-api.xml',
                                   'tos-accepted.json',
                                   # NOTE: The PE Infra Assistant stores sensitive data under conf.d/
                                   #       Therefore, we explicitly list what to gather.
                                   'infra-assistant/conf.d/analytics.conf',
                                   'infra-assistant/conf.d/global.conf',
                                   'infra-assistant/conf.d/metrics.conf',
                                   'infra-assistant/conf.d/pe-infra-assistant-database.conf',
                                   'infra-assistant/conf.d/pe-infra-assistant.conf',
                                   'infra-assistant/conf.d/webserver.conf'],
                            to: 'enterprise/etc/puppetlabs',
                            max_age: -1},
                           {from:     "/etc/default",
                            copy:     ['pe-infra-assistant-service'],
                            to:       "system/etc/default",
                            max_age: -1},
                           {from:     "/etc/sysconfig",
                            copy:     ['pe-infra-assistant-service'],
                            to:       "system/etc/sysconfig",
                            max_age: -1}])

    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            copy: ['infra-assistant'],
                            to: 'logs'}],
                   units: [{'UNIT' => 'pe-infra-assistant.service', 'SYSLOG_IDENTIFIER' => 'infra-assistant'}])

    self.add_child(Check::PeInfraAssistant,
                   name: 'status',
                   services: ['pe-infra-assistant'])
  end

  # Check the status of the PE Postgres service
  #
  # This check gathers:
  #
  #   - A list of settings values that the server is using while running
  #   - A list of currently established database connections and the queries
  #     being executed
  #   - A distribution of Puppet run start times for thundering herd detection
  #   - The status of any configured replication slots
  #   - The status of any active replication connections
  #   - The extensions installed in each database
  #   - The size of database directories on disk
  #   - The size of databases as reported by postgres
  #   - The size of tables and indicies within databases
  #   - Statistics related to autovacuum activity and index performance
  class Check::PePostgresqlStatus < Check::ServiceStatus
    def run
      super

      ent_directory = File.join(state[:drop_directory], 'enterprise')
      res_directory = File.join(state[:drop_directory], 'resources')
      pg_datadir = if pe_version && Gem::Requirement.new('>= 2021.6.0').satisfied_by?(pe_version)
                     "#{PUP_PATHS[:server_data]}/postgresql/14/data"
                   else
                     "#{PUP_PATHS[:server_data]}/postgresql/11/data"
                   end

      data_drop(psql_settings,                ent_directory, 'postgres_settings.txt')
      data_drop(psql_stat_activity,           ent_directory, 'db_stat_activity.txt')
      data_drop(psql_locks,                   ent_directory, 'pg_locks.txt')
      data_drop(psql_stat_bgwriter,           ent_directory, 'pg_stat_bgwriter.txt')
      data_drop(psql_thundering_herd,         ent_directory, 'thundering_herd_query.txt')
      data_drop(psql_replication_slots,       ent_directory, 'postgres_replication_slots.txt')
      data_drop(psql_replication_status,      ent_directory, 'postgres_replication_status.txt')

      exec_drop("#{PUP_PATHS[:server_bin]}/pg_controldata -D #{pg_datadir}", ent_directory, 'pg_controldata.txt')
      exec_drop("ls -d #{PUP_PATHS[:server_data]}/postgresql/*/data #{PUP_PATHS[:server_data]}/postgresql/*/PG_* | xargs du -sh", res_directory, 'db_sizes_from_du.txt')

      psql_data = psql_database_sizes
      data_drop(psql_data, res_directory, 'db_sizes_from_psql.txt')

      databases = psql_databases
      databases = databases.lines.map(&:strip).grep(%r{^pe\-}).sort
      databases.each do |database|
        data_drop(psql_database_table_sizes(database),             res_directory, 'db_table_sizes.txt')
        data_drop(psql_database_relation_sizes_by_table(database), res_directory, 'db_relation_sizes_by_table.txt')
        data_drop(psql_extension_info(database),                   ent_directory, 'postgres_extension_info.txt')
        data_drop(psql_db_stats(database),                         ent_directory, 'postgres_db_stats.txt')
      end
    end

    # Execute psql queries.
    #
    # @param sql [String] SQL to execute via a psql command.
    # @param psql_options [String] list of options to pass to the psql command.
    #
    # @return (see #exec_return_result)

    def psql_return_result(sql, psql_options = '')
      command = %(su pe-postgres --shell /bin/bash --command "cd /tmp && #{PUP_PATHS[:server_bin]}/psql #{psql_options} --command \\"#{sql}\\"")
      exec_return_result(command)
    end

    def psql_databases
      sql = %Q(SELECT datname FROM pg_catalog.pg_database;)
      psql_options = '--tuples-only'
      psql_return_result(sql)
    end

    def psql_database_sizes
      sql = %Q(
        SELECT t1.datname AS db_name, pg_size_pretty(pg_database_size(t1.datname))
        FROM pg_database t1
        ORDER BY pg_database_size(t1.datname) DESC;
      )
      psql_return_result(sql)
    end

    def psql_extension_info(database)
      result = "#{database}\n\n"
      sql = %Q(SELECT * FROM pg_extension;)
      psql_options = "--dbname #{database}"
      result << psql_return_result(sql, psql_options)
    end

    # pg_table_size: Disk space used by the specified table, excluding indexes but including TOAST, and all of the forks: 'main', 'fsm', 'vm', 'init'.

    def psql_database_table_sizes(database)
      result = "#{database}\n\n"
      sql = %Q(
        SELECT '#{database}' AS db_name, nspname || '.' || relname AS relation, pg_size_pretty(pg_table_size(C.oid))
        FROM pg_class C LEFT JOIN pg_namespace N ON (N.oid = C.relnamespace)
        WHERE nspname NOT IN ('information_schema', 'pg_catalog', 'pg_toast')
        ORDER BY pg_table_size(C.oid) DESC;
      )
      psql_options = "--dbname #{database}"
      result << psql_return_result(sql, psql_options)
    end

    def psql_database_relation_sizes_by_table(database)
      result = "#{database}\n\n"
      sql = %Q(
        WITH
          tables
            AS (
              SELECT
                c.oid AS rel_oid, pg_total_relation_size(c.oid) AS total_size, *
              FROM
                pg_catalog.pg_class AS c
                LEFT JOIN pg_catalog.pg_namespace AS n ON
                    n.oid = c.relnamespace
              WHERE
                relkind = 'r'
                AND n.nspname NOT IN ('information_schema', 'pg_catalog')
            ),
          toast
            AS (
              SELECT
                c.oid AS rel_oid, *
              FROM
                pg_catalog.pg_class AS c
                LEFT JOIN pg_catalog.pg_namespace AS n ON
                    n.oid = c.relnamespace
              WHERE
                relkind = 't'
                AND n.nspname NOT IN ('information_schema', 'pg_catalog')
            ),
          indices
            AS (
              SELECT
                c.oid AS rel_oid, *
              FROM
                pg_catalog.pg_class AS c
                LEFT JOIN pg_catalog.pg_namespace AS n ON
                    n.oid = c.relnamespace
              WHERE
                relkind = 'i'
                AND n.nspname NOT IN ('information_schema', 'pg_catalog')
            )
        SELECT
          tab.rel_oid AS oid,
          '#{database}' || '.' || tab.nspname || '.' || tab.relname AS name,
          'table' AS type,
          pg_size_pretty(pg_relation_size(tab.rel_oid)) AS size,
          tab.total_size,
          tab.relfilenode AS file_oid
        FROM
          tables AS tab
        UNION
          SELECT
            t.rel_oid AS oid,
            '#{database}' || '.' || r.nspname || '.' || r.relname || '.' || t.relname AS name,
            'toast' AS type,
            pg_size_pretty(pg_relation_size(t.rel_oid)) AS size,
            r.total_size AS total_size,
            t.relfilenode AS file_oid
          FROM
            toast AS t
            INNER JOIN tables AS r ON t.rel_oid = r.reltoastrelid
        UNION
          SELECT
            i.rel_oid AS oid,
            '#{database}' || '.' || r.nspname || '.' || r.relname || '.' || i.relname AS name,
            'index' AS type,
            pg_size_pretty(pg_relation_size(i.rel_oid)) AS size,
            r.total_size AS total_size,
            i.relfilenode AS file_oid
          FROM
            indices AS i
            LEFT JOIN pg_catalog.pg_index AS c ON
                i.rel_oid = c.indexrelid
            INNER JOIN tables AS r ON c.indrelid = r.rel_oid
        ORDER BY
          total_size DESC,
          name ASC;
      )
      psql_options = "--dbname #{database}"
      result << psql_return_result(sql, psql_options)
    end

    def psql_db_stats(database)
      result = "#{database}\n\n"
      psql_options = "--dbname #{database}"

      # psql only returns the result of the last query, so these need
      # to be executed as separate commands.
      result << "pg_stat_database: https://www.postgresql.org/docs/14/monitoring-stats.html#PG-STAT-DATABASE-VIEW\n"
      result << psql_return_result(<<EOF, psql_options)
SELECT * FROM pg_stat_database WHERE datname = '#{database}';
EOF
      result << "\n\npg_stat_user_tables: https://www.postgresql.org/docs/14/monitoring-stats.html#PG-STAT-ALL-TABLES-VIEW\n"
      result << psql_return_result(<<EOF, psql_options)
SELECT * FROM pg_stat_user_tables;
EOF
      result << "\n\npg_stat_user_indexes: https://www.postgresql.org/docs/14/monitoring-stats.html#PG-STAT-ALL-INDEXES-VIEW\n"
      result << psql_return_result(<<EOF, psql_options)
SELECT * FROM pg_stat_user_indexes;
EOF
      result << "\n\npg_statio_user_tables: https://www.postgresql.org/docs/14/monitoring-stats.html#PG-STATIO-ALL-TABLES-VIEW\n"
      result << psql_return_result(<<EOF, psql_options)
SELECT * FROM pg_statio_user_tables;
EOF
      result << "\n\npg_statio_user_indexes: https://www.postgresql.org/docs/14/monitoring-stats.html#PG-STATIO-ALL-INDEXES-VIEW\n"
      result << psql_return_result(<<EOF, psql_options)
SELECT * FROM pg_statio_user_indexes;
EOF

      result
    end

    def psql_settings
      sql = %Q(SELECT * FROM pg_settings;)
      psql_options = '--tuples-only'
      psql_return_result(sql, psql_options)
    end

    def psql_stat_activity
      sql = %Q(SELECT * FROM pg_stat_activity ORDER BY xact_start ASC NULLS LAST;)
      psql_return_result(sql)
    end

    def psql_locks
      sql = %Q(SELECT * FROM pg_locks;)
      psql_return_result(sql)
    end

    def psql_stat_bgwriter
      sql = %Q(SELECT * FROM pg_stat_bgwriter;)
      psql_return_result(sql)
    end

    def psql_replication_slots
      sql = %Q(SELECT *, pg_current_wal_lsn() AS local_lsn, pg_wal_lsn_diff(pg_current_wal_lsn(), restart_lsn) AS replication_lag_bytes FROM pg_replication_slots;)
      psql_return_result(sql)
    end

    def psql_replication_status
      sql = %Q(SELECT * FROM pg_stat_replication;)
      psql_return_result(sql)
    end

    def psql_thundering_herd
      sql = %Q(
        SELECT date_part('month', start_time) AS month,
        date_part('day', start_time) AS day,
        date_part('hour', start_time) AS hour,
        date_part('minute', start_time) as minute, count(*)
        FROM reports
        WHERE start_time BETWEEN now() - interval '7 days' AND now()
        GROUP BY date_part('month', start_time), date_part('day', start_time), date_part('hour', start_time), date_part('minute', start_time)
        ORDER BY date_part('month', start_time) DESC, date_part('day', start_time) DESC, date_part( 'hour', start_time ) DESC, date_part('minute', start_time) DESC;
      )
      psql_options = '--dbname pe-puppetdb'
      psql_return_result(sql, psql_options)
    end
  end

  # Scope which collects diagnostics related to the PE Postgres service
  #
  # This scope gathers:
  #
  #   - Configuration files from /opt/puppetlabs/server/data/postgresql
  #     for pe-postgresql
  #   - Logs from /var/log/puppetlabs for pe-postgresql
  #   - Logs from /opt/puppetlabs/server/data/postgresql related to
  #     pe-postgresql upgrades
  class Scope::Pe::Postgres < Scope
    def setup(**options)
      # TODO: Should confine based on whether the pe-postgres package is
      #       installed. But, the package includes a version number and
      #       package_installed? does exact matches only.
      confine { File.executable?("#{PUP_PATHS[:server_bin]}/psql") }
    end

    Scope::Pe.add_child(self, name: 'postgres')

    self.add_child(Check::ConfigFiles,
                   name: 'config',
                   files: [{from: '/opt/puppetlabs/server/data/postgresql',
                            copy: ['*/data/{postgresql.conf,postmaster.opts,pg_ident.conf,pg_hba.conf}'],
                            to: 'enterprise/etc/puppetlabs/postgres',
                            max_age: -1}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [{from: '/var/log/puppetlabs',
                            # First glob gathers files from per-version
                            # subdirectories used by PE 2019 and newer.
                            # Second glob gathers log files for PE 2018.
                            copy: ['postgresql/*/', 'postgresql/*.log', 'pe_databases_cron/'],
                            to: 'logs'},
                           {from: '/opt/puppetlabs/server/data/postgresql',
                            copy: ['pg_upgrade_internal.log',
                                   'pg_upgrade_server.log',
                                   'pg_upgrade_utility.log'],
                            to: 'logs/postgresql'}],
                   units: [{'UNIT' => 'pe-postgresql.service', 'SYSLOG_IDENTIFIER' => 'pg_ctl'}])
    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [],
                   services: ['pe_databases*'])
    self.add_child(Check::PePostgresqlStatus,
                   name: 'status',
                   services: ['pe-postgresql'])
    self.add_child(Check::GatherFiles,
                   name: 'metrics',
                   files: [{from: '/opt/puppetlabs/puppet-metrics-collector',
                            copy: ['postgres/'],
                            to: 'metrics'}])
  end

  class Scope::Pe::Metrics < Scope
    def setup(**options)
      super(**options)
    end

    Scope::Pe.add_child(self, name: 'pe_metrics')

    self.add_child(Check::ServiceLogs,
                   name: 'logs',
                   files: [],
                   services: ['puppet_*metrics'])
  end

  # Runtime logic for executing diagnostics
  #
  # This class implements the runtime logic of the support script which
  # consists of:
  #
  #   - Setting up runtime state, such as the output directory.
  #   - Initializng and then executing a list of {Scope} objects.
  #   - Generating output archives and disposing of any runtime state.
  class Runner
    include Configable
    include DiagnosticHelpers

    def initialize(**options)
      initialize_configable

      @child_specs = []
    end

    # Add a child to be executed by this Runner instance
    #
    # @param klass [Class] the class from which the child should be
    #   initialized. Must implement a `run` method.
    # @param options [Hash] a hash of options to pass when initializing
    #   the child.
    # @return [void]
    def add_child(klass, **options)
      @child_specs.push([klass, options])
    end

    # Initialize runtime state
    #
    # This method loads required libraries, validates the runtime environment
    # and initializes output directories.
    #
    # @return [true] if setup completes successfully.
    # @return [false] if setup encounters an error.
    def setup
      Signal.trap("INT") do
        puts "Interrupted!"

        begin
          unless settings[:z_do_not_delete_drop_directory]
            # We can't call the logger from a trap, so print a message here
            puts "Removing working directory: #{@config.state[:drop_directory]}"
            cleanup_output_directory(log_messages: false)
          end
        rescue StandardError => e
          puts e.message
        end

        exit 1
      end

      ['puppet', 'facter', 'semantic_puppet/version'].each do |lib|
        begin
          require lib
        rescue ScriptError, StandardError => e
          log.error("%{exception_class} raised when loading %{library}: %{message}\n\t%{backtrace}" %
                    {exception_class: e.class,
                     library: lib,
                     message: e.message,
                     backtrace: e.backtrace.join("\n\t")})
          return false
        end
      end

      # FIXME: Replace with Scopes that are confined to Linux.
      unless /linux/i.match(Facter.value('kernel'))
        log.error('The support script is limited to Linux operating systems.')
        return false
      end

      # FIXME: Replace with Scopes that are confined to requiring root.
      unless Facter.value('identity')['privileged']
        log.error('The support script must be run with root privilages.')
        return false
      end

      begin
        Settings.instance.validate
      rescue => e
        log.error("%{exception_class} raised when validating settings: %{message}" %
          {exception_class: e.class,
           message: e.message})
        return false
      end

      if settings[:encrypt]
        gpg_command = executable?('gpg2') || executable?('gpg')

        if gpg_command.nil?
          log.error('Could not find gpg or gpg2 on the PATH. GPG must be installed to use the --encrypt option')
          return false
        else
          state[:gpg_command] = gpg_command
        end
      end

      if settings[:upload]
        sftp_command = executable?('sftp')

        if sftp_command.nil?
          log.error('Could not find sftp on the PATH. SFTP must be installed to use the --upload option')
          return false
        else
          state[:sftp_command] = sftp_command
        end
      end

      state[:start_time] = DateTime.now

      true
    end

    # Execute diagnostics
    #
    # This manages the setup of output directories and other state, followed
    # by the execution of all diagnostic classes created via {add_child}, and
    # finally creates archive formats and tears down runtime state.
    #
    # @return [0] if all operations were successful.
    # @return [1] if any operation failed.
    def run
      setup or return 1

      begin
        children = @child_specs.map do |(klass, opts)|
          opts ||= {}
          klass.new(**opts)
        end


        if settings[:list]
          children.each do |child|
            # Delete the 'base-status' check from the base scope so it will not be described
            # This check is always enabled, so it should not be listed as an option to disable
            if child.is_a?(Scope::Base)
              child.children.delete_if do |check|
                check.is_a?(Check::BaseStatus)
              end
              child.describe
            end
          end


          return state[:exit_code]
        end

        setup_output_directory or return 1
        setup_logfile

        children.each(&:run)

        cleanup_logfile
        output_file = create_output_archive(state[:drop_directory])
        output_file = encrypt_output_archive(output_file) if settings[:encrypt]

        if settings[:upload]
          sftp_upload(output_file)
        else
          display_summary(output_file)
        end
      rescue => e
        log.error("%{exception_class} raised when executing diagnostics: %{message}\n\t%{backtrace}" %
                  {exception_class: e.class,
                   message: e.message,
                   backtrace: e.backtrace.join("\n\t")})

        return 1
      ensure
        cleanup_output_directory
      end

      return state[:exit_code]
    end

    private

    def setup_output_directory
      # Already set up.
      return true if state.key?(:drop_directory)

      parent_dir = File.realdirpath(settings[:dir])
      timestamp = state[:start_time].strftime('%Y%m%d%H%M%S')
      short_hostname = Facter.value('networking')['hostname'].to_s.split('.').first
      dirname = ['puppet_enterprise_support', settings[:ticket].to_s, short_hostname, timestamp].reject(&:empty?).join('_')

      drop_directory = File.join(parent_dir, dirname)

      display('Creating output directory: %{drop_directory}' %
              {drop_directory: drop_directory})

      return false unless create_path(drop_directory, :mode => 0700)

      # Store drop directory in state to make it available to other methods.
      state[:drop_directory] = drop_directory
      true
    end

    def cleanup_output_directory(log_messages: false)
      if state.key?(:drop_directory) &&
         File.directory?(state[:drop_directory]) &&
         (! settings[:z_do_not_delete_drop_directory])
         if log_messages
           log.info('Cleaning up output directory: %{drop_directory}' %
                    {drop_directory: state[:drop_directory]})
         end

        FileUtils.remove_entry_secure(state[:drop_directory], force: true)
        state.delete(:drop_directory)
      end
    end

    # Create a logfile inside the drop directory
    #
    # A handle to the {::Logger}} created is stored in {Settings#state}
    # under the `:log_file` key.
    #
    # @return [void]
    def setup_logfile
      return if noop? || ! ( state.key?(:drop_directory) &&
                             File.directory?(state[:drop_directory]))

      log_path = File.join(state[:drop_directory], 'support_script_log.jsonl')
      log_file = File.open(log_path, 'w')
      logger = LogManager.file_logger(log_file)

      state[:log_file] = logger
      log.add_logger(logger)
    end

    def cleanup_logfile
      if state.key?(:log_file)
        log.remove_logger(state[:log_file])
        state[:log_file].close
        state.delete(:log_file)
      end
    end

    # Create a tarball from a directory of support script output
    #
    # @param output_directory [String] Path to the support script output
    #   directory.
    #
    # @return [String] Path to the compressed archive.
    def create_output_archive(output_directory)
      tar_change_directory = File.dirname(output_directory)
      tar_directory = File.basename(output_directory)
      output_archive = File.join(settings[:dir], tar_directory + '.tar.gz')

      display('Creating output archive: %{output_archive}' %
              {output_archive: output_archive})

      return output_archive if noop?

      old_umask = File.umask
      begin
        File.umask(0077)
        exec_or_fail(%(tar --create --file - --directory '#{tar_change_directory}' '#{tar_directory}' | gzip --force -9 > '#{output_archive}'), 600)
      ensure
        File.umask(old_umask)
      end

      output_archive
    end

    def encrypt_output_archive(output_archive)
      encrypted_archive = output_archive + '.gpg'
      gpg_homedir = File.join(state[:drop_directory], 'gpg')

      display('Encrypting output archive file: %{output_archive}' %
              {output_archive: output_archive})

      return encrypted_archive if noop?

      FileUtils.mkdir(gpg_homedir, :mode => 0700)
      exec_or_fail(%(echo '#{PGP_KEY}' | '#{state[:gpg_command]}' --quiet --import --homedir '#{gpg_homedir}'))
      exec_or_fail(%(#{state[:gpg_command]} --quiet --homedir "#{gpg_homedir}" --trust-model always --recipient #{PGP_RECIPIENT} --encrypt "#{output_archive}"))
      FileUtils.safe_unlink(output_archive)

      encrypted_archive
    end

    def sftp_upload(output_archive)
      display('Uploading: %{output_archive} via SFTP' %
              {output_archive: output_archive})

      return if noop?

      if settings[:upload_disable_host_key_check]
        ssh_known_hosts = '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null'
      else
        ssh_known_hosts_file = Tempfile.new('csp.ky')
        ssh_known_hosts_file.write(SFTP_KNOWN_HOSTS)
        ssh_known_hosts_file.close
        ssh_known_hosts = "-o StrictHostKeyChecking=yes -o UserKnownHostsFile=#{ssh_known_hosts_file.path}"
      end
      if settings[:upload_user]
        sftp_url = "#{settings[:upload_user]}@#{SFTP_HOST}:/"
      else
        sftp_url = "#{SFTP_USER}@#{SFTP_HOST}:/drop/"
      end
      if settings[:upload_key]
        ssh_key_file = File.absolute_path(settings[:upload_key])
        ssh_identity = "IdentityFile=#{ssh_key_file}"
      else
        ssh_key_file = Tempfile.new('pes.ky')
        ssh_key_file.write(SFTP_KEY)
        ssh_key_file.close
        ssh_identity = "IdentityFile=#{ssh_key_file.path}"
      end
      # https://stribika.github.io/2015/01/04/secure-secure-shell.html
      # ssh_ciphers        = 'Ciphers=chacha20-poly1305@openssh.com,aes256-gcm@openssh.com,aes128-gcm@openssh.com,aes256-ctr,aes192-ctr,aes128-ctr'
      # ssh_kex_algorithms = 'KexAlgorithms=curve25519-sha256@libssh.org,diffie-hellman-group-exchange-sha256'
      # ssh_macs         = 'MACs=hmac-sha2-512-etm@openssh.com,hmac-sha2-256-etm@openssh.com,umac-128-etm@openssh.com,hmac-sha2-512,hmac-sha2-256,umac-128@openssh.com'
      ssh_ciphers        = 'Ciphers=aes256-ctr,aes192-ctr,aes128-ctr'
      ssh_kex_algorithms = 'KexAlgorithms=diffie-hellman-group-exchange-sha256'
      ssh_macs           = 'MACs=hmac-sha2-512,hmac-sha2-256'
      ssh_options        = %(-o "#{ssh_ciphers}" -o "#{ssh_kex_algorithms}" -o "#{ssh_macs}" -o "Protocol=2" -o "ConnectTimeout=16"  #{ssh_known_hosts} -o "#{ssh_identity}" -o "BatchMode=yes")
      sftp_command       = %(echo 'put #{output_archive}' | sftp #{ssh_options} #{sftp_url} 2>&1)

      begin
        sftp_result = Exec.exec_cmd(sftp_command)
        if sftp_result.status.zero? && sftp_result.error.nil?
          display('File uploaded to: %{sftp_host}' %
                  {sftp_host: SFTP_HOST})
          File.delete(output_archive)
        else
          ssh_key_file.unlink unless settings[:upload_key]
          ssh_known_hosts_file.unlink unless settings[:upload_disable_host_key_check]
          # FIXME: Make i18n friendly
          display ' ** Unable to upload the output archive file. SFTP Output:'
          display
          display sftp_result.stdout
          display
          display '    Please manualy upload the output archive file to Puppet Support.'
          display
          display "    Output archive file: #{output_archive}"
          display
        end
      rescue StandardError => e
        ssh_key_file.unlink unless settings[:upload_key]
        ssh_known_hosts_file.unlink unless settings[:upload_disable_host_key_check]
        # FIXME: Make i18n friendly
        display ' ** Unable to upload the output archive file: SFTP command error:'
        display
        display e
        display
        display '    Please manualy upload the output archive file to Puppet Support.'
        display
        display "    Output archive file: #{output_archive}"
        display
      end
    end

    def display_summary(output_archive)
      # FIXME: Make i18n friendly
      display 'Puppet Enterprise customers ...'
      display
      display '  We recommend that you examine the collected data before forwarding to Puppet,'
      display '  as it may contain sensitive information that you may wish to redact.'
      display
      display '  An overview of the data collected by this tool can be found at:'
      display "  #{DOC_URL}"
      display
      display '  Please upload the output archive file to Puppet Support.'
      display
      display "  Output archive file: #{output_archive}"
      display
    end
  end
end
end
end


# The following allows this class to be executed as a standalone script.

if File.expand_path(__FILE__) == File.expand_path($PROGRAM_NAME)
  require 'optparse'

  # See also: lib/puppet/face/enterprise/support.rb
  default_dir     = File.directory?('/var/tmp') ? '/var/tmp' : '/tmp'
  default_log_age = 7

  puts 'Puppet Enterprise Support Script v' + PuppetX::Puppetlabs::SupportScript::VERSION
  puts

  options = {}
  parser = OptionParser.new do |opts|
    opts.banner = "Usage: #{File.basename(__FILE__)} [options]"
    opts.separator ''
    opts.separator 'Summary: Collects Puppet Enterprise Support Diagnostics'
    opts.separator ''
    opts.separator 'Options:'
    opts.separator ''
    opts.on('-d', '--dir DIRECTORY', "Output directory. Defaults to: #{default_dir}") do |dir|
      options[:dir] = dir
    end
    opts.on('-e', '--encrypt', 'Encrypt output using GPG') do
      options[:encrypt] = true
    end
    opts.on('-l', '--log_age DAYS', "Log age (in days) to collect. Defaults to: #{default_log_age}") do |log_age|
      options[:log_age] = log_age
    end
    opts.on('-n', '--noop', 'Enable noop mode') do
      options[:noop] = true
    end
    opts.on('--enable LIST', Array, 'Comma-delimited list of scopes or checks to enable') do |list|
      options[:enable] ||= []
      options[:enable] += list
    end
    opts.on('--disable LIST', Array, 'Comma-delimited list of scopes or checks to disable') do |list|
      options[:disable] ||= []
      options[:disable] += list
    end
    opts.on('--only LIST', Array, 'Comma-delimited list of of scopes or checks to run, disabling all others') do |list|
      options[:only] ||= []
      options[:only] += list
    end
    opts.on('--list', 'List available scopes and checks that can be passed to --enable, --disable, or --only.') do |arg|
      options[:list] = true
    end
    opts.on('-t', '--ticket NUMBER', 'Support ticket number') do |ticket|
      options[:ticket] = ticket
    end
    opts.on('-u', '--upload', 'Upload to Puppet Support via SFTP. Requires the --ticket parameter') do
      options[:upload] = true
    end
    opts.on('--upload_disable_host_key_check', 'Disable SFTP Host Key Check. Requires the --upload parameter') do
      options[:upload_disable_host_key_check] = true
    end
    opts.on('--upload_key FILE', 'Key for SFTP. Requires the --upload parameter') do |upload_key|
      options[:upload_key] = upload_key
    end
    opts.on('--upload_user USER', 'User for SFTP. Requires the --upload parameter') do |upload_user|
      options[:upload_user] = upload_user
    end
    opts.on('-z', 'Do not delete output directory after archiving') do
      options[:z_do_not_delete_drop_directory] = true
    end
    opts.on('-h', '--help', 'Display help') do
      puts opts
      puts
      exit 0
    end
  end
  parser.parse!

  PuppetX::Puppetlabs::SupportScript::Settings.instance.configure(**options)
  PuppetX::Puppetlabs::SupportScript::Settings.instance.log.add_logger(PuppetX::Puppetlabs::SupportScript::LogManager.console_logger)
  support = PuppetX::Puppetlabs::SupportScript::Runner.new
  support.add_child(PuppetX::Puppetlabs::SupportScript::Scope::Base, name: '')

  exit support.run
end
