# frozen_string_literal: true

# Used for $ERROR_INFO. This *must* be capitalized!
require 'English'
require 'json'
require 'set'
require 'bolt/executor'
require 'bolt/result'
require 'bolt/config'
require 'bolt/result_set'
require 'bolt/apply_result'
require 'bolt/puppetdb'

SENSITIVE_KEY = "_sensitive"

module PlanRunner
  module BoltOverrides
    class Executor < Bolt::Executor
      APPLY_REPORT_KEYS = %w[metrics resource_statuses status].freeze

      attr_reader :noop, :logger, :in_parallel

      def initialize(http_client, plan_id, environment, encryption_key, noop = nil)
        @plan_logging = false
        @noop = noop
        # Set this to false here (see https://github.com/puppetlabs/bolt/commit/4aa849c5f83f37202bf5a27cf5f04df6d729d78d)
        # By disabling the `parralelize` boltlib function this should never be able to be set to `true`.
        @in_parallel = false
        @nested_plans = 0
        @http_client = http_client
        @plan_id = plan_id
        @environment = environment
        @fiber_executor = Bolt::FiberExecutor.new
        @encryption_key = encryption_key

        super()

        # These are set by Bolt's Executor but unneeded by our implementation and hold references to Concurrent objects.
        @pool = nil
        @transports = nil
        @publisher = nil

        @logger = Logging.logger[self]
        @logger.debug { 'Initialized PE executor for bolt' }
      end

      # Bolt's version of this shutsdown the publisher executor which we have already orphaned
      def shutdown; end

      def decrypt_output(output_data)
        JSON.parse(
          @encryption_key.decrypt(output_data)
        )['output']
      end

      def parse_api_response_as_bolt_result(response_body, targets_by_name, decrypt_sensitive_data)
        response_body['results'].map do |resp|
          if decrypt_sensitive_data && resp['value'].key?(SENSITIVE_KEY)
            resp['value'][SENSITIVE_KEY] = Puppet::Pops::Types::PSensitiveType::Sensitive.new(
              decrypt_output(resp['encrypted_value'])
            )
          end
          Bolt::Result.new(
            targets_by_name[resp['target']],
            value: resp['value'],
            action: resp['action'],
            object: resp['object']
          )
        end
      end

      def parse_failed_resources(report)
        report['resource_statuses'].select { |_, r| r['failed'] }.flat_map do |key, resource|
          resource['events'].select { |e| e['status'] == 'failure' }.map do |event|
            "\n  #{key}: #{event['message']}"
          end
        end.join
      end

      def parse_api_response_as_apply_result(response_body, targets_by_name)
        response_body['results'].map do |resp|
          report = resp['value']
          target = targets_by_name[resp['target']]
          missing_keys = APPLY_REPORT_KEYS.reject { |k| report.include?(k) }
          options = if report['_error']
                      { error: report['_error'], report: report }
                    elsif missing_keys.any?
                      # We have detected that the report does not have required shape. Raise an error
                      {
                        error: {
                          'msg' => "Report did not contain all expected keys. Missing: #{missing_keys.join(', ')}",
                          'kind' => 'bolt/invalid-report'
                        },
                        report: report.reject { |k| %w[_error _output].include?(k) }
                      }
                    elsif report['status'] =~ /fail/
                      # A resource has failed, generate an error showing which one(s)
                      {
                        error: {
                          'msg' => "Resources failed to apply for #{resp['target']}#{parse_failed_resources(report)}",
                          'kind' => 'bolt/resource-failure'
                        },
                        report: report.reject { |k| k == '_error' }
                      }
                    else
                      { report: report }
                    end
          Bolt::ApplyResult.new(target, **options)
        end
      end

      def parse_api_response_as_result_set(api_response, targets, decrypt_sensitive_data: false, apply_result: false)
        unless api_response['results']
          msg = "Response from results API is empty!"
          @logger.warn(msg)
          raise Bolt::Error.new(msg, 'bolt.plan-executor/unexpected-exception')
        end
        targets_by_name = targets.map(&:host).zip(targets).to_h
        all_responses = if apply_result
                          parse_api_response_as_apply_result(api_response, targets_by_name)
                        else
                          parse_api_response_as_bolt_result(api_response, targets_by_name, decrypt_sensitive_data)
                        end
        Bolt::ResultSet.new(all_responses)
      rescue StandardError => e
        msg = "Unexpected exception thrown while parsing results\n#{e.message}\n#{e.backtrace.join("\n")}\n"
        @logger.warn(msg)
        raise Bolt::Error.new(msg, 'bolt.plan-executor/unexpected-exception')
      end

      def fetch_result_from_api(results_url)
        result = nil
        loop do
          @logger.debug("Fetching results from API #{results_url}")
          result, http_code = @http_client.get_with_orch_cert(results_url)
          case http_code
          when 200
            break
          when 204
            sleep 2
            next
          else
            msg = "unknown API response with code #{http_code}: #{result}"
            @logger.warn(msg)
            raise Bolt::Error.new(msg, 'bolt.plan-executor/unexpected-response')
          end
        end
        result
      end

      # Run a POST to Orchestrator, raise on anything that's not HTTP 200, then
      # parse the response body as a Bolt::Result
      def run_post(url, body)
        post_response, http_code = @http_client.post_with_orch_cert(url, body)
        if http_code != 200
          msg = "Failed API response with code #{http_code}: #{post_response}"
          @logger.warn(msg)
          raise Bolt::Error.new(msg, 'bolt.plan-executor/errored-response')
        end
        fetch_result_from_api(post_response['tracking']['url'])
      end

      # Unwraps any Sensitive data in an arguments Hash, so the plain-text is passed
      # to the Task/Script.
      #
      # This works on deeply nested data structures composed of Hashes, Arrays, and
      # and plain-old data types (int, string, etc).
      def unwrap_sensitive_args(arguments)
        case arguments
        when Array
          # iterate over the array, unwrapping all elements
          arguments.map { |x| unwrap_sensitive_args(x) }
        when Hash
          # iterate over the arguments hash and unwrap all keys and values
          arguments.each_with_object({}) do |(k, v), h|
            h[unwrap_sensitive_args(k)] = unwrap_sensitive_args(v)
          end
        when Puppet::Pops::Types::PSensitiveType::Sensitive
          # this value is Sensitive, unwrap it
          unwrap_sensitive_args(arguments.unwrap)
        else
          # unknown data type, just return it
          arguments
        end
      end

      # In Bolt::PAL this is an accessible attribute. PE plans do not support setting _run_as.
      # Implement this as a method that raises a helpful error message.
      def run_as
        msg = "The '_run_as' option is not supported for the run_plan action."
        raise Bolt::Error.new(msg, 'bolt.plan-executor/not-implemented')
      end

      def forbid_pe_incompatible_action_opts(opts, action)
        forbidden_opts = Set.new([:run_as])
        opts_intersection = forbidden_opts.intersection(Set.new(opts.keys))
        unless opts_intersection.empty?
          plural = opts_intersection.count > 1 ? 'options are' : 'option is'
          opt_keys_to_string = opts_intersection.map { |opt| "_#{opt}" }.join(', ')
          msg = "The '#{opt_keys_to_string}' #{plural} not supported for the #{action} action."
          raise Bolt::Error.new(msg, 'bolt.plan-executor/not-implemented')
        end
      end

      # Event should be a Hash with at least a `:type` key whos value is a symbol.
      # This signature is used by internal Bolt functions (eg out::message).
      def publish_event(event)
        type = event.delete(:type)
        type = type == :message ? 'out_message' : type.to_s

        url = '/v2/private/plans/' + @plan_id + '/event/' + type
        @http_client.post_with_orch_cert(url, { details: event })
      end

      # The `log_plan` method is where the events are published to Orchestrator.
      # Notice the call to `publish_event` near the bottom of the method.
      def log_plan(plan_name)
        @logger.info("Starting: plan #{plan_name}")
        start_time = Time.now
        # We only want nested plans to publish events, not the top-level plan.
        if @nested_plans > 0
          publish_event(type: 'plan_start', event: { plan: plan_name })
        end
        @nested_plans += 1
        results = nil
        begin
          results = yield
        ensure
          duration = Time.now - start_time
          @logger.info("Finished: plan #{plan_name} in #{duration.round(2)} sec")
        end
        @nested_plans -= 1
        if @nested_plans > 0
          publish_event(type: 'plan_end', event: { plan: plan_name, duration: duration })
        end
        results
      end

      # Reporting functions that Bolt uses to collect analytics. For now just make them do nothing
      # in PE
      def report_bundled_content(mode, name); end

      def report_file_source(_plan_function, _source); end

      def report_function_call(function); end

      def report_noop_mode(_mode); end

      def report_yaml_plan(plan); end

      def deprecation(msg); end

      # ---------------------------------------------------------------------------------
      #
      # The following defines the actual executor functions called when a plan includes
      # a function call of the same name.
      #
      # ---------------------------------------------------------------------------------

      def action(type, options, default_description)
        @logger.debug("Executing #{type} through Orchestrator")
        forbid_pe_incompatible_action_opts(options, type)
        description = options.fetch(:description, default_description)
        yield description
      rescue StandardError => e
        if e.is_a?(Bolt::Error)
          # It's already formatted and logged, just raise it
          raise e
        end
        msg = "Unexpected exception thrown during #{type}\n#{e.message}\n#{e.backtrace.join("\n")}\n"
        @logger.warn(msg)
        raise Bolt::Error.new(msg, 'bolt.plan-executor/unexpected-exception')
      end

      def run_task_with(_target_mapping, _task, _options = {}, _position = [])
        msg = "The 'run_task_with' feature is not allowed in PE"
        raise Bolt::Error.new(msg, 'bolt.plan-executor/not-implemented')
      end

      def download_file(_targets, _source, _destination, _options = {}, _position = [])
        msg = "The 'download_file' feature is not allowed in PE"
        raise Bolt::Error.new(msg, 'bolt.plan-executor/not-implemented')
      end

      def run_plan(scope, plan, params)
        plan.call_by_name_with_scope(scope, params, true)
      end

      def run_task(targets, task, arguments, options = {}, _position = [])
        action("run_task", options, "Run task #{task.name}") do |description|
          unwrapped_args = unwrap_sensitive_args(arguments)
          # The run_task puppet function shipped with bolt adds _noop to task parameter if set:
          # In addition to that it adds :noop to the options argument. Orchestrator handles noop
          # (including adding it as a special task parameter) so remove it from params here. Note
          # that this will remove any "special" parameter that starts with `_` even though the
          # puppet function will have presumably only added back _noop
          unwrapped_args.delete_if { |k, _v| k.start_with?('_') }
          body = {
            description: description,
            task: task.name,
            noop: options[:noop] || false,
            environment: @environment,
            params: unwrapped_args,
            targets: targets
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/run_task'
          @logger.debug("POST to API to start task #{task.name}")
          parse_api_response_as_result_set(
            run_post(url, body),
            targets,
            decrypt_sensitive_data: true
          )
        end
      end

      def run_command(targets, command, options = {}, _position = [])
        action('run_command', options, "Run command '#{command}'") do |description|
          body = {
            description: description,
            command: command,
            environment: @environment,
            targets: targets
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/run_command'
          @logger.debug("POST to API to start run_command")
          parse_api_response_as_result_set(
            run_post(url, body),
            targets
          )
        end
      end

      def run_script(targets, script, arguments, options = {}, _position = [])
        action('run_script', options, "Run script #{script}") do |description|
          unwrapped_args = unwrap_sensitive_args(arguments)
          body = {
            description: description,
            script: script,
            arguments: unwrapped_args,
            environment: @environment,
            targets: targets
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/run_script'
          @logger.debug("POST to API to start run_script")
          parse_api_response_as_result_set(
            run_post(url, body),
            targets
          )
        end
      end

      def upload_file(targets, source, destination, options = {}, _position = [])
        action('upload_file', options, "Upload file #{source}") do |description|
          body = {
            source: source,
            destination: destination,
            description: description,
            environment: @environment,
            targets: targets
          }
          url = "/v2/private/plans/#{@plan_id}/executor/upload_file"
          @logger.debug("POST to API to start upload_file")
          parse_api_response_as_result_set(
            run_post(url, body),
            targets
          )
        end
      end

      def apply_prep(targets, options = {}, _position = [])
        action('apply_prep', options, 'apply_prep') do |description|
          body = {
            description: description,
            environment: @environment,
            targets: targets
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/apply_prep'
          @logger.debug("POST to API to start apply_prep")
          result = run_post(url, body)
          result['results'].map! do |target_result|
            # Orchestrator returns the fact set in the encrypted data. The plan function
            # expects the facts as the result of this function call, so just replace
            # 'value' with the full decrypted fact set.
            #
            # If there is no encrypted value we assume there was an error, and do nothing
            if target_result['encrypted_value']
              target_result['value'] = decrypt_output(target_result['encrypted_value'])
            end
            target_result.delete('encrypted_value')
            target_result
          end
          parse_api_response_as_result_set(
            result,
            targets
          )
        end
      end

      def apply_puppet_code(targets, compile_context, options = {}, _position = [])
        action('apply', options, 'Apply puppet code') do |description|
          body = {
            description: description,
            environment: @environment,
            'compile-context': compile_context,
            'target-specs': targets.map(&:detail),
            apply_options: {}
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/apply'
          @logger.debug("POST to API to start apply")
          result = run_post(url, body)
          result['results'].map! do |target_result|
            # Orchestrator returns the full report in the encrypted data. The plan function
            # expects the full report as the result of this function call, so just replace
            # 'value' with the full decrypted report.
            #
            # If there is no encrypted value we assume there was an error, and do nothing
            if target_result['encrypted_value']
              target_result['value'] = decrypt_output(target_result['encrypted_value'])
            end
            target_result.delete('encrypted_value')
            target_result
          end
          parse_api_response_as_result_set(
            result,
            targets,
            apply_result: true
          )
        end
      end

      def without_default_logging
        # In PE we honor the @plan_logging variable for several functions and
        # this will ensure that further code is executed with @plan_runner = false.
        # Because we currently set @plan_runner to false and then never reference
        # it again in this class, this function is currently a noop.
        yield
      end

      def wait_until_available(targets, options = {})
        action('wait_until_available', options, 'wait_until_available') do |description|
          wait_time = options.fetch(:wait_time, 120)
          retry_interval = options.fetch(:retry_interval, 1)
          body = {
            description: description,
            targets: targets,
            environment: @environment,
            retry_interval: retry_interval,
            wait_time: wait_time
          }
          url = '/v2/private/plans/' + @plan_id + '/executor/wait_until_available'
          @logger.debug("POST to API to start wait_until_available")
          parse_api_response_as_result_set(
            run_post(url, body),
            targets
          )
        end
      end
    end
  end
end
