#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../lib/enterprise_tasks/node_groups'
require_relative '../files/enterprise_task_helper.rb'

class CheckPpAuthRoleRule < EnterpriseTaskHelper
  include EnterpriseTasks::NodeGroups

  def _pinned_nodes?
    PuppetX::Util::Classifier.nodes_pinned_to_group(all_groups, 'PE Master').size.positive?
  end

  def _rule_exist?(group)
    PuppetX::Util::Classification.rule_exists?(group['rule'], ['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler'])
  end

  def _we_have_rules?(rules)
    !rules.empty? && ((_pinned_nodes? && rules[1].any? { |i| ['or', 'and'].include? i }) || !_pinned_nodes?)
  end

  def task(**_kwargs)
    initialize_pe_modules

    group = get_group('PE Master')

    # If rule exists we have nothing to do
    if _rule_exist?(group)
      {
        success: true,
      }
    # If no rule object exists for the group, just add the rule
    elsif !group.key?('rule')
      group['rule'] = ['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler']
      update_group({
                     id: group['id'],
                     rule: group['rule'],
                   })
      {
        success: true,
      }
    # 'And' rule exists, this means we have to error out
    elsif (group['rule'][0] == 'and') | (group['rule'][1][0] == 'and')
      raise EnterpriseTaskHelper::Error.new(
        'It appears the PE Master node group rules are set to match "all rules". In order to be able to append a new rule to set up PE Compilers for the first time, this needs to be set to "any rule". Please modify your rules and try this command again.',
        'puppetlabs.check-pp-auth-role-rule/incompatible_node_group_rules',
        {
          rule: group['rule'],
        },
      )
    else
      if _pinned_nodes?
        if _we_have_rules?(group['rule'])
          group['rule'][1].push(['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler'])
        else
          group['rule'].insert(1, ['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler'])
        end
      else
        group['rule'].push(['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler'])
      end
      update_group({
                     id: group['id'],
                     rule: group['rule'],
                   })
      {
        success: true,
      }
    end
  end
end

CheckPpAuthRoleRule.run if __FILE__ == $PROGRAM_NAME
