require_relative '../../../shared/config_translate'

# Parse inputs to the code_management class into data usable by the
# 'inner' classes like code_manager and pe_r10k
Puppet::Functions.create_function(:'puppet_enterprise::parse_code_mgmt_git_config') do
  dispatch :parse do
    param 'Optional[String]', :git_provider
    param 'Optional[String]', :git_private_key
    param 'Optional[String]', :git_default_ref
    param 'Optional[String]', :git_oauth_token
    param 'Optional[String]', :git_proxy
    param 'Optional[Array[Hash]]', :git_repositories
  end

  def parse(git_provider, git_private_key, git_default_ref, git_oauth_token, git_proxy, git_repositories)
    user_values = {
      'private-key'  => git_private_key,
      'default-ref'  => git_default_ref,
      'oauth-token'  => git_oauth_token,
      'proxy'        => git_proxy,
      'repositories' => git_repositories
    }.select { |_, value| value && !value.empty? }

    # Code manager, both the class in this module as well as the application itself,
    # treat these config values specially so we need to not manage them with the
    # other git settings but they should be included in the git block of an r10k config
    r10k_specific_user_values = {
      'provider' => git_provider || 'rugged'
    }.select { |_, value| value && !value.empty? }

    cm_config = nil
    r10k_config = nil

    # Set r10k specific values independently in case there are no other user values set
    if not r10k_specific_user_values.empty?
      # R10K needs all keys to use underscores, not dashes
      r10k_config = ConfigTranslate.key_dashes_to_underscores(r10k_specific_user_values)
    end

    # Set any config set by the user, including merging any r10k specific config to the r10k output
    if not user_values.empty?
      cm_config = user_values
      # R10K needs all keys to use underscores, not dashes
      r10k_config = ConfigTranslate.key_dashes_to_underscores(r10k_specific_user_values.merge(user_values))
    end

    [cm_config, r10k_config]
  end
end
