# coding: utf-8
require 'puppet/indirector/face'
require 'puppet_x/util/service_status'
require 'json'
require 'uri'

Puppet::Face.define(:infrastructure, '1.0.0') do
  ### Puppet face actions

  action :status do
    summary _('Displays status issues and service alerts for Puppet services.')
    description <<-EOT
      Check the status of all PE services
    EOT

    option('--format json') do
      summary _("Outputs in JSON format")
      description <<-EOT
        Outputs in JSON format.
      EOT
    end

    option("--host HOSTNAME") do
      summary _("Displays status and alerts for all services on a specified host")
      description <<-EOT
        Displays status and alerts for all services on a specified host (ex: Primary, Replica).
      EOT
    end

    option("--service SERVICENAME") do
      summary _("Displays status and alerts for a specified service")
      description <<-EOT
        Displays status and alerts for a specified service.
        Valid services include:
          activity
          classifier
          code-manager
          file-sync-client
          file-sync-storage
          master
          orchestrator
          pcp-broker
          pe-host-action-collector
          postgresql
          puppetdb
          rbac
      EOT
    end

    option("--timeout TIMEOUT_SECONDS") do
      summary _("Per-node status timeout, in seconds (default 5)")
      default_to { 5 }
      description <<-EOT
        Specify the how long to wait for each node to gather status information. (default 5 seconds)
      EOT
    end

    when_invoked do |options|
      host = options[:host]
      service = options[:service]
      timeout_seconds = options[:timeout]
      services_config = PuppetX::Util::ServiceStatus.load_services_config()
      nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config()

      Puppet.notice _("Contacting services for status information...") unless options[:format]

      PuppetX::Util::ServiceStatus.services_for(host, service, services_config, timeout_seconds, nodes_config: nodes_config)
    end

    when_rendering :console do |statuses, options|
      nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config()
      output = ''

      if options[:format] == 'json'
        output = JSON.pretty_generate(statuses)
      else

        # Add a header server display name and general node order to each
        # status, or a catch all if nodes info is missing.
        statuses.each do |status|
          node = nodes_config.find { |n| n[:certname] == status[:server] }
          if !node.nil?
            status[:server_display_name] = node[:display_name]
            status[:order] = node[:order]
          else
            status[:server_display_name] = 'Infrastructure Host'
            status[:order] = 99
          end
        end

        # Sort statuses by the order from service.conf nodes, the server name,
        # then the service display name to both order them by node role ('master',
        # 'replica', etc.) and group them by server.
        current_host = ''
        statuses.sort_by { |h| [h[:order], h[:server], h[:display_name]] }.each do |status|
          if current_host != status[:server]
            output << PuppetX::Util::ServiceStatus.format_server_status_header(status[:server_display_name], status[:server])
            current_host = status[:server]
          end
          output << PuppetX::Util::ServiceStatus.format_status(status)
        end
        output << PuppetX::Util::ServiceStatus.format_server_status_footer unless current_host == ''

        error_statuses_count = statuses.count { |status_hash|
          PuppetX::Util::ServiceStatus.service_has_error_state(status_hash) ||
            PuppetX::Util::ServiceStatus.service_has_important_alert(status_hash)
        }
        output << PuppetX::Util::ServiceStatus.format_summary_message(statuses.count, error_statuses_count)
      end

      output
    end
  end
end
