module PuppetX
  module Util
    class Orchestrator

      # Methods for retrieving and processing /orchestrator/v1/job/#/report
      # specifically for logging and displaying the results of Puppet runs.
      module Report

        # Used by save_report_logs to fetch the job report from the
        # orchestration service. A valid response will be JSON-parsable, have a
        # 'report' key with an array of reports, and the 'events' key in this
        # report should either be nil (nothing changed during the puppet run),
        # or a non-empty array of events that happened during the run.
        #
        # Sometimes you can get malformed objects back if services on either the
        # remote or primary node have a hiccup, like getting an empty events array
        # or a bad response code. This function retries fetching the report for
        # up to 10 seconds when this happens, in case the error can correct
        # itself.
        #
        # PE-28018: Figure out which of these error symptoms are not actually
        # correctable and don't retry in those cases.
        #
        # @return [Array{Hash}] array of puppet report hashes, or an empty array.
        def get_job_reports(job_url)
          url = "#{job_url}/report"
          result = nil

          # It can take a second for the report to actually show up once
          # the job finishes.
          retry_time = 10
          start_time = Time.now
          while Time.now - start_time < retry_time
            uri = URI(url)
            response = orch_connection.get(uri, headers: orch_request_headers)
            begin
              check_orch_response(response, 200)
              reports = JSON.parse(response.body)['report']

              reports.each do |r|
                # If the events array is empty, something bad probably happened.
                # If no changes were made, events should be nil, not an empty
                # array.
                if r['events']
                  if r['events'].empty?
                    Puppet.warning(_("Report returned from orchestration service for '%{certname}' contained an empty events array") % { certname: r['certname'] })
                    next
                  end
                  r['events'] = r['events'].sort_by { |e| Time.parse(e['timestamp']) }
                end
              end
              result = reports
              break
            rescue PuppetX::Util::OrchestratorResponseError => e
              Puppet.debug(_('Bad response from orchestration service when fetching report'))
              Puppet.debug(_("Status code: %{code}" % { code: e.status_code }))
              Puppet.debug(_("Response body: %{body}" % { body: e.body }))
            rescue PuppetX::Util::OrchestratorEmptyResponseError
              Puppet.debug(_('Empty body in response from orchestration service when fetching report'))
            rescue StandardError => e
              Puppet.warning(_("Error parsing response from orchestrator with the following message: %{msg}") % { msg: e.message })
              break
            end
            Puppet.debug(_("Retrying fetching of job report at %{url}" % { url: url }))
            sleep(orch_status_wait_seconds)
          end
          Puppet.warning(_('Could not fetch a valid response from orchestration service')) if result.nil?
          result || []
        end

        # Expects a report as returned by get_job_reports, or a Puppet::Transaction::Report object
        # This will extract all failed event types from the report and return them in an array.
        def find_errors(report)
          errors = []
          begin
            if report.is_a?(Puppet::Transaction::Report)
              report.resource_statuses.each do |res, status|
                if status.failed || status.failed_to_restart
                  message = ''
                  status.events.each do |thisevent|
                    message = message + thisevent.message + "\n" if thisevent.status =~ /fail/
                  end
                  event = {
                    'file'           => status.file,
                    'line'           => status.line,
                    'resource_type'  => status.resource_type,
                    'resource_title' => status.title,
                    'message'        => message,
                  }
                  errors << event
                end
              end
            else
              if report['events']
                report['events'].each do |event|
                  errors << event if event['status'] && event['status'] =~ /failure/
                end
              end
            end
          rescue StandardError => e
            Puppet.debug(_("Error parsing report to find errors with the following message: %{msg}") % { msg: e.message })
          end
          errors
        end

        # Expects a report as returned by get_job_reports, or a Puppet::Transaction::Report object
        # Calls find_errors from within this function for convenience of not needing to call
        # both functions when you want to print errors.
        def print_errors(report)
          if report.is_a?(Puppet::Transaction::Report)
            return if report.status != 'failed'
          else
            return if report.empty? || (report['events'] && report['events'].empty?)
          end
          errors = find_errors(report)
          return if errors.empty?
          Puppet.err(_('*** Run Errors ***'))
          errors.each do |error|
            Puppet.err(<<~EOS)

              File:      #{error['file']}
              Line:      #{error['line']}
              Class:     #{error['class']}
              Resource:  #{error['resource_type'].capitalize}[#{error['resource_title']}]
              Message:   #{error['message']}

            EOS
          end
        end

        # Attempts to fetch the job run report from the orchestration service.
        # This should contain a report for each node. If it can, it writes
        # reports to files based on the given log_context and prints any errors
        # from each report.
        #
        # If it can't fetch a valid report from the orchestration service:
        #   If the puppet run was on the local node (primary)
        #     It attempts to load the last_run_report.yaml file out of cache. If
        #     that works, it saves it and prints the errors found.
        #     If the last_run_report is for a different job ID, then the puppet
        #     run probably never happened.
        #   If the puppet run was on a remote node
        #     TODO: we could attempt a task to get the remote last_run_report.yaml,
        #     but we need to be sure that we continue if this attempt fails.
        def save_report_logs(job, log_context)
          if job.nil?
            Puppet.warning(_('Cannot attempt to retrieve a run report from a nil job.'))
            return
          end

          begin
            reports = get_job_reports(job.url)
            if reports.empty?
              Puppet.warning(_('Could not fetch the run report from the orchestration service.'))
              node_certnames = job.node_items.map { |n| n['name'] }
              node_certnames.each { |c| retrieve_last_run_report(c, job, log_context)  }
            else
              reports.each { |r| save_report_log(r, log_context) }
            end
          rescue StandardError => e
            Puppet.err(_("Error saving Puppet report logs: %{msg}") % { msg: e.message })
          end
        end

        def retrieve_last_run_report(certname, job, log_context)
          last_run_report = '/opt/puppetlabs/puppet/cache/state/last_run_report.yaml'
          if service_config[:server] == certname
            r = File.read(last_run_report)
            last_run = YAML.safe_load(r, :permitted_classes => [Puppet::Transaction::Report])
            if last_run.job_id.to_s == job.id.to_s
              Puppet.notice(_("Retrieved the local %{last_run_report} from '%{certname}' instead.") % { last_run_report: last_run_report, certname: certname })
              save_report_log(last_run, log_context, certname: certname)
            else
              Puppet.warning(_("We couldn't find a run report that matches this job ID, so the Puppet run likely never occurred. Verify that PE services are running."))
              return
            end
          else
            Puppet.notice(_('More information about this Puppet run (job ID %{job_id}) might be available in the run report stored on %{certname} at %{last_run_report}.' % { job_id: job.id, last_run_report: last_run_report, certname: certname }))
            return
          end
        rescue StandardError => e
          Puppet.err(_("Error retrieving %{last_run_report} for '%{certname}' with the following message: %{msg}") % { last_run_report: last_run_report, certname: certname, msg: e.message })
        end

        def save_report_log(report, log_context, certname: nil)
          certname ||= report['certname']
          report_log = log_context.logfile(certname)
          File.write(report_log, report.to_yaml)
          print_errors(report)
          Puppet.notice(_("Puppet run report for '%{certname}' saved to %{report_log}" % { certname: certname, report_log: report_log }))
        rescue StandardError => e
          Puppet.err(_("Error saving report log for '%{certname}' with the following message: %{msg}") % { certname: certname, msg: e.message })
        end
      end
    end
  end
end
