# frozen_string_literal: true

require 'bolt/error'

# THIS IS A PE SPECIFIC IMPLEMENTATION OF UPLOAD_FILE: THIS IMPLEMENTATION WILL OVERRIDE THE ONE
# THAT COMES WITH A BOLT INSTALLATION
#
# Open-source bolt implementation:
# https://github.com/puppetlabs/bolt/blob/master/bolt-modules/boltlib/lib/puppet/functions/upload_file.rb
#
# upload_file for plans-in-PE will only upload files within modules
# (for security reasons). This is handled by Orchestrator's executor,
# which will fetch these files from puppetserver. This PE implementation
# will send the `source` parameter raw to `executor.upload_file` rather
# than fully qualifying it.
Puppet::Functions.create_function(:upload_file, Puppet::Functions::InternalFunction) do
  dispatch :upload_file do
    scope_param
    param 'String[1]', :source
    param 'String[1]', :destination
    param 'Boltlib::TargetSpec', :targets
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  dispatch :upload_file_with_description do
    scope_param
    param 'String[1]', :source
    param 'String[1]', :destination
    param 'Boltlib::TargetSpec', :targets
    param 'String', :description
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  def upload_file(scope, source, destination, targets, options = {})
    upload_file_with_description(scope, source, destination, targets, nil, options)
  end

  def upload_file_with_description(scope, source, destination, targets, description = nil, options = {})
    unless Puppet[:tasks]
      raise Puppet::ParseErrorWithIssue
        .from_issue_and_stack(Bolt::PAL::Issues::PLAN_OPERATION_NOT_SUPPORTED_WHEN_COMPILING, action: 'upload_file')
    end

    options = options.select { |opt| opt.start_with?('_') }.map { |k, v| [k.sub(/^_/, '').to_sym, v] }.to_h
    options[:description] = description if description
    executor = Puppet.lookup(:bolt_executor)
    inventory = Puppet.lookup(:bolt_inventory)

    executor.report_function_call(self.class.name)

    targets = inventory.get_targets(targets)
    if targets.empty?
      call_function('debug', "Simulating file upload of '#{source}' - no targets given - no action taken")
      r = Bolt::ResultSet.new([])
    else
      r = executor.upload_file(targets, source, destination, options, Puppet::Pops::PuppetStack.top_of_stack)
    end

    if !r.ok && !options[:catch_errors]
      raise Bolt::RunFailure.new(r, 'upload_file', source)
    end
    r
  end
end
