# This runs the provision_compiler plan via the orchestrator.  If --use-ssh is
# provided, it uses Bolt over SSH instead.  By default, ‘puppet’ is used as a
# subject alt name unless --dns-alt-names is provided with different alt names,
# or --no-dns-alt-names is given.
#
# Currently, if in an environment with a load balancer, the node must first be
# pinned to PE Infrastructure Agent so that its pxp-agent is configured to connect
# directly to the primary, rather than the load balancer.  If this isn’t done
# first, the provision will still likely work, but an extra puppet run may be
# needed to apply any relevant settings in hieradata.
#
# Steps the plan will take:
# 1. Check if PE Master node group contains the trusted.extensions.pp_auth_role
#    = pe_compiler rule. If not there we try to add, but if conflicting rules
#    exist plan fails and customer needs to manually add the rule.
# 2. Disable the agent on the primary and postgres node (postgres == primary on a
#    non-split install) using puppet agent --disable.
# 3. Add “puppet_enterprise::profile::database::private_temp_puppetdb_hosts”:
#    [“compiler.certname”] to pe.conf on the primary, and run puppet on the postgres
#    node.  This puts the soon-to-be compiler in the postgres allowlist for the
#    puppetdb database, which is needed in order to bootstrap the PuppetDB service
#    onto the node.
# 4. enterprise_tasks::configure_agent plan:
#        a. If we are using a dns_alt_name (including the default of ‘puppet’),
#           verify that one of the puppetserver conf files in /etc/puppetlabs/puppetserver/conf.d
#           has certificate-authority.allow-subject-alt-names set to true.  This
#           should be managed to true by default.
#        b. Check if the node already has an agent on it.
#             i. If so, check if the agent version matches the primary agent version.
#               1. If not, disable the agent via puppet agent --disable, then
#                  upgrade it via the pe_bootstrap task.
#             ii. Regenerate the cert with the pp_auth_role = pe_compiler
#                 extension request and given subject alt names.
#        c. If the node does not have an agent on it, use the pe_bootstrap task
#           to put an agent on it, with the pp_auth_role = pe_compiler extension
#           and given subject alt names, sign the cert, then run puppet on the node.
# 5. Remove the puppet_enterprise::profile::database::private_temp_puppetdb_hosts
#    value from pe.conf on the primary.
# 6. Run puppet on the postgres node if it is not on the primary node (split install).
# 7. Run puppet on other compilers.
# 8. Re-enable puppet on primary/postgres with puppet agent --enable.
# 9. Run puppet on the primary.
plan enterprise_tasks::provision_compiler(
  TargetSpec $compiler,
  Optional[TargetSpec] $primary   = 'localhost',
  Optional[String] $dns_alt_names = undef,
  Optional[Boolean] $force        = false,
) {
  $constants = constants()
  enterprise_tasks::test_connection($primary)
  enterprise_tasks::verify_node($primary, 'primary', $force)

  $result = run_task(enterprise_tasks::check_pp_auth_role_rule, $primary, _catch_errors => true)
  if !$result.ok() {
    enterprise_tasks::message('provision_compiler',$result.first.error.msg)
    fail_plan('Error adding pp_auth_role rule')
  }

  # Verify that we can sign a CSR with alt names if defined
  # Even though the configure_agent plan would do this, we want to do it first
  # so that we don't mess with allowlisting the node if we aren't going to be able
  # to sign its CSR
  if $dns_alt_names {
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force => $force)
  }

  $dbnodes = enterprise_tasks::get_nodes_with_profile('database')
  # Ignore replicas
  $postgres_nodes = $dbnodes.filter |$node| { !enterprise_tasks::node_has_profile($node, 'primary_master_replica') }
  $postgres_certname = $postgres_nodes[0]
  # If postgres is on the primary, and primary is 'localhost', we want $postgres to be 'localhost' as well
  # so the primary doesn't try SSHing into itself
  $primary_certname = strip(run_command('/opt/puppetlabs/bin/puppet config print certname --section agent', $primary).first.value['stdout'])
  $postgres = $postgres_certname ? {
    $primary_certname => $primary,
    default => $postgres_certname
  }

  $all_puppetservers = enterprise_tasks::get_nodes_with_profile('primary')
  $all_non_primary_puppetservers = $all_puppetservers.filter |$host| { $host != $primary_certname }
  enterprise_tasks::test_connection([$compiler, $postgres])

  # configure_agent already takes care of disabling the agent on the 
  # new compiler node if necessary. We don't want to do it here in case
  # the node is brand new and doesn't have the agent/ruby installed. We
  # don't test connections to other compilers since we want to tolerate
  # if any are down.
  enterprise_tasks::with_agent_disabled([$primary, $postgres].flatten.unique) || {
    # This allows the compiler to be added to the postgres allowlist in advance so that
    # when the compiler starts the pe-puppetdb service for the first time, it can connect.
    run_task(enterprise_tasks::add_modify_conf_keys, $primary,
      file    => $constants['pe_conf'],
      hash    => { $constants['temp_allowlist_key'] => [$compiler] },
    )
    run_task(enterprise_tasks::run_puppet, $postgres)

    # We already checked if we could sign the CSR, so tell configure_agent not to do it again
    run_plan(enterprise_tasks::configure_agent, agent => $compiler,
      primary                       => $primary,
      extension_requests            => { 'pp_auth_role' => 'pe_compiler' },
      dns_alt_names                 => $dns_alt_names,
      check_allow_subject_alt_names => false,
      force                         => $force,
    )

    run_task(enterprise_tasks::remove_conf_keys, $primary,
      file    => $constants['pe_conf'],
      keys    => $constants['temp_allowlist_key'],
    )
    # Do not run puppet on primary in this plan, will restart orchestration services
    unless $postgres_certname == $primary_certname {
      run_task(enterprise_tasks::run_puppet, $postgres)
    }

    # Run on other compilers to update services.conf and crl.pem
    $failed_nodes = $all_non_primary_puppetservers.reduce([]) |$memo, $node| {
      # PlanExecutor will handle printing errors if this fails. We
      # don't want to fail the plan if any of these runs fail, as they
      # are not critical to the provisioning of the new compiler.
      $result = run_task(enterprise_tasks::run_puppet, $node,
          max_timeout   => 256,
          _catch_errors => true,
      )
      if !$result.ok {
        $memo + [$node]
      } else { $memo }
    }
    if $failed_nodes.length > 0 {
      enterprise_tasks::message('provision_compiler', 'We were unable to run puppet on the following compilers. This is not necessary for the new compiler to function, but please run puppet on these compilers when possible.')
      enterprise_tasks::message('provision_compiler', "${failed_nodes.join(', ')}")
    }
    # the above puppet runs can trigger service restarts
    # wait_until_available will ensure connection is back before subsequent tasks are run
    wait_until_available([$compiler, $postgres])
  }
  enterprise_tasks::verify_node($compiler, 'pe_compiler', $force)
}
