require 'puppet_x/util/interview'

module PuppetX
module Puppetlabs
module Meep
module Enable
  class Interview < PuppetX::Util::Interview

    def self.host_port_prompt(role, port)
      return PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
        _("What is the hostname and port of your load balancer for %{role}?") % { role: role },
        _("If you have more than one load balancer, enter them as a comma-separated string, for example:"),
        _("lb1.mycorp.net:%{port},lb2.mycorp.net:%{port}") % { port: port }])
    end

    def self.enable_interview(replica_certname, master_certname, all_groups, options)
      if any_custom_flags?(options)
        validate_infra_flags('enable', options)
        return parse_infra_flags(options)
      else
        conf = PuppetX::Util::Classification.get_config(all_groups)
        suggestions = PuppetX::Util::Classification.suggest_enable_config(conf, master_certname, replica_certname, !options[:skip_agent_config])
        unless STDIN.tty?
          if options[:topology].nil? || options[:yes].nil?
            Puppet.err(_("The enable interview requires a tty to proceed.") + " " + _("To enable without an interview select your topology and pass the --topology and --yes flags to `puppet infrastructure enable`."))
            exit(1)
          # This ensures that we have necessary data (agent_server_urls and
          # pcp_broker_list) to skip the interview if you have compilers
          # (location of load balancer is what we're after). If you've told us
          # to skip the agent configuration though, this is info isn't needed.
          elsif options[:topology] == 'mono-with-compile'
            if options[:skip_agent_config]
              nil
            elsif options[:agent_server_urls].nil? || options[:pcp_brokers].nil?
              Puppet.err(_("The enable interview requires a tty to proceed.") + " " + _("To enable without an interview select your topology and pass the --topology, --yes, --agent-server-urls, and --pcp-brokers flags to `puppet infrastructure enable`."))
              exit(1)
            end
          end
        end
        topo = get_topo(options)

        if PuppetX::Util::Classification.have_config_to_print?(conf)
          Puppet.notice(_("This is your current configuration:\n") +
                        PuppetX::Util::Classification.config_output(conf))
        end

        configure_agent = !options[:skip_agent_config]
        if configure_agent
          puts
          case topo
          when :a
            if options[:agent_server_urls] || options[:pcp_brokers] || options[:primary_uris]
              raise _("--agent-server-urls, --pcp-brokers, and --primary-uris cannot be passed when enabling the 'mono' topology.") + " " + _("Use 'mono-with-compile'")
            end
          when :b
            if options[:agent_server_urls]
              suggestions[:agent_list] = parse_server_list(options[:agent_server_urls])
            else
              server_list = ask_server_list(host_port_prompt('puppet-server', 8140))
              suggestions[:agent_list] = server_list
            end
            if options[:pcp_brokers]
              suggestions[:pcp_broker_list] = parse_server_list(options[:pcp_brokers])
            else
              broker_list = ask_server_list(host_port_prompt('pcp-broker', 8142))
              suggestions[:pcp_broker_list] = broker_list
            end
            if options[:primary_uris]
              suggestions[:primary_uris_list] = parse_server_list(options[:primary_uris])
            else
              # primary_uris should be the same as server_list, so no need to ask twice
              suggestions[:primary_uris_list] = suggestions[:agent_list]
            end
          end
        end

        puts
        Puppet.notice(
          _("When the replica is enabled, your new configuration will be:\n") +
          PuppetX::Util::Classification.config_output(suggestions, configure_agent)
        )
        unless options[:yes]
          puts
          if not configure_agent
            Puppet.notice(_("Note: With the `--skip-agent-config` option, the 'agent-server-urls', 'pcp-brokers', and 'primary-uris' settings aren't configured. You must manually configure these settings for agent failover."))
          end
          unless confirm(PuppetX::Util::String::Formatter.wrap_no_indent(_("Use the new configuration and continue enabling the replica by running puppet on all infrastructure nodes?") + " [Y/n]"))
            exit(1)
          end
        end
        return suggestions
      end
    end
  end
end
end
end
end

