#pragma once
#include <exception>
#include <string>

#include <boost/filesystem.hpp>
#include <leatherman/locale/locale.hpp>
#include <puppet-access/request.hpp>

namespace puppet_access{
class runtime_error : public std::runtime_error {
  public:
    explicit runtime_error(const char *what_arg) : std::runtime_error(what_arg) {}
    explicit runtime_error(const std::string& what_arg) : std::runtime_error(what_arg) {}
};

class file_error : public runtime_error {
  public:
    explicit file_error(const char *desc) : runtime_error(desc) {}
    explicit file_error(const std::string& desc) : runtime_error(desc) {}
};

class file_not_found_exception : public runtime_error {
  public:
    file_not_found_exception(const boost::filesystem::path& path) :
        runtime_error(leatherman::locale::format("No file exists at path {1}.", path.string())) {}
};

class token_not_found_exception : public runtime_error {
  public:
    token_not_found_exception(const boost::filesystem::path& path) :
        runtime_error(leatherman::locale::format("No authentication token exists at path {1}. "
                                                 "Please log in with `puppet access login`", path.string())) {}
};

class http_error : public runtime_error {
  public:
    explicit http_error(const char *desc) : runtime_error(desc) {}
    explicit http_error(const std::string& desc) : runtime_error(desc) {}
};

class api_error : public http_error {
  public:
    request::error_response error;
    explicit api_error(const char *desc, const request::error_response& error) : http_error(desc), error(error) {}
    explicit api_error(const std::string& desc, const request::error_response& error) : http_error(desc), error(error) {}
};

class config_error : public runtime_error {
    public:
        explicit config_error(const char *desc) : runtime_error(desc) {}
        explicit config_error(const std::string& desc) : runtime_error(desc) {}
};

class data_error : public runtime_error {
    public:
        explicit data_error(const char *desc) : runtime_error(desc) {}
        explicit data_error(const std::string& desc) : runtime_error(desc) {}
};
}
