# Removes old PE and Agent packages from primary server, replicas, compilers and database nodes
#
# @param primary [TargetSpec], primary server, default:localhost
# @param pe_version [String], remove PE package versions older than the given pe_version, default: current PE version
# @param force [Boolean], skip node verification, default:false
#
# This plan is run with orchestrator from the primary server.
# This will remove old PE and Agent packages that remain after
# upgrade on the primary server, replicas, compilers and database nodes
#
# Steps this plan takes:
# 1. Delete PE packages older than the current PE version from /opt/puppetlabs/server/data/packages/public
#    on primary server. If a value for pe_version is given, only versions older than pe_version is removed
# 2. Delete agent packages older than the current agent version from /opt/puppetlabs/server/data/staging
# 3. Delete PE and Agent packages from the same locations on replicas, compilers and database nodes
#
plan enterprise_tasks::remove_old_pe_packages(
  Optional[TargetSpec] $primary  = 'localhost',
  Optional[Boolean] $force       = false,
  Optional[String] $pe_version   = 'current'
) {
  enterprise_tasks::test_connection($primary)
  enterprise_tasks::verify_node($primary, 'primary', $force)
  $primary_certname = strip(run_command('/opt/puppetlabs/bin/puppet config print certname --section agent', $primary).first.value['stdout'])
  $all_server_nodes = enterprise_tasks::get_nodes_with_profile('master')
  $all_db_nodes = enterprise_tasks::get_nodes_with_profile('database')
  $all_infra_nodes = ($all_server_nodes + $all_db_nodes).unique

  $current_pe_build = run_command('cat /opt/puppetlabs/server/pe_build', $primary).first['stdout'].strip
  $pe_build = ($pe_version == 'current') ? {
    true => $current_pe_build,
    default => $pe_version
  }

  $agent_version = run_task(enterprise_tasks::get_agent_version, $primary).first().value()['version']

  $pe_cleanup_dir = '/opt/puppetlabs/server/data/packages/public'
  $agent_cleanup_dir = '/opt/puppetlabs/server/data/staging'

  $targets = get_targets($all_infra_nodes)
  enterprise_tasks::message('remove_old_pe_packages', "Checking for old PE packages on ${targets}")
  $r = run_task(enterprise_tasks::remove_old_packages, $targets, pe_version => $pe_build, agent_version => $agent_version, '_catch_errors' => true)
  $r.each |$result| {
    $target = $result.target.name
    if $result.ok {
      $removed_pe_versions_message = empty($result.value['removed_pe_versions']) ? {
        true => "No old PE packages found on ${target}",
        false => "Removed the following old PE package versions: ${result.value['removed_pe_versions']} from ${pe_cleanup_dir} on ${target}",
      }
      $removed_agent_versions_message = empty($result.value['removed_agent_versions']) ? {
        true => "No old agent packages found on ${target}",
        false => "Removed the following old agent package versions: ${result.value['removed_agent_versions']} from ${agent_cleanup_dir} on ${target}",
      }
      enterprise_tasks::message('remove_old_pe_packages', $removed_pe_versions_message)
      enterprise_tasks::message('remove_old_pe_packages', $removed_agent_versions_message)
    }
  }
  unless $r.ok {
    fail_plan('removed_old_pe_packages', "ERROR: Failed on nodes ${r.error_set.names}")
  }
}
