# Upgrades a replica to the primary's version and reprovisions its database.
#
# This upgrade plan relies on first running the standard secondary
# upgrade plan to upgrade the agent on the replica and run puppet
# to complete the upgrade of packages and configuration on the replica.
#
# After that the replica is reprovisioned to quickly get puppetdb back
# in sync, even if we've just upgraded to a new postgres version.
#
# An upgrade from 2018.1 has a special task run to upgrade its
# agent early in order to get the replica running a more pxp-agent
# which the rest of the plan can execute against (the older
# 5.5.x pxp-agents in 2018.1 can't handle run_command() or tasks
# composed of multiple files, for example).
#
# @param replica [Variant[Target,String]] the replica node to upgrade.
# @param primary [Variant[Target,String]] the primary node.
# @param force [Boolean] skip node verification, default:false.
# @param only_recreate_databases [Boolean] Skip upgrade step and just recreate databases on replica by running pg_basebackup.
#
# Steps this plan takes:
# 1. If upgrading from a version prior to 2019.7, delete the reports in the
#    replica’s PuppetDB to speed upgrading PuppetDB. If deleting reports
#    failed, plan will still continue as it is not a blocker for upgrade.
# 2. Follow all the steps in upgrade compiler, as it uses the same plan
#    underneath.
# 3. If upgrading from a version older than 2019.7:
#      a. Disable the agent on the primary with puppet agent --disable.
#      b. Stop all PE services on the replica.
#      c. Follow the step in provision replica to run pg_basebackup on the
#         replica.
#      d. Run puppet infrastructure reinitialize replica -y on the replica.
#      e. Re-enable the agent on the primary with puppet agent --enable.
# 4. Add the replica to puppet_enterprise::profile::puppetdb::sync_allowlist
#    in the PE HA Master node group.
# 5. Add the replica to puppet_enterprise::profile::puppetdb::sync_peers.
#    This will look something like:
#    [{“host”:”replica.certname”,”port”:8081,”sync_interval_minutes”:2}].
# 6. Run puppet on the primary.
# 7. Verify PuppetDB sync status with puppet infra status.  Note that this can
#    take a few minutes, as you have to wait for the replica to try to sync with
#    the primary again.
plan enterprise_tasks::upgrade_and_reprovision_replica(
  Variant[Target,String] $replica,
  Variant[Target,String] $primary,
  Optional[Boolean] $force = false,
  Optional[Boolean] $only_recreate_databases = false,
) {
  $replica_target = get_target($replica)
  $primary_target = get_target($primary)

  # If we are running with --only-recreate-databases, then a previous upgrade run failed at the
  # pg_basebackup step, so the replica is already upgraded and the postgres service is likely down anyway.
  unless $only_recreate_databases {
    # When reprovisioning, delete reports on the replica before upgrade to speed up the process
    $delete_result = run_task(enterprise_tasks::delete_reports, $replica_target, '_catch_errors' => true)
    if !$delete_result.ok() {
      enterprise_tasks::message('upgrade_and_reprovision_replica', "Deleting reports on replica node ${$replica_target} failed. Since the reports are not deleted on the replica, the upgrade may take longer to complete.")
    }
  }

  enterprise_tasks::with_agent_disabled($replica_target) || {
    # Need to get packages installed.
    unless $only_recreate_databases {
      $upgrade_result = run_plan('enterprise_tasks::upgrade_secondary',
        'target'  => $replica_target,
        'primary' => $primary_target,
        'force'   => $force,
        '_catch_errors' => true,
      )
      if $upgrade_result =~ Error {
        enterprise_tasks::message('upgrade_and_reprovision_replica',"ERROR: Upgrading replica ${replica} failed.")
        enterprise_tasks::message('upgrade_and_reprovision_replica',"${upgrade_result.kind()}:${upgrade_result.msg()}")
        enterprise_tasks::message('upgrade_and_reprovision_replica',"${$upgrade_result.details()['result_set']}")
        fail_plan($upgrade_result)
      }
    }
    # Now reprovision
    $reprovision_result = run_plan('enterprise_tasks::reprovision_replica',
      'replica'  => $replica_target,
      'primary'  => $primary_target,
      'force'    => $force,
      '_catch_errors' => true,
    )
    if $reprovision_result =~ Error {
      enterprise_tasks::message('upgrade_and_reprovision_replica',"ERROR: Recreating databases on replica ${replica} failed.")
      enterprise_tasks::message('upgrade_and_reprovision_replica',"${reprovision_result.kind()}:${reprovision_result.msg()}")
      enterprise_tasks::message('upgrade_and_reprovision_replica',"${$reprovision_result.details()['result_set']}")
      enterprise_tasks::message('upgrade_and_reprovision_replica', 'After fixing any issues, please run puppet infra upgrade replica again with the --only-recreate-databases flag. At this point, the replica has been upgraded, but PuppetDB remains unsynced and all reports were deleted on the replica to speed up the upgrade process. The --only-recreate-databases flag will restore all Puppet Enterprise databases to match the primary so that PuppetDB does not need to resync all reports, then finish the upgrade process to restore PuppetDB syncing.')
      fail_plan($reprovision_result)
    }
  }
}
