# This plan will check if an agent is already installed on a non-infra node. If it
# isn't, it will use pe_bootstrap to securely install the agent with the given
# extension_requests and/or custom_attributes. If an agent is already installed,
# it will upgrade the agent to match the primary's version, if necessary, and then
# it will regenerate the cert if extension_requests or custom_attributes is defined.
#
# Note that if extension_requests is defined in order to put a trusted fact in the
# cert that will classify the node as a new infra node, the node will turn into
# the desired infra node since this plan performs a puppet run at the end.
plan enterprise_tasks::configure_agent(
  TargetSpec $agent,
  Optional[TargetSpec] $primary                    = 'localhost',
  Optional[Hash] $extension_requests               = undef,
  Optional[Hash] $custom_attributes                = undef,
  Optional[String] $dns_alt_names                  = undef,
  Optional[Boolean] $check_allow_subject_alt_names = true,
  Optional[Boolean] $manage_pxp_service            = true,
  Optional[Boolean] $force                         = false,
) {
  $constants = constants()
  enterprise_tasks::test_connection([$agent, $primary])
  $primary_certname = run_command("${constants['puppet_bin']} config print certname", $primary).first['stdout'].strip

  enterprise_tasks::verify_node($primary, 'primary', $force)

  enterprise_tasks::message('configure_agent','Obtaining versions...')
  $agent_version = run_task(enterprise_tasks::get_agent_version, $agent).first().value()['version']
  $primary_agent_version = run_task(enterprise_tasks::get_agent_version, $primary).first().value()['version']
  $cacert_contents = run_command("cat ${constants['ca_pem']}", $primary).first['stdout']

  if $check_allow_subject_alt_names and $dns_alt_names {
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force => $force)
  }

  $allow_subject_alt_names = $dns_alt_names ? {
    undef => false,
    default => true
  }

  $dns_alt_names_param = $dns_alt_names ? {
    undef => {},
    default => { dns_alt_names => $dns_alt_names },
  }

  if $agent_version {
    if $agent_version != $primary_agent_version {
      # pe_bootstrap appears not to add the extension requests if you are upgrading the agent, so we'll need
      # to do agent cert regen after this in that case. We should first verify this is not an infra node already.
      # Also, pe_bootstrap passes through dns_alt_names='null' when dns_alt_names=undef
      # (see https://www.puppet.com/docs/bolt/latest/writing_tasks.html#defining-parameters-in-tasks)
      # so we have to mess with the parameters like this until pe_bootstrap is fixed.

      enterprise_tasks::verify_node($agent, 'agent', $force)
      enterprise_tasks::with_agent_disabled([$agent]) || {
        enterprise_tasks::message('configure_agent','Upgrading agent via pe_bootstrap...')
        $upgrade_bootstrap_params = {
          cacert_content => $cacert_contents,
          server         => $primary_certname,
        } + $dns_alt_names_param
        run_task(pe_bootstrap, $agent, $upgrade_bootstrap_params)
      }
    }
    if $extension_requests or $custom_attributes {
      enterprise_tasks::message('configure_agent','Regenerating agent certificate...')
      run_plan(enterprise_tasks::agent_cert_regen, agent => $agent,
        primary            => $primary,
        extension_requests => $extension_requests,
        custom_attributes  => $custom_attributes,
        dns_alt_names      => $dns_alt_names,
        manage_pxp_service => $manage_pxp_service,
        force              => $force,
      )
      run_task(enterprise_tasks::run_puppet, $agent,
        max_timeout => 256,
      )
    }
  }
  else {
    if $extension_requests {
      $extension = $extension_requests.map |$key, $value| { "${key}=${value}" }
    } else {
      $extension = []
    }
    if $custom_attributes {
      $custom = $custom_attributes.map |$key, $value| { "${key}=${value}" }
    } else {
      $custom = []
    }
    enterprise_tasks::message('configure_agent','Installing agent via pe_bootstrap...')
    # pe_bootstrap passes through dns_alt_names='null' when dns_alt_names=undef
    # (see https://www.puppet.com/docs/bolt/latest/writing_tasks.html#defining-parameters-in-tasks)
    # so we have to mess with the parameters like this until pe_bootstrap is fixed.
    $install_bootstrap_params = {
      cacert_content    => $cacert_contents,
      server            => $primary_certname,
      extension_request => $extension,
      custom_attribute  => $custom,
    } + $dns_alt_names_param
    run_task(pe_bootstrap, $agent, $install_bootstrap_params)

    enterprise_tasks::message('configure_agent','Signing agent certs...')
    run_task(enterprise_tasks::puppetserver_ca_sign_api, $primary,
      certname                => $agent,
      allow_subject_alt_names => $allow_subject_alt_names,
      dns_alt_names           => $dns_alt_names,
    )
    # Since this is a brand new node, it should be safe to stop/start the service without checking
    # the existing service status beforehand. We actually need to disable the service here, rather than
    # disabling the agent itself, due to how the agent works when it is first installed.
    run_task(enterprise_tasks::pe_services, $agent, state => 'stopped') # Stops only 'puppet' since we pass in no role
    run_task(enterprise_tasks::run_puppet, $agent,
      max_timeout => 256,
    )
    run_task(enterprise_tasks::pe_services, $agent, state => 'running')
  }
}
