define puppet_enterprise::trapperkeeper::classifier (
  $client_certname               = $facts['clientcert'],
  $container                     = $title,
  $database_host                 = 'localhost',
  $database_name                 = $puppet_enterprise::params::classifier_database_name,
  $database_user                 = $puppet_enterprise::classifier_service_regular_db_user,
  $database_migration_user       = $puppet_enterprise::classifier_service_migration_db_user,
  $database_password             = undef,
  $database_port                 = $puppet_enterprise::database_port,
  $database_properties           = '',
  $allow_config_data             = $puppet_enterprise::params::classifier_allow_config_data,
  $group                         = "pe-${title}",
  $localcacert                   = $puppet_enterprise::params::localcacert,
  Optional[String] $master_host = undef,
  Optional[Integer] $master_port = undef,
  String $server_host  = pe_pick($master_host, $puppet_enterprise::puppet_master_host),
  Integer $server_port  = pe_pick($master_port, $puppet_enterprise::puppet_master_port),
  $prune_days_threshold          = $puppet_enterprise::params::classifier_prune_threshold,
  $synchronization_period        = $puppet_enterprise::params::classifier_synchronization_period,
  $user                          = "pe-${title}",
  Boolean $node_check_in_storage = false,
  Array[String]  $puppetdb_hosts = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_ports = $puppet_enterprise::puppetdb_ports_array,
  $fact_path_sync_period         = $puppet_enterprise::params::classifier_fact_path_sync_period

) {
  puppet_enterprise::deprecated_parameter { 'master_host': replacement => 'server_host' }
  puppet_enterprise::deprecated_parameter { 'master_port': replacement => 'server_port' }
  $cert_dir = "${puppet_enterprise::server_data_dir}/${container}/certs"
  $client_ssl_key  = "${cert_dir}/${client_certname}.private_key.pem"
  $client_ssl_cert = "${cert_dir}/${client_certname}.cert.pem"
  $classifier_conf_path = "/etc/puppetlabs/${container}/conf.d/classifier.conf"
  File {
    owner => $user,
    group => $group,
    mode  => '0640',
  }

  Pe_hocon_setting {
    ensure  => present,
    notify  => Service["pe-${container}"],
  }

  # Uses
  #   $server_host
  #   $client_ssl_key
  #   $client_ssl_cert
  #   $localcacert
  #   $synchronization_period
  #   $prune_days_threshold
  #   $fact_path_sync_period
  #   $puppetdb_hosts
  #   $puppetdb_ports
  file { $classifier_conf_path:
    ensure => present,
  }

  # URL for the puppet server's rest interface
  pe_hocon_setting { "${container}.classifier.puppet-master":
    ensure  => absent,
    path    => $classifier_conf_path,
    setting => 'classifier.puppet-master',
    value   => "https://${server_host}:${server_port}",
  }

  # URL for the puppet server's rest interface
  pe_hocon_setting { "${container}.classifier.puppet-server":
    ensure  => present,
    path    => $classifier_conf_path,
    setting => 'classifier.puppet-server',
    value   => "https://${server_host}:${server_port}",
  }

  pe_hocon_setting { "${container}.classifier.allow-config-data":
    ensure  => present,
    path    => $classifier_conf_path,
    setting => 'classifier.allow-config-data',
    value   => $allow_config_data,
  }

  # Configure the SSL settings to enable SSL when communicating with puppet server
  pe_hocon_setting { "${container}.classifier.ssl-key":
    path    => $classifier_conf_path,
    setting => 'classifier.ssl-key',
    value   => $client_ssl_key,
  }
  pe_hocon_setting { "${container}.classifier.ssl-cert":
    path    => $classifier_conf_path,
    setting => 'classifier.ssl-cert',
    value   => $client_ssl_cert,
  }
  pe_hocon_setting { "${container}.classifier.ssl-ca-cert":
    path    => $classifier_conf_path,
    setting => 'classifier.ssl-ca-cert',
    value   => $localcacert,
  }
  # How often (in seconds) the classifier refreshes classes from the Primary
  pe_hocon_setting { "${container}.classifier.synchronization-period":
    path    => $classifier_conf_path,
    setting => 'classifier.synchronization-period',
    value   => $synchronization_period,
  }
  pe_hocon_setting { "${container}.classifier.prune-days-threshold":
    path    => $classifier_conf_path,
    setting => 'classifier.prune-days-threshold',
    value   => $prune_days_threshold,
  }
  pe_hocon_setting { "${container}.classifier.node-check-in-storage":
    path    => $classifier_conf_path,
    setting => 'classifier.node-check-in-storage',
    value   => $node_check_in_storage,
  }
  # How often (in seconds) the classifier refreshes the fact-path cache from the Primary
  pe_hocon_setting {"${container}.classifier.fact-path-sync-period":
    path    => $classifier_conf_path,
    setting => 'classifier.fact-path-sync-period',
    value   => $fact_path_sync_period,
  }

  # For PuppetDB HA, a user may pass in an Array to specify their PuppetDBs
  $first_puppetdb_host = $puppetdb_hosts[0]
  $first_puppetdb_port = pe_any2array($puppetdb_ports)[0]
  pe_hocon_setting { "${container}.classifier.puppetdb-server":
    path    => $classifier_conf_path,
    setting => 'classifier.puppetdb-server',
    value   => "https://${first_puppetdb_host}:${first_puppetdb_port}",
  }

  puppet_enterprise::trapperkeeper::database_settings { 'classifier' :
    container           => $container,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    database_port       => Integer($database_port),
    database_properties => $database_properties,
    database_user       => $database_user,
    migration_user      => $database_migration_user,
    migration_password  => $database_password,
    group               => $group,
    user                => $user,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:classifier classifier-service" :
    container => $container,
    namespace => 'puppetlabs.classifier.main',
    service   => 'classifier-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:classifier fact-path-cache-service" :
    container => $container,
    namespace => 'puppetlabs.classifier.main',
    service   => 'fact-path-cache-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:classifier activity-reporting-service" :
    container => $container,
    namespace => 'puppetlabs.activity.services',
    service   => 'activity-reporting-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:classifier jetty10-service" :
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.webserver.jetty10-service',
    service   => 'jetty10-service',
  }

}
