require 'json'
require 'net/https'
require 'hocon'
require 'open3'

Puppet::Face.define(:node, '0.0.1') do

  action(:purge) do

    summary "Deactivate nodes, delete from inventory, delete PuppetDB data, and clean node info from the primary"
    arguments "<node> [<node> ...]"
    description <<-DESC
      Calls "puppet node deactivate", deletes the node from the inventory service
      via the /inventory/v1/command/delete-connection endpoint,
      deletes PuppetDB data for the node via the /pdb/admin/v1/cmd endpoint, then
      runs "puppet node clean" to remove the node's information and
      certificates from the primary.
    DESC

    when_invoked do |*args|
      success = true
      opts = args.pop
      nodes = args
      raise ArgumentError, "Please provide at least one node" if nodes.empty?

      # The deactivate action should only be available on primary nodes as it is
      # included in the puppetdb-terminus package.
      if Puppet::Interface.find_action(:node, :deactivate, :current)
        # Deactivate before deleting
        # This will hopefully help fix an edge case race condition between submitting
        # the delete request to a replicating PDB. Note that we can't really check
        # if there were real errors inside the command, so assume if we got something
        # back, we're okay.
        deactivate_result = Puppet::Face[:node, :current].deactivate(*nodes)
        success &= !deactivate_result.nil?
        nodes.each do |node|
          success &= delete_node_in_inventory(node)
          success &= delete_node_in_pdb(node)
        end
        # This will print an error if there are no certs to clean, which isn't great,
        # but don't treat that as an error for the purposes of this command. We can't
        # really find what those errors were, so for the purposes of the status message
        # at the end, assume if we got a result, we're okay.
        node_list = Puppet::Face[:node, :current].clean(*nodes)
        success &= !node_list.nil?
        {:node_list => node_list, :success => success}
      else
        raise "This command must be run from the primary server."
      end
    end

    when_rendering(:console) do |result|
      multiple_nodes = result[:node_list].length > 1
      s = multiple_nodes ? 's' : ''
      waswere = multiple_nodes ? 'were' : 'was'
      node_list_str = multiple_nodes ? result[:node_list] : %Q|"#{result[:node_list][0]}"|
      if result[:success]
        purged_msg = "Node#{s} #{node_list_str} #{waswere} purged."
      else
        purged_msg = "There were errors attempting to fully purge node#{s} #{node_list_str}. After correcting these issues, you may wish to run this command again to finish purging all traces of the node#{s}."
      end

      <<-MSG
#{purged_msg}

To ensure a node can not check into any compilers, run 'puppet agent -t' on all compilers.

To re-add a node to your Puppet Enterprise infrastructure:
On the node:
   1. Run 'puppet ssl clean' to purge that node's local certificates.
   2. Run 'puppet agent -t' to generate a new certificate request.
      MSG
    end
  end

  def get_pdb_uris
    config = Puppet::Util::Puppetdb.config
    (config.server_urls || []) + (config.submit_only_server_urls || [])
  end

  def delete_node_in_inventory(node)
    begin
      uri = URI.parse("https://#{Puppet[:certname]}:8143/inventory/v1/command/delete-connection")
      http_client = Puppet.runtime[:http]
      headers = { 'Content-Type' => 'application/json' }
      body = { 'certnames': [node] }

      res = http_client.post(uri, body.to_json, headers: headers)
      if res.code != 204
        Puppet.err _("An error occurred getting a response from the inventory service while running command %{cmd}:\nHTTP %{rescode}\n%{resbody}") % {cmd: 'delete-connection', rescode: res.code, resbody: res.body}
        return false
      end
    rescue Exception => e
      Puppet.err _("An exception occurred while attempting to contact the inventory service\n  %{class}: %{message}") % {class: e.class, message: e.message}
      if Puppet[:debug] || Puppet[:trace]
        Puppet.err _("%{full_message}") % {full_message: e.full_message}
      end
      return false
    end
    true
  end

  def delete_node_in_pdb(node)
    http_client = Puppet.runtime[:http]
    pdb_uris = get_pdb_uris

    if pdb_uris.empty?
      Puppet.err _('Error loading PuppetDB config. Ensure server_urls is defined in /etc/puppetlabs/puppet/puppetdb.conf')
      return false
    end

    headers = { 
      'Content-Type' => 'application/json',
      'Accept'       => 'application/json',
    }
    body = {
      'command': 'delete',
      'version': 1,
      'payload': { 'certname': node }
    }

    pdb_uris.reduce(true) do |ok, uri|
      begin
        uri.path = "/pdb/admin/v1/cmd"
        res = http_client.post(uri, body.to_json, headers: headers)
        if res.code != 200
          Puppet.err _("An error occurred getting a response from the PuppetDB service on #{uri.host} while running command %{cmd}:\nHTTP %{rescode}\n%{resbody}") % {cmd: 'delete', rescode: res.code, resbody: res.body}
          # Set 'ok' memo to false
          false
        else
          # Set 'ok' memo to true if it was true before
          ok && true
        end
      rescue Exception => e
        Puppet.err _("An exception occurred while attempting to contact the PuppetDB service at %{host}\n  %{class}: %{message}") % {host: uri.host, class: e.class, message: e.message}
        if Puppet[:debug] || Puppet[:trace]
          Puppet.err _("%{full_message}") % {full_message: e.full_message}
        end
        # Set 'ok' memo to false
        false
      end
    end
  end
end
