# This profile sets up the PE Bolt Server service.
# @param certname [String] The name of the service SSL certificate.
# @param run_service [Boolean] Should the service be running
# @param ssl_listen_port [Boolean] The port used for SSL connections.
# @param master_host [String] The hostname of the primary.
# @param master_port [Integer] The port of puppet server on the primary.
# @param whitelist [Optional[Array[String]]] * Deprecated * Use allowlist instead.
# @param allowlist [Array[String]] Certificates that can be used to make requests to the service.
# @param ssl_cipher_suites [Array[String]] Cipher suites for TLS in preference order.
# @param loglevel [Enum[debug, info, error, warn]] Set service log level.
# @param container [String] The name of the service to run
# @param settings [Hash] Any additional hocon settings to set in the service configuration
# @param projects_codedir [Optional[String]] The directory where bolt-server looks for code when using projects
# @param environments_codedir [Optional[String]] The directory where bolt-server looks for code when using environments
# @param basemodulepath [Optional[String]] The location where the plan executor will expect to load base modules
# @param environmentpath [Optional[String]] The location where the plan executor will expect to load environments

class puppet_enterprise::profile::bolt_server(
  Integer $ssl_listen_port               = $puppet_enterprise::bolt_server_port,
  String $certname                       = $facts['clientcert'],
  String $master_host                    = $puppet_enterprise::puppet_master_host,
  Integer $master_port                   = $puppet_enterprise::puppet_master_port,
  Boolean $run_service                   = true,
  Optional[Array[String]] $whitelist     = undef,
  Array[String] $allowlist               = pe_pick($whitelist, [$certname]),
  Array[String] $ssl_cipher_suites       = $puppet_enterprise::ssl_cipher_suites_non_java,
  Enum[debug, info, error, warn] $service_loglevel = 'info',
  Integer $concurrency                   = $puppet_enterprise::bolt_server_concurrency,
  Hash $settings                         = {},
  Optional[String] $projects_codedir     = '/opt/puppetlabs/server/data/orchestration-services/projects/',
  Optional[Array] $builtin_content_dir   = $puppet_enterprise::params::builtin_bolt_content_dir,
  Optional[String] $environments_codedir = undef,
  Optional[String] $basemodulepath       = undef,
  Optional[String] $environmentpath      = undef,
) {
  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::profile::bolt_server::whitelist': replacement => 'puppet_enterprise::profile::bolt_server::allowlist' }
  $container = 'bolt-server'

  # The following three settings are _intentionally obfuscated_ from the user. These
  # puppet vars can be found in the Console UI with a little digging, but we intentionally
  # do not want them to look like default settings that are meant to be used. So the following
  # code ensures that if these settings are set in puppet code they show up in the bolt-server
  # configuration, but if they are not set these settings aren't shown at all in the conf files.
  #
  # All default values are designed to work correctly together to allow bolt content to
  # be found. These settings are made available in the rare case a user cannot use
  # code manager/file sync and they wish to point bolt-server at a specific location for their
  # code.
  #
  # USING THESE SETTINGS TO AVOID CODE MANAGER/FILE SYNC IS HIGHLY DISCOURAGED. It should work
  # but we need to ensure that users who ignore code manager/file sync understand it could be
  # potentially damaging to their infrastructure if they change code while running a plan.
  if $environments_codedir {
    $_environments_codedir_hash = { 'environments-codedir' => $environments_codedir }
  } else {
    $_environments_codedir_hash = {}
  }

  if $basemodulepath {
    $_basemodulepath_hash = { 'basemodulepath' => $basemodulepath }
  } else {
    $_basemodulepath_hash = {}
  }

  if $environmentpath {
    $_environmentpath_hash = { 'environmentpath' => $environmentpath }
  } else {
    $_environmentpath_hash = {}
  }

  puppet_enterprise::orchestrator::ruby_service { $container:
    container         => $container,
    ssl_listen_port   => $ssl_listen_port,
    certname          => $certname,
    run_service       => $run_service,
    allowlist         => $allowlist,
    ssl_cipher_suites => $ssl_cipher_suites,
    service_loglevel  => $service_loglevel,
    settings          =>  { 'concurrency' => $concurrency,
                            'cache-dir'           => "/opt/puppetlabs/server/data/${container}/cache",
                            'file-server-uri'     => "https://${master_host}:${master_port}",
                            'projects-dir'        => $projects_codedir,
                            'builtin-content-dir' => $builtin_content_dir
                          } + $settings + $_environments_codedir_hash + $_basemodulepath_hash + $_environmentpath_hash
  }

  # The bolt-server service won't be able to serve plan metadata from the orchestrator
  # codedir unless the pe-bolt-server user has the pe-orchestration-services group.
  user { 'pe-bolt-server':
    groups => [ 'pe-bolt-server', 'pe-orchestration-services'],
    require => [ Package['pe-bolt-server'], Package['pe-orchestration-services'] ]
  }
}
