# frozen_string_literal: true

require 'bolt'
require 'bolt/analytics'
require 'bolt/config'
require 'bolt/error'
require 'bolt/inventory'
require 'bolt/pal'
require 'bolt/plugin'
require 'bolt/puppetdb'
require 'bolt/version'
require 'json'
require 'plan_runner/bolt_overrides/applicator'
require 'plan_runner/bolt_overrides/executor'
require 'plan_runner/bolt_overrides/inventory'
require 'plan_runner/service/http_client'
require 'plan_runner/service/encryption'

module PlanRunner
  module Execution
    class EnvironmentPlan
      attr_reader :result, :error

      def initialize(service_config, plan_id, plan_name, environment, params, sensitive_params, aes_key)
        @default_pdb = service_config[:default_pdb]
        @pdb_instances = service_config[:pdb_instances]
        @logger = Logging.logger[self]
        @codedir = service_config[:codedir]

        @logger.debug("Initializing HTTP Client")
        @http_client = service_config[:http_client] ||
                       Service::HttpClient.new(
                         service_config[:orch_url],
                         service_config
                       )

        @encryption_key = Service::AESEncryptionCipher.new(aes_key)
        @result = {}

        @plan_id = plan_id
        @plan_name = plan_name
        @environment = environment
        @params = if sensitive_params
                    params.merge!(
                      JSON.parse(@encryption_key.decrypt(sensitive_params))['parameters']
                    )
                  else
                    params
                  end
        @stopped = false

        @failed = false
        @error = nil
      end

      def failed?
        @failed
      end

      def fail_plan_with(error)
        @failed = true
        @error = error
      end

      def set_signal_handler
        Signal.trap("TERM") do
          @stopped = true
          exit
        end
      end

      # runs a plan from an environment
      def run(
        plan_id = @plan_id,
        plan_name = @plan_name,
        environment = @environment,
        params = @params
      )
        plan_result = with_environment_pal(environment) do |pal|
          run_plan(pal, plan_name, params, plan_id, environment)
        end
        @result = process_plan_run_result(plan_result)
      # I CAN RESCUE WHATEVER I WANT RUBOCOP
      # rubocop:disable Lint/RescueException
      rescue Exception => e
        fail_plan_with(e)
        @logger.debug("Exception raised during plan run, #{e.full_message}")
        plan_result = if e.is_a?(Bolt::Error)
                        scrub_bolt_error(e)
                      else
                        build_error_result(e.message, 'bolt/plan-failure', { 'class' => e.class.to_s })
                      end

        @result = {
          'result' => plan_result,
          'status' => 'failure'
        }
      end
      # rubocop:enable Lint/RescueException

      def scrub_bolt_error(err)
        if err.kind == 'bolt/unknown-plan'
          msg = "Could not find plan named \"#{plan_name}\". Code manager is required to run plans. " \
                'If code manager is enabled, please contact Support'
          build_error_result(msg, 'bolt/unknown-plan')
        else
          # catchall for custom Bolt::Error parameters from fail_plan
          build_error_result(err.message, err.kind, err.details)
        end
      end

      # This function is nearly identical to Bolt::Pal's `with_puppet_settings` with the
      # one difference that we set the codedir to point to actual code, rather than the
      # tmpdir. We only use this funtion inside the Modulepath initializer so that Puppet
      # is correctly configured to pull environment configuration correctly. If we don't
      # set codedir in this way: when we try to load and interpolate the modulepath it
      # won't correctly load.
      def with_pe_pal_init_settings(codedir, environmentpath, basemodulepath)
        @logger.debug("Clearing settings to ensure no interferance with puppet")
        Dir.mktmpdir('pe-bolt') do |dir|
          cli = []
          Puppet::Settings::REQUIRED_APP_SETTINGS.each do |setting|
            dir = codedir if setting == :codedir
            cli << "--#{setting}" << dir
          end
          cli << '--environmentpath' << environmentpath
          cli << '--basemodulepath' << basemodulepath
          Puppet.settings.send(:clear_everything_for_tests)
          Puppet.initialize_settings(cli)
          Puppet[:versioned_environment_dirs] = true
          yield
        end
      end

      # Use puppet to identify the relevant paths for an environment.
      # These are currently the modulepath and environment hiera.yaml
      # files.
      def environment_paths(environment_name)
        codedir = @codedir
        environmentpath = "#{codedir}/environments"
        basemodulepath = "#{codedir}/modules:/opt/puppetlabs/puppet/modules"
        paths = {}
        with_pe_pal_init_settings(codedir, environmentpath, basemodulepath) do
          environment = Puppet.lookup(:environments).get!(environment_name)
          path_to_env = environment.configuration.path_to_env

          # If we find a bolt.yaml config file in environment root, warn that we are ignoring it
          if path_to_env && File.exist?(File.join(path_to_env, 'bolt.yaml'))
            @logger.warn("Ignoring bolt.yaml configuration file in #{environment_name}")
          end

          paths[:modulepath] = environment.modulepath

          if path_to_env
            hiera_yaml = File.join(path_to_env, 'hiera.yaml')
            if File.exist?(hiera_yaml)
              @logger.debug("For environment #{environment_name}: loading #{hiera_yaml}")
              paths[:hiera] = hiera_yaml
            else
              msg = "For environment #{environment_name}: #{hiera_yaml} does not exist so it will not be loaded"
              @logger.debug(msg)
            end
          end
        end
        paths
      end

      def with_environment_pal(environment)
        paths = environment_paths(environment)
        modulepath_obj = Bolt::Config::Modulepath.new(
          paths[:modulepath],
          boltlib_path: ['/opt/puppetlabs/server/apps/bolt-server/pe-bolt-modules',
                         Bolt::Config::Modulepath::BOLTLIB_PATH]
        )
        @logger.trace("User modulepath: #{modulepath_obj.user_modulepath}")
        @logger.trace("Full modulepath: #{modulepath_obj.full_modulepath}")
        yield Bolt::PAL.new(modulepath_obj, paths[:hiera], nil)
      ensure
        Puppet.settings.send(:clear_everything_for_tests)
      end

      def run_plan(pal, plan_name, params, plan_id = nil, environment = nil)
        @logger.debug("Initializing Executor")
        executor = BoltOverrides::Executor.new(@http_client, plan_id, environment, @encryption_key)

        @logger.debug("Initializing Inventory")
        config = Bolt::Config.default
        plugins = Bolt::Plugin.new(config, nil)
        inventory = BoltOverrides::Inventory::Inventory.new(
          {},
          config.transport,
          config.transports,
          plugins
        )

        @logger.debug("Initializing Applicator")
        applicator = BoltOverrides::Applicator.new(inventory, executor, nil)

        @logger.debug("Initializing PuppetDB Client")
        puppetdb_client = Bolt::PuppetDB::Client.new(config: @default_pdb, instances: @pdb_instances)
        puppetdb_client.instance_variable_get(:@instances).each do |inst|
          @logger.debug("Configuring PuppetDB instance '#{inst}' http client")
          inst.instance_variable_set(:@http, @http_client)
        end
        @logger.debug("Configuring default PuppetDB instance http client")
        puppetdb_client
          .instance_variable_get(:@default_instance)
          .instance_variable_set(:@http, @http_client)

        @logger.info("Starting execution of #{plan_name}")
        pal.run_plan(plan_name, params, executor, inventory, puppetdb_client, applicator)
      end

      def process_plan_run_result(plan_result)
        # Iterate over values and clean them, this includes converting to POD and scrubbing Bolt::Errors
        result = Bolt::Util.walk_vals(plan_result.value) do |val|
          if val.respond_to? :to_data
            # In Bolt, a Result and ResultSet respond to `to_data`
            val.to_data
          elsif val.is_a? Bolt::Target
            val.safe_name
          elsif val.is_a? Exception
            fail_plan_with(val)
            @logger.debug("Exception returned from plan run, #{val.full_message}")
            case val
            when Bolt::RunFailure || Bolt::ApplyFailure
              # Structure RunFailure output the same way Bolt does, with a `result_set` array in the `details` key
              build_error_result(val.message, val.kind, { 'class' => val.class.to_s,
                                                          'action' => val.details['action'],
                                                          'object' => val.details['object'],
                                                          'result_set' => val.result_set.to_data })
            when Bolt::Error
              case val.kind
              when 'bolt/unknown-plan'
                msg = val.message.sub(' For a list of available plans, run "bolt plan show"', '')
                build_error_result(msg, 'bolt/unknown-plan')
              when 'bolt/unknown-task'
                msg = val.message.sub(' For a list of available tasks, run "bolt task show"', '')
                build_error_result(msg, 'bolt/unknown-task')
              else
                # catchall for custom Bolt::Error parameters from fail_plan
                build_error_result(val.message, val.kind, val.details)
              end
            else
              build_error_result(val.message, 'bolt/plan-failure', { 'class' => val.class.to_s })
            end
          else
            val
          end
        end

        # Ensure all keys are strings in hashes
        result = Bolt::Util.walk_keys(result, &:to_s)

        {
          'result' => result,
          'status' => plan_result.status
        }
      end

      def build_error_result(msg, kind, details = {})
        {
          'msg' => msg,
          'kind' => kind,
          'details' => details
        }
      end

      def submit_result
        # Don't attempt to send this result if the plan has been stopped. The
        # runner will take care of sending a 'stopped plan' result
        return if @stopped
        url = '/v2/private/plans/' + @plan_id + '/result'
        @logger.debug("PUT to API to finish plan #{@plan_id}")
        response, http_code = @http_client.put_with_orch_cert(url, @result)
        if http_code != 200
          msg = "Failed API response with code #{http_code}: #{response}"
          raise Bolt::Error.new(msg, 'bolt.plan-executor/errored-response')
        end
      end
    end
  end
end
