# This plan is run with Bolt using localhost, so SSH is not required.
# This is used for removing old postgres versions that remain on the primary
# after we’ve migrated to a new major postgres version (e.g. 9.6 -> 11).
# We don’t clean this up automatically, just in case a user has to roll back.
#
# Steps this plan takes, for each old version present:
# 1. Remove /opt/puppetlabs/server/apps/postgresql/<version>
# 2. Remove /opt/puppetlabs/server/data/postgresql/<version>
# 3. Remove tablespace directories. These will look like
#    /opt/puppetlabs/server/data/postgresql/<table_name>/PG_<version>_<number>.
#    Tablespaces should be activity, classifier, orchestrator, puppetdb, and
#    rbac.
# 4. Remove pe-postgresql packages as below for old versions if they are present.
#      a. pe-postgresql<version>-pglogical
#      b. pe-postgresql<version>-contrib
#      c. pe-postgresql<version>-server
#      d. pe-postgresql<version>
# 5. Restart the pe-postgresql service, as removing the old packages kills the
#    service.
plan enterprise_tasks::remove_old_postgresql_versions(
  TargetSpec $primary = 'localhost',
) {
  $constants = constants()
  $postgres_dir = '/opt/puppetlabs/server/data/postgresql/'
  enterprise_tasks::test_connection($primary)

  $result_or_error = catch_errors() || {
    $pg_present = run_task(enterprise_tasks::check_for_postgres, $primary).first.value['pg_present']
    if !$pg_present {
      fail_plan('This node does not appear to have PE Postgresql installed.')
    }

    $old_versions_info = run_task(enterprise_tasks::get_old_postgres_info, $primary).first.value['info']
    $versions = $old_versions_info['versions']
    $packages = $old_versions_info['packages']
    $dirs     = $old_versions_info['dirs']
    if !empty($versions) {
      enterprise_tasks::message('remove_old_postgresql_versions', "Removing old postgresql versions ${versions}")
      if !empty($packages) {
        enterprise_tasks::message('remove_old_postgresql_versions', "Removing old postgresql packages ${packages}")
        $packages.each |$p| {
          run_task('package', $primary, 'action' => 'uninstall', 'name' => $p)
        }
      }
      if !empty($dirs) {
        enterprise_tasks::message('remove_old_postgresql_versions', "Removing old postgresql directories ${dirs}")
        $delete_dirs_result = apply($primary) {
          file { $dirs:
            ensure  => absent,
            recurse => true,
            purge   => true,
            force   => true,
          }
        }
      }
      # Removing the old postgres packages kills the pe-postgres service; so we should restart it
      run_command("${constants['puppet_bin']} resource service pe-postgresql ensure=running enable=true", $primary)
    } else {
      enterprise_tasks::message('remove_old_postgresql_versions', 'No old postgresql versions to remove')
    }
  }
  if $result_or_error =~ Error {
    enterprise_tasks::message('remove_old_postgresql_versions', "${result_or_error.kind()}:${result_or_error.details()}")
    return fail_plan($result_or_error)
  }
}
