#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require_relative '../files/api_helper.rb'

class PuppetserverCACleanAPI < EnterpriseTaskHelper
  def task(certname:, **_kwargs)
    begin # rubocop:disable Style/RedundantBegin (it is, but not worth changing now)
      options = { 'open_timeout' => 30, 'read_timeout' => 30 }
      headers = { 'Content-Type' => 'application/json' }
      # Since we don't specify 'server', it defaults to Puppet[:certname]. Since this only runs on the primary, we're good.
      api = APIHelper.new(8140, 'puppet-ca/v1', options: options, headers: headers)

      certnames = [certname].flatten
      # In case all the steps in the plan failed before we got here
      return if certnames.empty?

      response = nil
      begin
        body = "{\"certnames\":[\"#{certnames.join('","')}\"]}"
        response = api.request('clean', :put, body: body, valid_codes: 200)
      rescue StandardError => e
        raise EnterpriseTaskHelper::Error.new(
          "#{e.class} when attempting to clean certs via API: #{e.message}",
          'puppetlabs.puppetserver-ca-clean-api/clean-error',
          'errored_certnames' => certnames,
          'exception' => e,
        )
      end

      # These response strings have been unchanged since the clean endpoint was created.
      # However, it is not ideal to have to parse these strings. At some point, we should
      # make the CA return JSON instead.
      if response.include?('Successfully cleaned all certs')
        { cleaned: certnames, not_found: [] }.to_json
      elsif response.include?('The following certs do not exist')
        # If the certs don't exist, then for the purposes of this task, consider them cleaned.
        # They may or may not be on the CRL since the clean action fails, but soon we won't be
        # adding agent certs to the CRL during regen anyway.

        # Barf. This can probably be less jank with a better regex.
        not_found = response.scan(/The following certs do not exist and cannot be revoked: \[(.*)\]/).flatten.first.scan(/\"(\S+)\"/).flatten
        { cleaned: certnames - not_found, not_found: not_found }.to_json
      else
        raise EnterpriseTaskHelper::Error.new(
          "Unexpected response from clean endpoint: #{response}",
          'puppetlabs.puppetserver-ca-clean-api/unexpected-response',
          'errored_certnames' => certnames,
        )
      end
    rescue EnterpriseTaskHelper::Error
      raise
    rescue StandardError => e # Catching StandardError rather than Exception here and above so as not to catch signals intended to manage the process
      # The reason we repackage the exception here is so that the agent_cert_regen
      # plan can continue to try to re-enable the puppet service and agent.
      # Otherwise, the plan would bail immediately since it is looking for
      # errored_certnames, which we use to handle known errors and continue
      # with the plan.
      raise EnterpriseTaskHelper::Error.new(
        "Unhandled exception: #{e}",
        'puppetlabs.puppetserver-ca-clean-api/unhandled-exception',
        'type' => e.class,
        'backtrace' => e.backtrace,
        'full_message' => e.full_message,
        'errored_certnames' => certnames,
      )
    end
  end
end

PuppetserverCACleanAPI.run if __FILE__ == $PROGRAM_NAME
