#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require_relative '../files/api_helper.rb'
require 'json'
require 'time'

class PuppetserverCASignAPI < EnterpriseTaskHelper

  def task(certname:, allow_subject_alt_names: false, csr_timeout: 300, dns_alt_names: nil, **_kwargs)
    begin # rubocop:disable Style/RedundantBegin (it is, but not worth changing now)
      options = { 'open_timeout' => 30, 'read_timeout' => 30 }
      headers = { 'Content-Type' => 'text/pson' }
      # Since we don't specify 'server', it defaults to Puppet[:certname]. Since this only runs on the primary, we're good.
      api = APIHelper.new(8140, 'puppet-ca/v1', options: options, headers: headers)

      # The original list of certnames passed in
      certnames = [certname].flatten
      # In case all the steps in the plan failed before we got here
      return if certnames.empty?
      # The certnames that did not error on previous steps
      current_certnames = certnames
      # Collection of error messages
      errors = []
      # Collection of certnames that failed in any step in the process.
      # agent_cert_regen will use this to filter out nodes that failed
      # in this task.
      errored_certnames = []

      # This will carry the data from the response of the certificate_statuses endpoing
      cert_status_data = nil

      # Get status of all certs/CSRs, wait for all nodes to appear
      start = Time.now
      all_present = false
      missing = []
      while !all_present && Time.now < start + csr_timeout
        # This endpoint needs a key that it completely ignores for some reason.
        # https://puppet.com/docs/puppet/7/server/http_certificate_status.html#search
        begin
          response = api.request('certificate_statuses/x', :get, retry_on_invalid_code: true, valid_codes: 200)
          cert_status_data = JSON.parse(response)
        rescue StandardError => e
          raise EnterpriseTaskHelper::Error.new(
            "#{e.class} exception when polling the CA endpoint for certificate statuses: #{e.message}",
            'puppetlabs.puppetserver-ca-sign-api/certificate-statuses-error',
            'errored_certnames' => current_certnames,
            'exception' => e,
          )
        end
        missing = current_certnames - cert_status_data.map { |d| d['name'] }
        all_present = missing.empty?
        sleep(1) unless all_present
      end
      if !all_present
        errors << [
          'No CSR detected for the following nodes:',
          missing,
        ]
        current_certnames -= missing
      end

      # If we are disallowing alt names, ensure none of the CSRs or autosigned certs contain alt names.
      # All of them have their certname in the alt names list, so ignore that one.
      unless current_certnames.empty?
        artifacts_with_alt_names = cert_status_data.select { |d| current_certnames.include?(d['name']) && d['dns_alt_names'].length > 1 }
        if !allow_subject_alt_names && !artifacts_with_alt_names.empty?
          errors << [
            'The following nodes contain subject alt names and allow_subject_alt_names == false:',
            artifacts_with_alt_names.map { |a| [a['name'], a['dns_alt_names']] },
          ]
          errored_certnames += artifacts_with_alt_names.map { |a| a['name'] }
          current_certnames -= errored_certnames
        end
      end

      # Sign nodes with state == 'requested'
      unless current_certnames.empty?
        certnames_to_sign = cert_status_data.select { |d| current_certnames.include?(d['name']) && d['state'] == 'requested' }.map { |d| d['name'] }
        begin
          body = "{\"certnames\":#{certnames_to_sign}}"
          headers = { 'Content-Type' => 'application/json' }
          # Signing takes roughly 8.5 seconds per 100 nodes. Use 15 seconds to be safe, min 30 seconds.
          timeout = [(certnames_to_sign.count / 100.0).ceil * 15, 30].max
          options = { 'open_timeout' => timeout, 'read_timeout' => timeout }
          response = api.request('sign', :post, body: body, headers: headers, options: options, valid_codes: 200)
          sign_data = JSON.parse(response)
          no_csr = sign_data['no-csr']
          signing_errors = sign_data['signing-errors']
          unless no_csr.empty?
            errors << [
              'The following nodes did not have a valid CSR when attempting to sign:',
              no_csr,
            ]
            errored_certnames += no_csr
          end
          unless signing_errors.empty?
            errors << [
              'Errors encountered when attempting to sign the CSR for the following certnames:',
              signing_errors,
            ]
            errored_certnames += signing_errors
          end
        rescue StandardError => e
          errors << [
            "#{e.class} exception signing certs: #{e.message}",
            e,
          ]
          errored_certnames += certnames_to_sign
        end
        current_certnames -= errored_certnames
      end

      # Verify certs are signed, and check for dns_alt_names if we passed those in
      unless current_certnames.empty?
        begin
          response = api.request('certificate_statuses/x', :get, retry_on_invalid_code: true, valid_codes: 200)
          cert_status_data = JSON.parse(response)
        rescue StandardError => e
          raise EnterpriseTaskHelper::Error.new(
            "#{e.class} exception when polling the CA endpoint for certificate statuses: #{e.message}",
            'puppetlabs.puppetserver-ca-sign-api/certificate-statuses-error',
            'errored_certnames' => current_certnames,
            'exception' => e,
          )
        end
        not_signed = cert_status_data.select { |d| current_certnames.include?(d['name']) && d['state'] != 'signed' }.map { |d| d['name'] }
        unless not_signed.empty?
          errors << [
            'The following nodes unexpectedly do not contain a signed cert after attempting to sign the CSR:',
            not_signed,
          ]
          errored_certnames += not_signed
          current_certnames -= errored_certnames
        end
      end

      unless dns_alt_names.nil? || dns_alt_names.empty? || current_certnames.empty?
        current_certnames.each do |c|
          cert_status_data_for_node = cert_status_data.find { |d| d['name'] == c }
          alt_names = cert_status_data_for_node['dns_alt_names']
          expected_alt_names = ["DNS:#{c}"] + dns_alt_names.split(',').map { |dan| "DNS:#{dan}" }
          next if alt_names.sort == expected_alt_names.sort
          errors << [
            "#{c} did not have the expected list of alt names",
            "Expected alt names: #{expected_alt_names}",
            "Actual alt names: #{alt_names}",
          ]
          errored_certnames += [c]
          current_certnames -= [c]
        end
      end

      if errors.empty?
        # current_certnames == certnames in this case
        { signed: current_certnames }.to_json
      else
        raise EnterpriseTaskHelper::Error.new(
          "One or more errors occurred attempting to sign the given certname(s): #{errors.join("\n")}",
          'puppetlabs.puppetserver-ca-sign-api/task-errors',
          'errored_certnames' => errored_certnames,
          'errors' => errors,
          'signed' => current_certnames,
        )
      end
    rescue EnterpriseTaskHelper::Error
      raise
    rescue StandardError => e # Catching StandardError rather than Exception here and above so as not to catch signals intended to manage the process
      # The reason we repackage the exception here is so that the agent_cert_regen
      # plan can continue to try to re-enable the puppet service and agent.
      # Otherwise, the plan would bail immediately since it is looking for
      # errored_certnames, which we use to handle known errors and continue
      # with the plan.
      raise EnterpriseTaskHelper::Error.new(
        "Unhandled exception: #{e}",
        'puppetlabs.puppetserver-ca-sign-api/unhandled-exception',
        'type' => e.class,
        'backtrace' => e.backtrace,
        'full_message' => e.full_message,
        'errored_certnames' => certnames,
      )
    end
  end
end

PuppetserverCASignAPI.run if __FILE__ == $PROGRAM_NAME
