class puppet_enterprise::profile::plan_runner(
  Boolean $active                    = true,
  $user                              = "pe-bolt-server",
  $group                             = "pe-bolt-server",
  $orch_host                         = $puppet_enterprise::puppet_master_host,
  Integer $orch_port                 = $puppet_enterprise::orchestrator_port,
  $ssl_ca_cert                       = $puppet_enterprise::params::localcacert,
  $ssl_crl_path                      = $puppet_enterprise::params::hostcrl,
  Array[String]  $puppetdb_host      = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_port      = $puppet_enterprise::puppetdb_ports_array,
  Enum['stopped', 'running'] $ensure = running,
  Boolean $enable                    = true,
  Enum['fatal', 'error', 'warn', 'info', 'debug', 'trace'] $log_level = 'info',
) inherits puppet_enterprise {
  $container = 'plan-runner'
  $pe_container = "pe-${container}"
  $confdir = "/etc/puppetlabs/${pe_container}/conf.d"
  $confpath = "${confdir}/pe-plan-runner.conf"
  $ssl_dir = "/etc/puppetlabs/${pe_container}/ssl"

  $client_certname = $orch_host
  $_ensure = $active ? {
    true  => $ensure,
    false => 'stopped',
  }

  $_enable = $active ? {
    true  => $enable,
    false => false,
  }

  service { $pe_container:
    ensure     => $_ensure,
    enable     => $_enable,
    hasrestart => true,
  }

  if $active {
    file { $confdir:
      ensure  => 'directory',
      owner   => $user,
      group   => $group,
      mode    => '0755',
    }

    file { $confpath:
      ensure  => 'present',
      owner   => $user,
      group   => $group,
      mode    => '0644',
      require => File[$confdir],
      before  => Service[$pe_container],
    }

    include puppet_enterprise::packages

    Package <| tag == 'pe-bolt-server-packages' |> {
      before => [Puppet_enterprise::Certs[$pe_container],
                 File[$confdir],
                 Service[$pe_container]]
    }

    # Create the SSL Certs for Plan Runner
    puppet_enterprise::certs { $pe_container:
      certname => $client_certname,
      cert_dir => $ssl_dir,
      owner => $user,
      group => $group,
      container => $container,
      package_name => 'pe-bolt-server',
      full_restart => [],
      make_pk8_cert => false,
    }

    $first_puppetdb_host = $puppetdb_host[0]
    $first_puppetdb_port = pe_any2array($puppetdb_port)[0]
    $puppetdb_url = "https://${first_puppetdb_host}:${first_puppetdb_port}"
    $orchestrator_url = "https://${orch_host}:${orch_port}/orchestrator/v1"

    pe_hocon_setting { 'pe-plan-runner.puppetdb.server_urls':
      path => $confpath,
      ensure => present,
      type => 'array',
      value => $puppetdb_url
    }

    pe_hocon_setting { 'pe-plan-runner.certs.ssl-key':
      path    => $confpath,
      value   => "/etc/puppetlabs/${pe_container}/ssl/${client_certname}.private_key.pem",
    }

    pe_hocon_setting { 'pe-plan-runner.certs.ssl-cert':
      path    => $confpath,
      value   => "/etc/puppetlabs/${pe_container}/ssl/${client_certname}.cert.pem",
    }

    pe_hocon_setting { 'pe-plan-runner.certs.ssl-ca-cert':
      path    => $confpath,
      value   => $ssl_ca_cert,
    }

    pe_hocon_setting { 'pe-plan-runner.certs.ssl-crl-path':
      path    => $confpath,
      value   => $ssl_crl_path,
    }

    pe_hocon_setting { 'pe-plan-runner.orchestrator-url':
      path    => $confpath,
      value   => $orchestrator_url,
    }

    pe_hocon_setting { 'pe-plan-runner.log-level':
      path    => $confpath,
      value   => $log_level,
    }
  }
}
