require 'puppet_x/util/orchestrator/connection'
require 'puppet_x/util/orchestrator/plan_runner'
require 'puppet_x/util/orchestrator/puppet_runner'
require 'puppet_x/util/orchestrator/task_runner'

module PuppetX
  module Util

    # These class methods represent the API used by other puppet-infra actions
    # for interacting with the Orchestrator service.
    #
    # It allows you to check that a node is in the Orchestrator inventory, or to run
    # puppet, a task or a plan on some set of nodes via the Orchestrator.
    #
    # Each method expects to be given a service config Hash for the local Orchestrator
    # service, such as {PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator')}
    # would return (from /etc/puppetlabs/client-tools/services.conf), and an RBAC token,
    # such as {PuppetX::Util::RBAC.load_token()} would return.
    class Orchestrator

      # Check whether a node is connected in the Orchestrator's inventory and
      # reachable via PCP.
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned by
      # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
      # @param rbac_token [String] RBAC token, can be read via PuppetX::Util::RBAC.load_token
      # @param node_certname [String] certname of the node to test.
      # @return [Boolean] true if connected.
      def self.node_in_orch_inventory(orch_service, rbac_token, node_certname)
        path = "/v1/inventory/#{node_certname}"
        body = Connection.get(orch_service, rbac_token, path)
        return body['connected']
      end

      # Given a list of node certnames, query the Orchestrator inventory endpoint once
      # for a list of the connectivity status.
      #
      #   [
      #     {
      #       'name' => 'node.one',
      #       'connected' => true,
      #       'broker' => 'broker.one',
      #       'timestamp' => ...,
      #     },
      #     {
      #       'name' => 'node.one',
      #       'connected' => false,
      #     },
      #     ...
      #   ]
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned by
      # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
      # @param rbac_token [String] RBAC token, can be read via PuppetX::Util::RBAC.load_token
      # @param node_certnames [Array<String>] List of node certs to check.
      # @return [Array<Hash>] Returns an array of hashes with name, connected, broker
      # and timestamp information (POST /orchestrator/v1/inventory body).
      def self.get_pcp_broker_connectivity(orch_service, rbac_token, node_certnames)
        return [] if node_certnames.empty?

        url = "#{orch_service[:url]}/v1/inventory"
        connection = Connection.new(orch_service, rbac_token)
        payload = { nodes: node_certnames }
        uri = URI(url)

        response = connection.orch_connection.post(uri, payload.to_json, headers: connection.orch_request_headers)
        connection.check_orch_response(response, 200)

        JSON.parse(response.body)['items']
      end

      # Return only given targets whose pcp-brokers do not match the given
      # broker cert. Does not return nodes that are not connected at all.
      # We don't know if those targets are misconfigured, just that they aren't
      # connected. That condition is caught by test_connection calls in the plans
      # themselves.
      #
      # See get_pcp_broker_connectivity()...
      # @param broker_cert [String] the certname for the primary node that we
      # expect to be the listed broker for the connections.
      # @return [Array<Hash>] just the set from get_pcp_broker_connectivity() that are
      # not connected to the given master_certname pcp-broker.
      def self.get_misconfigured_pcp_broker_connections(orch_service, rbac_token, node_certnames, master_certname)
        connections = get_pcp_broker_connectivity(orch_service, rbac_token, node_certnames)
        connections.select do |c|
          c['connected'] && c['broker'] !~ /#{master_certname}/
        end
      end

      # Run puppet on a remote node via the Orchestrator.
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned by
      # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
      # @param display_scope [String] Node(s) identifier that gets printed with status messages.
      # @param scope [Hash] See https://puppet.com/docs/pe/2019.2/orchestrator_api_commands_endpoint.html#scope
      # @param rbac_token [String] RBAC token, can be read via PuppetX::Util::RBAC.load_token.
      # @param allow_empty [Boolean] if true, an empty scope (a puppetdb query
      # that nets no target nodes, for example) will not raise an error.
      # @param log_context [PuppetX::Puppetlabs::Meep::Util::InfraLogContext]
      # utility class providing contextual state for logfile name generation.
      # @return [String] final job state.
      def self.run_puppet(orch_service, scope:, rbac_token:, log_context:, display_scope: nil, allow_empty: false)
        display_scope = build_display_scope(display_scope, scope)
        impl = PuppetRunner.new(orch_service, rbac_token, log_context)
        impl.run(
          display_scope: display_scope,
          scope: scope,
          allow_empty: allow_empty,
        )
      end

      # Run a task on a remote node via Orchestrator.
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned by
      # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
      # @param display_scope [String] Node(s) identifier that gets printed with status messages.
      # @param scope [Hash] See https://puppet.com/docs/pe/2019.2/orchestrator_api_commands_endpoint.html#scope
      # @param rbac_token [String] RBAC token, can be read via PuppetX::Util::RBAC.load_token.
      # @param task [String] Task name.
      # @param params [Hash] Parameter names and values for the given task.
      # @param environment [String] Environment that contains the task, default to 'production'.
      # @param allow_empty [Boolean] if true, an empty scope (a puppetdb query
      # that nets no target nodes, for example) will not raise an error.
      # @param log_context [PuppetX::Puppetlabs::Meep::Util::InfraLogContext]
      # utility class providing contextual state for logfile name generation.
      # @return [Array[Hash]] An array of Hashes detailing the results of task
      # execution on each target node. (See {PuppetX::Util::Orchestrator::Job#node_items}
      # for details)
      def self.run_task(orch_service, scope:, rbac_token:, task:, params:, display_scope: nil, environment: 'production', allow_empty: false, log_context:)
        display_scope = build_display_scope(display_scope, scope)
        impl = TaskRunner.new(orch_service, rbac_token, log_context)
        impl.run(
          display_scope: display_scope,
          scope: scope,
          task: task,
          params: params,
          environment: environment,
          allow_empty: allow_empty,
        )
      end

      # Run a plan on a remote node via Orchestrator.
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned by
      # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
      # @param rbac_token [String] RBAC token, can be read via PuppetX::Util::RBAC.load_token.
      # @param plan [String] Plan name.
      # @param params [Hash] Parameter names and values for the given plan.
      # @param environment [String] Environment that contains the plan, default to 'production'.
      # @param log [String] Full path to the logfile used to capture plan execution details.
      # @return [String] final job state.
      def self.run_plan(orch_service, plan:, rbac_token:, description:, params: {}, environment: 'production', log:)
        impl = PlanRunner.new(orch_service, rbac_token, log)
        impl.run(
          plan: plan,
          description: description,
          params: params,
          environment: environment,
        )
      end

      # Return the metadata Hash that describes the given plan and its parameters.
      #
      # @param orch_service [Hash] Service config hash from services.conf, returned
      # by PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'),
      # for example.
      # @param rbac_token [String] RBAC token, can be read via
      # PuppetX::Util::RBAC.load_token.
      # @param plan [String] Plan name.
      # @return [Hash] of plan metadata (name:, description:, parameters:)
      def self.get_plan_metadata(orch_service, plan:, rbac_token:)
        path_elements = plan.split('::')
        plan_path = path_elements.size == 1 ?
          "#{path_elements.first}/init" :
          path_elements.join('/')

        path = "/v1/plans/#{plan_path}"
        body = Connection.get(orch_service, rbac_token, path)
        return body['metadata'] || {}
      end

      # Helper for general case of extracting display_scope from the orchestrator scope
      # map nodes array.
      def self.build_display_scope(display_scope, scope)
        case display_scope
        when nil
          s = scope.transform_keys { |k| k.to_s }
          raise(ArgumentError, _("Orchestrator.run_puppet() no display_scope set and scope does not contain nodes: '%{scope}'") % { scope: scope }) if s['nodes'].nil?
          s['nodes'].join(', ')
        else
          display_scope
        end
      end
    end
  end
end
