# Run via the orchestrator.  Unlike other plans that cause a cert to get
# regenerated, this one will automatically pick up any subject alt names already
# defined for the node and use them when regenerating the certificate.
#
# When all=true, this does a PuppetDB lookup for all nodes with the 'master'
# profile applied, and filters out any already-converted compilers (PE Compilers
# with PuppetDB), primary, and replica.
#
# In these steps, “database node” refers either to the primary or a standalone
# postgres node, if it exists.
#
# Steps this plan takes:
# 1. Check if PE Master node group contains the trusted.extensions.pp_auth_role
#    = pe_compiler rule. If not there we try to add, but if conflicting rules
#    exist plan fails and customer needs to manually add the rule.
# 2. Verify that one of the puppetserver conf files in
#    /etc/puppetlabs/puppetserver/conf.d has
#    certificate-authority.allow-authorization-extensions set to true.  This is
#    needed in order to sign a cert with a pp_auth_role extension.
# 3. Verify all compilers to be converted have been upgraded to match the
#    primary’s version.
# 4. Add “puppet_enterprise::profile::database::private_temp_puppetdb_hosts”:
#    [“compiler.certname”] to pe.conf on the primary, and run puppet on the postgres
#    node.  This puts the soon-to-be compiler in the postgres allowlist for the
# 5. Run puppet on the database node.
# 6. Disable the agent on the primary, database (if separate), and target nodes
#    with puppet agent --disable.
# 7. Stop pe-puppetserver service on the targets.
# 8. Run the agent_cert_regen plan on each target (see that plan's help for
#    details), with dns_alt_names set to the current subject alt names for the
#    node, and {‘pp_auth_role’ => ‘pe_compiler’} for extension_requests.
# 9. Unpin the targets from the PE Master node group.
# 10. Remove “puppet_enterprise::profile::database::private_temp_puppetdb_hosts”
#     from pe.conf.
# 11. Re-enable the agent on the primary, database (if separate), and target nodes
#     with puppet agent --enable.
# 12. Run puppet on the database node.
# 13. Run puppet on the primary and all compilers.
#
# @api private
plan enterprise_tasks::convert_legacy_compiler(
  Optional[TargetSpec] $compiler = undef,
  Optional[Boolean] $all         = false,
  Optional[TargetSpec] $primary  = 'localhost',
  Optional[Boolean] $force       = false,
) {
  unless ($compiler or $all) {
    fail_plan('ERROR: Missing parameter. Convert_legacy_compiler plan requires a value for either compiler or all', 'convert_legacy_compiler')
  }

  $constants = constants()
  enterprise_tasks::test_connection($primary)

  $result = run_task(enterprise_tasks::check_pp_auth_role_rule, $primary, _catch_errors => true)
  if !$result.ok() {
    enterprise_tasks::message('convert_legacy_compiler',$result.first.error.msg)
    fail_plan('Error adding pp_auth_role rule')
  }

  enterprise_tasks::message('convert_legacy_compiler','Checking if authorization-extensions are allowed')
  $ca_setting = run_task(enterprise_tasks::get_conf_values, $primary,
    path => $constants['puppetserver_conf_dir'],
    keys => 'certificate-authority',
  ).first.value['certificate-authority']
  if !$ca_setting or !$ca_setting['allow-authorization-extensions'] {
    fail_plan("ERROR: 'allow-authorization-extensions' parameter is not set in ${constants()['puppetserver_conf_dir']}/ca.conf.", 'convert_legacy_compiler')
  }

  $primary_certname = strip(run_command('/opt/puppetlabs/bin/puppet config print certname --section agent', $primary).first.value['stdout'])
  $all_puppetservers = enterprise_tasks::get_nodes_with_profile('master')
  $replicas = enterprise_tasks::get_nodes_with_profile('primary_master_replica')
  $all_non_primary_puppetservers = $all_puppetservers.filter |$host| {
    $host != $primary_certname and !($host in $replicas)
  }
  $infra_targets = get_targets($all_non_primary_puppetservers)

  $primary_node_facts = run_plan('facts', $primary, '_catch_errors' => true)
  if $primary_node_facts =~ Error {
    if $primary_node_facts.details()['result_set'].first.value['aio_agent_version'] {
      $primary_agent_version = $primary_node_facts.details()['result_set'].first.value['aio_agent_version']
    } else {
      enterprise_tasks::message('convert_legacy_compiler', "${primary_node_facts.kind()}:${primary_node_facts.msg()}")
      fail_plan('convert_legacy_compiler', "Failed to retrieve os facts from ${node}.")
    }
  } else {
    $primary_agent_version = $primary_node_facts.first.value['aio_agent_version']
  }

  enterprise_tasks::message('convert_legacy_compiler','Checking for compilers in need of upgrade')
  enterprise_tasks::test_connection($infra_targets)
  $infra_targets.each |$node| {
    $infra_node_facts = run_plan('facts', $node, '_catch_errors' => true)
    if $infra_node_facts =~ Error {
      if $infra_node_facts.details()['result_set'].first.value['aio_agent_version'] {
        $infra_agent_version = $infra_node_facts.details()['result_set'].first.value['aio_agent_version']
      } else {
        enterprise_tasks::message('convert_legacy_compiler', "${infra_node_facts.kind()}:${infra_node_facts.msg()}")
        fail_plan('convert_legacy_compiler', "Failed to retrieve os facts from ${node}.")
      }
    } else {
      $infra_agent_version = $infra_node_facts.first.value['aio_agent_version']
    }

    if $primary_agent_version != $infra_agent_version {
      $node.set_var('upgrade_needed', true)
    } else {
      $node.set_var('upgrade_needed', false)
    }
  }

  $nodes_to_upgrade = $infra_targets.filter |$node| { $node.vars['upgrade_needed'] }
  if !empty($nodes_to_upgrade) {
    fail_plan("ERROR: Found non-upgraded primary infrastructure. To convert your compilers, your infrastructure must be fully upgraded. Please upgrade these nodes: ${nodes_to_upgrade}", 'convert_legacy_compiler')
  }

  if ($all) {
    $compilers_and_primary = enterprise_tasks::get_nodes_with_profile('master')
    $pe_compilers = enterprise_tasks::get_nodes_with_profile('pe_compiler')

    $_primary_certname = run_command("${constants()['puppet_bin']} config print certname --section agent", $primary).first.value['stdout'][0,-2]
    $pe_comp_and_primary = $pe_compilers + $_primary_certname

    # this filters the actual primary's hostname and PE Compilers out of the list of nodes with the 'master' role
    $targets = $compilers_and_primary.filter |$host| {
      !($host in $pe_comp_and_primary) and !($host in $replicas)
    }
  } elsif type($compiler) =~ Type[Array] {
    $targets = $compiler
  } else {
    $targets = split($compiler, ',')
  }
  enterprise_tasks::test_connection($targets)
  enterprise_tasks::verify_node($primary, 'primary', $force)

  $dbnodes = enterprise_tasks::get_nodes_with_profile('database')
  # Ignore replicas
  $postgres_nodes = $dbnodes.filter |$node| { !enterprise_tasks::node_has_profile($node, 'primary_master_replica') }
  $postgres_certname = $postgres_nodes[0]
  # If postgres is on the primary, and primary is 'localhost', we want $postgres to be 'localhost' as well
  # so the primary doesn't try SSHing into itself
  $postgres = $postgres_certname ? {
    $primary_certname => $primary,
    default => $postgres_certname
  }

  enterprise_tasks::test_connection($postgres)
  enterprise_tasks::message('convert_legacy_compiler',"Converting the following legacy compilers to compilers with the PuppetDB service: ${targets}")

  run_task(enterprise_tasks::add_modify_conf_keys, $primary,
    file    => $constants['pe_conf'],
    hash    => { $constants['temp_allowlist_key'] => $targets },
  )
  run_task(enterprise_tasks::run_puppet, $postgres,
    max_timeout => 256,
  )

  $targets.each |$target| {
    enterprise_tasks::with_agent_disabled([$target, $primary, $postgres].flatten.unique) || {
      enterprise_tasks::message('convert_legacy_compiler',"Converting ${target}")

      # Node is verified to be a compiler in agent_cert_regen, which also performs the puppet
      # run that turns it into a new compiler+PDB
      run_task(enterprise_tasks::pe_services, $target,
        role => 'legacy_compiler',
        state => 'stopped',
        include_puppet => false,
      )

      $current_alt_names = run_task(enterprise_tasks::get_subject_alt_names, $target).first.value['altnames']
      if $current_alt_names {
        $dns_alt_names = join($current_alt_names, ',')
      } else {
        $dns_alt_names = undef
      }
      run_plan(enterprise_tasks::agent_cert_regen, agent => $target,
        primary             => $primary,
        node_type           => 'legacy_compiler',
        extension_requests  => { 'pp_auth_role' => 'pe_compiler' },
        dns_alt_names       => $dns_alt_names,
        force               => $force,
      )
      run_task(enterprise_tasks::run_puppet, $target,
        max_timeout => 256,
      )

      run_command("/opt/puppetlabs/bin/puppet resource pe_node_group \"PE Master\" unpinned=\"${target}\"", $primary)
      run_task(enterprise_tasks::remove_conf_keys, $primary,
        file    => $constants['pe_conf'],
        keys    => $constants['temp_allowlist_key'],
      )
    }
    enterprise_tasks::verify_node($target, 'pe_compiler', $force)
    enterprise_tasks::message('convert_legacy_compiler',"Successfully converted node ${target}")
  }

  run_task(enterprise_tasks::run_puppet, $postgres,
    max_timeout => 256,
  )

  # Run on MoM to update services.conf, and all the compilers to update services.conf and crl.pem
  $all_puppetservers.each |$node| {
    # Allow for $primary == 'localhost'
    $node_to_run = $node ? {
      $primary_certname => $primary,
      default => $node
    }
    # We already ran on the postgres node if postgres == primary.
    # Now run puppet on rest of the infrastructure nodes
    unless $node == $postgres_certname {
      run_task(enterprise_tasks::run_puppet, $node_to_run,
        max_timeout => 256,
      )
    }
  }
}
