# Enable or disable the pe-infra-assistant.
#
# On failure, should return a standard error hash with msg, kind and details.
#
# @param primary [TargetSpec] The certname of the primary.
# @param enable [Boolean] True to enable the pe-infra-assistant.
# @param force [Boolean] Can be set true to skip verifying that the given
#   primary certificate name has the Primary role. There shouldn't be a
#   good reason to use this, since inability to verify would be indicative
#   of a failure in puppetdb, which should probably be addressed before
#   attempting to modify infra assistant service status. But verify_node expects a
#   force value.
# @param timeout [Integer] Number of seconds to wait for infra assistant service status before failing.
# @param tos_accepted_by [String] The user id of the user who accepted the ToS. Must be set when enabling the service.
# @param enable_mcp_server [Boolean] True to enable the MCP (Model Context Protocol) server for enhanced AI capabilities.
plan enterprise_tasks::toggle_infra_assistant(
  TargetSpec $primary,
  Boolean $enable  = true,
  Boolean $force   = false,
  Integer $timeout = 300,
  Optional[String] $tos_accepted_by = undef,
  Boolean $enable_mcp_server = false,
) {
  if(!$tos_accepted_by and $enable) {
    fail_plan('The tos_accepted_by parameter must be set when enabling the Infra Assistant service.')
  }

  $plan = 'enterprise_tasks::toggle_infra_assistant'
  $service_name = 'pe-infra-assistant'

  $primary_target = get_targets($primary)[0]

  enterprise_tasks::test_connection($primary_target)
  enterprise_tasks::verify_node($primary_target, 'primary', $force)

  if $enable {
    $terms_dir_path = '/etc/puppetlabs/infra-assistant'
    $terms_file_path = "${terms_dir_path}/tos-accepted.json"
    $timestamp = strftime(Timestamp.new, '%Y-%m-%d_%H-%M-%S')

    $terms_content = @("END":json)
    {
      "tos_accepted_by": "${tos_accepted_by}",
      "timestamp": "${timestamp}"
    }
    | END

    run_plan('facts', 'targets' => $primary_target)

    apply($primary_target) {
      file { $terms_file_path:
        ensure  => file,
        content => $terms_content,
        mode    => '0600',
        owner   => 'pe-infra-assistant',
        group   => 'pe-infra-assistant',
      }
    }
  }

  # Before doing anything, stop the service in case it was running
  run_task(service, $primary_target,
    action  => 'stop',
    name    => $service_name,
  )

  run_task('enterprise_tasks::update_node_group', $primary_target,
    'group_name'       => 'PE Master',
    'class_parameters' => {
      'puppet_enterprise::profile::master' => {
        'enable_infra_assistant'            => $enable,
        'infra_assistant_enable_mcp_server' => $enable_mcp_server,
      },
    },
  )

  run_task('enterprise_tasks::update_node_group', $primary_target,
    'group_name'       => 'PE Console',
    'class_parameters' => {
      'puppet_enterprise::profile::console' => {
          'infra_assistant_proxy' => $enable,
          'mcp_server_proxy'      => $enable_mcp_server,
      },
    },
  )

  enterprise_tasks::with_agent_disabled($primary_target) |$agent| {
    enterprise_tasks::message($plan, 'Run Puppet to update service status.')
    run_task('enterprise_tasks::run_puppet', $agent)
    enterprise_tasks::message($plan, 'Run Puppet again to update services.conf.')
    run_task('enterprise_tasks::run_puppet', $agent)
  }

  if $enable {
    run_task('enterprise_tasks::wait_for_service', $primary_target,
      'host'    => $primary_target.facts()['fqdn'],
      'service' => $service_name,
      'timeout' => $timeout,
    )
  }

  $plan_result = { 'success' => true }
  return($plan_result)
}
