#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require_relative '../lib/enterprise_tasks/puppet_helper.rb'
require 'puppet'
require 'json'
require 'open3'

class PEServices < EnterpriseTaskHelper
  include EnterpriseTasks::PuppetHelper

  PATCHING_SERVICE = 'pe-patching-service'
  INFRA_ASSISTANT = 'pe-infra-assistant'
  WORKFLOW_SERVICE = 'pe-workflow-service'

  # The patching service, infra-assistant, and workflow service are conditionally enabled, and should not
  # be started if currently disabled.
  #
  # @return true if service is the pe-patching-service, the infra-assistant, or the workflow-service,
  # desire_state is running and service is currently disabled.
  def ignore_disabled_service(service, desired_state)
    if [PATCHING_SERVICE, INFRA_ASSISTANT, WORKFLOW_SERVICE].include?(service) && desired_state == 'running'
      cmd = [puppet_bin, 'resource', 'service', service]
      output, _stderr, _status = Open3.capture3(*cmd)
      output.strip.match(/enable\s+=>\s+'false'/)
    else
      false
    end
  end

  # Default to only stopping the puppet agent service
  def task(role: '', state: 'stopped', include_puppet: true, include_pxp_agent: false, **_kwargs)
    certname_cmd = [puppet_bin, 'config', 'print', 'certname']
    certname = Open3.capture2e(*certname_cmd)[0].strip

    services = case role
    when 'primary'
      ['pe-ace-server', 'pe-bolt-server', 'pe-console-services', 'pe-nginx', 'pe-orchestration-services', 'pe-puppetserver', 'pe-puppetdb', 'pe-postgresql', 'pe-host-action-collector', PATCHING_SERVICE, INFRA_ASSISTANT, WORKFLOW_SERVICE]
    when 'replica'
      ['pe-console-services', 'pe-puppetdb', 'pe-puppetserver', 'pe-postgresql']
    when 'pe_compiler'
      ['pe-puppetserver', 'pe-puppetdb']
    when 'legacy_compiler'
      ['pe-puppetserver']
    when 'external_postgres'
      ['pe-postgresql']
    else
      []
    end
    services = ['puppet'] + services if include_puppet
    services = ['pxp-agent'] + services if include_pxp_agent
    all_output = []
    services.each do |service|
      next if ignore_disabled_service(service, state)
      cmd = [puppet_bin, 'resource', 'service', service, "ensure=#{state}"]
      output, _stderr, _status = Open3.capture3(*cmd)
      if !output.strip.match(/ensure\s+=>\s+'#{state}'/)
        raise EnterpriseTaskHelper::Error.new("Unable to change service #{service} status on host with certname #{certname}", 'puppetlabs.pe-services/service-status-change-failed', 'output' => output)
      end
      all_output << output
    end
    result = { _output: all_output.join("\n") }
    result.to_json
  end
end

PEServices.run if __FILE__ == $PROGRAM_NAME
