class puppet_enterprise::trapperkeeper::workflow (
  String $puppetdb_url,
  String $container                      = 'workflow-service',
  String $service_name                   = 'pe-workflow-service',
  String $user                           = "pe-${container}",
  String $group                          = "pe-${container}",
  String $database_host                  = $puppet_enterprise::workflow_database_host,
  Puppet_enterprise::Port $database_port = $puppet_enterprise::database_port,
  String $orchestrator_url               = "${puppet_enterprise::orchestrator_url}${puppet_enterprise::orchestrator_url_prefix}",
  String $classifier_url                 = "https://${puppet_enterprise::puppet_master_host}:${puppet_enterprise::params::console_services_api_ssl_listen_port}${puppet_enterprise::classifier_url_prefix}",
  String $database_name                  = $puppet_enterprise::workflow_database_name,
  String $database_user                  = $puppet_enterprise::workflow_service_regular_db_user,
  String $database_migration_user        = $puppet_enterprise::workflow_service_migration_db_user,
  String $database_properties            = $puppet_enterprise::database_properties,
  Optional[String] $database_password    = undef,
  String $rbac_url                       = $puppet_enterprise::rbac_url,
  String $activity_url                   = $puppet_enterprise::activity_url,
  String $workflow_callback_url          = $puppet_enterprise::workflow_service_url,
) {
  $confdir      = "/etc/puppetlabs/${container}/conf.d"
  $service_conf = "${confdir}/pe-workflow-service.conf"

  ### Trapperkeeper bootstrap cfg entries ###
  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}: pe-workflow-service":
    namespace => 'puppetlabs.pe-workflow-service.service',
    service   => 'pe-workflow-service',
    container => $container,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}: remote-rbac-consumer-service":
    namespace => 'puppetlabs.rbac-client.services.rbac',
    service   => 'remote-rbac-consumer-service',
    container => $container,
  }

  ### Hocon configuration ###
  $common_hocon_settings = {
    ensure => present,
    notify => Service["pe-${container}"],
  }

  file { $service_conf:
    ensure => file,
    owner  => $user,
    group  => $group,
    mode   => '0640',
  }

  file { "${confdir}/secrets":
    ensure => directory,
    owner  => $user,
    group  => $group,
    mode   => '0640',
  }

  pe_hocon_setting { "${container}.puppetdb-url":
    path    => $service_conf,
    setting => 'pe-workflow-service.puppetdb-url',
    value   => $puppetdb_url,
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.orchestrator-url":
    path    => $service_conf,
    setting => 'pe-workflow-service.orchestrator-url',
    value   => $orchestrator_url,
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.classifier-url":
    path    => $service_conf,
    setting => 'pe-workflow-service.classifier-url',
    value   => $classifier_url,
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.rbac-consumer.api-url":
    path    => $service_conf,
    setting => 'rbac-consumer.api-url',
    value   => $rbac_url,
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.activity-consumer.api-url":
    path    => $service_conf,
    setting => 'activity-consumer.api-url',
    value   => $activity_url,
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.keypath":
    path    => $service_conf,
    setting => 'pe-workflow-service.keypath',
    value   => "${confdir}/secrets",
    *       => $common_hocon_settings,
  }

  pe_hocon_setting { "${container}.workflow-service-callback-url":
    path    => $service_conf,
    setting => 'pe-workflow-service.workflow-service-callback-url',
    value   => $workflow_callback_url,
    *       => $common_hocon_settings,
  }

  ### Database configuration ###
  puppet_enterprise::trapperkeeper::database_settings { $service_name:
    container           => $container,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    database_port       => Integer($database_port),
    database_properties => $database_properties,
    database_user       => $database_user,
    migration_user      => $database_migration_user,
    migration_password  => $database_password,
    group               => $group,
    user                => $user,
    notify              => Service["pe-${container}"],
    require             => Package["pe-${container}"],
  }
}
