#!/usr/bin/env ruby
# frozen_string_literal: true

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

# Unlock a datastore that you have locked
#
# This task unlocks a datastore that was previously locked by your session.
# It will fail if you don't hold the lock.

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Smart default: use candidate if supported, otherwise running
  datastore = session.task_params['datastore']
  if datastore.nil?
    datastore = session.supports_candidate? ? 'candidate' : 'running'
  end

  begin
    session.unlock(datastore)

    session.report_result({
                            status: 'success',
      message: "Successfully unlocked #{datastore} datastore",
      datastore: datastore
                          })
  rescue PuppetX::Puppetlabs::Netconf::NetconfError::InvalidValue => e
    # This typically means we don't hold the lock
    session.report_result({
                            status: 'error',
      message: "Cannot unlock #{datastore}: #{e.message}",
      datastore: datastore,
      note: 'You can only unlock a datastore that your session has locked'
                          })
  rescue => e
    session.report_result({
                            status: 'error',
      message: "Failed to unlock #{datastore}: #{e.message}",
      datastore: datastore
                          })
  end
end

puts result.to_json
