# This library is for using the internal version of Bolt that pe-installer lays down
require 'pty'
require 'json'

module PuppetX
  module Util
    class Bolt
      class BoltException < RuntimeError; end

      def self.env
        "BOLT_DISABLE_ANALYTICS=true BOLT_GEM=true"
      end

      def self.bolt_bin
        "/opt/puppetlabs/installer/bin/bolt"
      end

      def self.boltdir
        "--project /opt/puppetlabs/installer/share/Boltdir"
      end

      def self.installer_bolt_call(bolt_action)
        "#{env} #{bolt_bin} #{boltdir} #{bolt_action}"
      end

      # replacements is a hash where each value found in the output is replaced by the key. This is used primarily by the
      # puppet infra run action to rename enterprise_tasks plan names to appear as puppet infra run commands
      # run_bolt streams bolt output, and as such is visible to the user
      def self.run_bolt(bolt_action, replacements = {})
        output = ''
        process_status = nil
        Puppet.notice('Plan log captured at /var/log/puppetlabs/installer/bolt_info.log')
        PTY.spawn(installer_bolt_call(bolt_action)) do |read, write, pid|
          write.close
          while process_status == nil
            begin
              line = read.gets
              replacements.each do |k, v|
                line = line.gsub(v, k)
              end

              puts line
              output += line
            rescue EOFError, Errno::EIO
              # GNU/Linux raises EIO on read operation when pty slave is closed - see pty.rb docs
              # Ensure the child process finishes and then pass through to ensure block below
              # to get the child status
              nil
            rescue IO::WaitReadable, IO::WaitWritable
              retry
            ensure
              # Get Process::Status so we can report the exit code
              process_status ||= PTY.check(pid, raise=false)
            end
          end

          Process.waitall
          # Ensure we have captured the Process::Status
          process_status ||= PTY.check(pid, raise=false)
        end

        return process_status, output
      end

      # run_bolt_json does not stream bolt output, and as such is not visible to the user
      # Also, this method will return a JSON object instead of the command line output of a
      #   normal bolt call
      def self.run_bolt_json(bolt_action)
        bolt_output, bolt_stderr, bolt_status = Open3.capture3("#{installer_bolt_call(bolt_action)} --format json")
        raise _("Bolt command %{bolt_action} failed: %{bolt_stderr}" % {bolt_action: bolt_action, bolt_stderr: bolt_stderr}) if !bolt_status.exitstatus.zero?
        bolt_output_json = JSON.parse(bolt_output.strip)
        return bolt_status, bolt_output_json
      end

      def self.options_to_string(options)
        # This filters out flags that are not meant to be passed to Bolt
        valid_options = [:help, :host_key_check, :ssl_verify, :connect_timeout, :inventoryfile, :sudo_password, :run_as, :private_key, :username, :password, :tmpdir, :tty, :debug, :trace, :transport]
        options.collect do |k, v|
          next unless valid_options.include?(k)
          k = :user if k == :username
          if v.is_a?(TrueClass)
            "--#{k}"
          elsif v.is_a?(FalseClass)
            "--no-#{k}"
          else
            "--#{k}='#{v}'"
          end
        end.reject(&:nil?).join(' ')
      end
    end
  end
end
