require 'json'
require 'puppet_x/util/orchestrator/runner'
module PuppetX
  module Util
    class Orchestrator

      # Handles Orchestrator:
      #
      # /orchestrator/v1/command/task
      #
      # Results are primarily available through the Job class.
      class TaskRunner < Runner
        def initialize(service_config, rbac_token, log_context)
          super(service_config, rbac_token)
          @command_type = 'task'
          @log_context = log_context
        end

        # Start a task on a remote node via orchestrator.
        # @param scope [Hash] See https://puppet.com/docs/pe/2019.2/orchestrator_api_commands_endpoint.html#scope
        # @param task [String] Task name
        # @param params [Hash] Parameter names and values for the given task
        # @param environment [String] Environment that contains the task, default to 'production'
        # @return [Hash] The job ID and job URL
        def start_command(scope:, task:, params:, environment: 'production', **_kwargs)
          url = "#{service_config[:url]}/v1/command/task"
          payload = {:environment => environment, :scope => scope, :task => task, :params => params, timeout: ONE_YEAR_IN_SECONDS}
          uri = URI(url)
          response = orch_connection.post(uri, payload.to_json, headers: orch_request_headers)
          check_orch_response(response, 202)
          body = JSON.parse(response.body)
          id = body['job']['name']
          job_for(id)
        end

        # In the case of a task, the result contains output we may want to
        # return to the user.
        def _get_job_result(job)
          job.node_items
        end

        def _handle_job_failure(job)
          errors = job.task_errors
          errors.each do |node, error|
            Puppet.err("Task failed on #{node}: #{error}")
          end
          first_error = errors.first[1]
          raise PuppetX::Util::OrchestratorTaskFailedError.new(first_error['msg'], first_error['kind'], first_error['details']) #rubocop:disable GetText/DecorateFunctionMessage
        end

        def _save_log(job, args)
          if job.nil?
            Puppet.warning(_('Unable to log task; no job data retrieved.'))
            return
          end

          exact_log_context = log_context.description.nil? ?
            log_context.step("task_#{args[:task]}") :
            log_context
          task_log_file = exact_log_context.logfile
          begin
            job_body = job.job_body
            File.open(task_log_file, 'w') do |log|
              log.puts("#{job.started} [Notice]: Started #{job.type} #{job.action} on #{job_body['node_count']} nodes")
              job.node_items.sort_by { |i| i['start_timestamp'] }.each do |item|
                log.puts("#{item['start_timestamp']} [Notice]: Started node #{item['name']}")
                level = item['state'] == 'finished' ? 'Notice' : 'Error'
                log.puts("#{item['finish_timestamp']} [#{level}]: Finished node #{item['name']}, status: #{item['state']}, duration: #{item['duration']}, results:\n#{item['result'].pretty_inspect}")
              end
              log.puts("#{job.timestamp} [Notice]: Finished #{job.type} #{job.action} on #{job_body['node_count']} nodes, state: #{job_body['state']}, node_states: #{job_body['node_states']}")
            end
            Puppet.notice(_("Task run report saved to %{task_log}" % { task_log: task_log_file }))
          rescue StandardError => e
            Puppet.err(_('Error saving task log with the following message: %{msg}') % { msg: e.message })
          end
        end
      end
    end
  end
end
