# This profile sets up the Orchestrator service.
#
# @param certname [String] The name of the service SSL certificate.
# @param database_host [String] The hostname running PostgreSQL.
# @param database_port [Integer] The port that PostgreSQL is listening on.
# @param database_name [String] The name of the orchestrator database.
# @param database_user [String] The user account for the orchestrator database.
# @param database_migration_user [String] The user account for migrating the orchestrator database.
# @param database_password [String] The password for the user defined by database_user and database_migration_user.
# @param database_properties [String] A url encoded string of JDBC options.
# @param inventory_database_host [String] The hostname running PostgreSQL.
# @param inventory_database_port [Puppet_enterprise::Port] The port that PostgreSQL is listening on.
# @param inventory_database_name [String] The name of the inventory database.
# @param inventory_database_user [String] The user account for the inventory database.
# @param inventory_database_migration_user [String] The user account for migrating the inventory database.
# @param inventory_database_password [String] The password for the user defined by inventory_database_user and inventory_database_migration_user.
# @param ssl_listen_address [String] The network interface used for SSL connections.
# @param ssl_listen_port [Boolean] The port used for SSL connections.
# @param ssl_protocols [Array[String]] The list of SSL protocols to allow.
# @param ssl_cipher_suites [Array] The list of cipher suites to allow.
# @param pcp_listen_port [Boolean] The port used for PCP connections.
# @param master_host [String] The hostname of the primary.
# @param master_port [Integer] The port of puppet server on the primary.
# @param puppetdb_host Array[String] The hostname of the PuppetDB server.
# @param puppetdb_port Array[Integer] The port for the PuppetDB Servers
# @param rbac_host [String] The hostname of the RBAC service.
# @param rbac_port [String] The port of the RBAC service.
# @param rbac_prefix [String] The URL prefix of the RBAC service.
# @param activity_host [String] The hostname of the activity service.
# @param activity_port [String] The port of the activity service.
# @param activity_prefix [String] The URL prefix of the activity service.
# @param run_service [Boolean] Should pe-orchestration-services be running
# @param allowed_pcp_status_timeouts [Integer] The number of times a status request timeout is allowed before failing an individual job. Defaults in Orchestrator to 35 timeouts
# @param global_concurrent_compiles [Integer] The maximum number of Puppet runs to allow at once, across all jobs.
# @param task_concurrency [Integer] The maximum number of task runs to allow at once, across all jobs.
# @param bolt_server_max_connections [Integer] The maximum number of connections to bolt-server to allow at once, across all jobs.
# @param java_args [Hash] Key-value pairs describing the java arguments to be passed when starting the orchestrator
# @param disable_string_deduplication [Boolean] When using the default G1 garbage collector, this disables
#        the use of string deduplication for optimization.
# @param djava_io_tmpdir Directory to use as a tmpdir for orchestrator instead of the system default tmpdir.
# @param jruby_borrow_timeout [Integer] Set the timeout when attempting to borrow an instance from the JRuby pool in milliseconds.
# @param jruby_max_active_instances [Integer] The maximum number of jruby instances that the Orchestrator will spawn to execute Plans (this value is effectively the maximum number of concurrent plans).
# @param jruby_max_requests_per_instance [Integer] The maximum number of requests a single JRuby instance will handle before it is flushed from memory and refreshed. Defaults to 0 (Disabled).
# @param jruby_compile_mode [Enum] Turn JIT compiling on or off.  Default is on with 'jit', can also be 'off'.
# @param jruby_multithreaded [Boolean] Whether to run multiple request threads through a single JRuby instance concurrently.
# @param job_prune_threshold [Integer] The length of time in days that job history should be retained.
# @param use_application_services [Boolean] Whether to enable application management features in orchestrator.
# @param allow_unauthenticated_status [Boolean] True allows unauthenticated access, by default. False requires authentication on the orchestrator status-service endpoint.
# @param ace_server_max_connections [Integer] The maximum number of connections to ace-server to allow at once, across all jobs.
# @param reserved_code_cache [Optional[Pattern[/^\d+[kmgKMG]$/]] How much java codecache to allow orchestrator to use. Values should be strings that are in units of memory (i.e. something like 2048m)
# @param flight_recorder_enabled [Boolean] Enable/disable Java Flight Recorder for all Java services
# @param flight_recorder_settings [String] Settings value for JFR. The 'default' setting is low overhead, 'profile' gives more data.
# @param flight_recorder_stackdepth [Integer] Number of stack frames to capture. Default is 128 to account for Clojure/JRuby's deeply nested function calls.
# @param flight_recorder_maxage [String] Maximum amount of time to keep recorded data for each service. Can be specified by s (seconds), m (minutes), h (hours), or d (days) suffixes.
# @param flight_recorder_maxsize [String] Maximum size of recorded data files for each service. Can be specified with k (kilobytes), m (megabytes) and g (gigabytes) suffixes.
# @param flight_recorder_repository [String] Directory to save JFR chunks while recording.
# @param flight_recorder_filename [String] File to save completed recording.
# @param default_task_node_timeout [Integer] (Positive integer) The maximum number of seconds a task can execute on any individual node. If not set, orchestrator will use an internal default of forty minutes.
# @param default_deploy_node_timeout [Integer] (Positive integer) The maximum number of seconds a puppet run can execute on any individual node. If not set, orchestrator will use an internal default of thirty minutes.
# @param default_plan_timeout [Integer] (Positive integer) The maximum number of seconds the plan can execute. If not set, orchestrator will use an internal default of one hour.
# @param max_connections_per_route_authenticated [Integer] The maximum number of connections in a combination of host+port with requests that include a client certificate
# @param max_connections_total_authenticated [Integer] The maximum number of connections allowed at once for all routes with requests that include a client certificate
# @param max_connections_per_route_unauthenticated [Integer] The maximum number of connections in a combination of host+port with requests that do not include a client certificate
# @param max_connections_total_unauthenticated [Integer] The maximum number of connections allowed at once for all routes with requests that do not include a client certificate
# @param socket_timeout [Integer] The maximum number of milliseconds that the client will allow for no data to be available on the socket before closing the underlying connection
# @param host_action_collector_host [String] the host for the host-action-collector
# @param host_action_collector_port [Integer] the port for the host-action-collector service
# @param license_check_interval_seconds [Integer] Number of seconds between checking the license for changes/updates
# @param license_check_retries [Integer] Number of times to retry fetching license data if the URL fails to return data.
# @param plan_runner_active [Boolean] Whether or not to use the pe-plan-runner service to execute plans
class puppet_enterprise::profile::orchestrator (
  $certname                                                = $facts['clientcert'],
  $database_host                                           = $puppet_enterprise::orchestrator_database_host,
  $database_port                                           = $puppet_enterprise::database_port,
  $database_name                                           = $puppet_enterprise::orchestrator_database_name,
  $database_user                                           = $puppet_enterprise::orchestrator_service_regular_db_user,
  $database_migration_user                                 = $puppet_enterprise::orchestrator_service_migration_db_user,
  $database_password                                       = $puppet_enterprise::orchestrator_database_password,
  $database_properties                                     = $puppet_enterprise::database_properties,
  String $inventory_database_host                          = $puppet_enterprise::inventory_database_host,
  Puppet_enterprise::Port $inventory_database_port         = $puppet_enterprise::database_port,
  String $inventory_database_name                          = $puppet_enterprise::inventory_database_name,
  String $inventory_database_user                          = $puppet_enterprise::inventory_service_regular_db_user,
  String $inventory_database_migration_user                = $puppet_enterprise::inventory_service_migration_db_user,
  Optional[String] $inventory_database_password            = $puppet_enterprise::inventory_database_password,
  $ssl_listen_address                                      = $puppet_enterprise::params::ssl_address,
  $ssl_listen_port                                         = $puppet_enterprise::orchestrator_port,
  Array[String] $ssl_protocols                             = $puppet_enterprise::ssl_protocols,
  Array[String] $ssl_cipher_suites                         = $puppet_enterprise::ssl_cipher_suites,
  $master_host                                             = $puppet_enterprise::puppet_master_host,
  Integer $master_port                                     = $puppet_enterprise::puppet_master_port,
  Array[String]  $puppetdb_host                            = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_port                            = $puppet_enterprise::puppetdb_ports_array,
  $classifier_host                                         = $puppet_enterprise::console_host,
  $classifier_port                                         = $puppet_enterprise::api_port,
  $classifier_prefix                                       = $puppet_enterprise::classifier_url_prefix,
  $rbac_host                                               = $puppet_enterprise::console_host,
  $rbac_port                                               = $puppet_enterprise::api_port,
  $rbac_prefix                                             = $puppet_enterprise::rbac_url_prefix,
  $activity_host                                           = $puppet_enterprise::console_host,
  $activity_port                                           = $puppet_enterprise::api_port,
  $activity_prefix                                         = $puppet_enterprise::activity_url_prefix,
  $console_host                                            = $puppet_enterprise::console_host,
  $console_services_host                                   = $puppet_enterprise::console_host,
  $console_services_port                                   = $puppet_enterprise::api_port,
  Integer $inventory_port                                  = $puppet_enterprise::orchestrator_port,
  String $inventory_prefix                                 = $puppet_enterprise::inventory_url_prefix,
  $pcp_listen_port                                         = $puppet_enterprise::pcp_broker_port,
  $pcp_accept_consumers                                    = $puppet_enterprise::params::pcp_broker_accept_consumers,
  $pcp_delivery_consumers                                  = $puppet_enterprise::params::pcp_broker_delivery_consumers,
  Optional[Integer] $allowed_pcp_status_timeouts           = undef,
  $run_service                                             = true,
  Optional[Integer] $pcp_timeout                           = undef,
  Optional[Integer] $global_concurrent_compiles            = undef,
  Optional[Integer] $task_concurrency                      = undef,
  Integer $bolt_server_max_connections                     = $puppet_enterprise::bolt_server_concurrency,
  Optional[Integer] $job_prune_threshold                   = undef,
  Hash $java_args                                          = $puppet_enterprise::params::orchestrator_java_args,
  Boolean $disable_string_deduplication                    = $puppet_enterprise::disable_string_deduplication,
  Optional[String]  $djava_io_tmpdir                       = undef,
  Optional[Integer] $jruby_borrow_timeout                  = undef,
  Integer $jruby_max_active_instances                      = pe_max(Integer(puppet_enterprise::memorysize_to_megabytes($java_args['Xmx']) / 1024), 1),
  Integer $jruby_max_requests_per_instance                 = 0,
  Enum['jit', 'off'] $jruby_compile_mode                   = 'jit',
  Boolean $jruby_multithreaded                             = false,
  Optional[Boolean] $enable_gc_logging                     = undef,
  $use_application_services                                = $puppet_enterprise::use_application_services,
  Boolean $allow_unauthenticated_status                    = $puppet_enterprise::allow_unauthenticated_status,
  Integer $ace_server_max_connections                      = $puppet_enterprise::ace_server_concurrency,
  Optional[Pattern[/^\d+[kmgKMG]$/]] $reserved_code_cache  = undef,
  Integer $service_stop_retries                            = 60,
  Integer $start_timeout                                   = 300,
  Boolean $flight_recorder_enabled                         = $puppet_enterprise::flight_recorder_enabled,
  String $flight_recorder_settings                         = $puppet_enterprise::flight_recorder_settings,
  Integer $flight_recorder_stackdepth                      = $puppet_enterprise::flight_recorder_stackdepth,
  String $flight_recorder_maxage                           = $puppet_enterprise::flight_recorder_maxage,
  String $flight_recorder_maxsize                          = $puppet_enterprise::flight_recorder_maxsize,
  String $flight_recorder_repository                       = '/var/log/puppetlabs/orchestration-services/jfr',
  String $flight_recorder_filename                         = '/var/log/puppetlabs/orchestration-services/last_recording.jfr',
  Optional[Integer] $default_task_node_timeout             = undef,
  Optional[Integer] $default_deploy_node_timeout           = undef,
  Optional[Integer] $default_plan_timeout                  = undef,
  Integer $max_connections_per_route_authenticated         = 20,
  Integer $max_connections_total_authenticated             = 20,
  Integer $max_connections_per_route_unauthenticated       = 20,
  Integer $max_connections_total_unauthenticated           = 20,
  Integer $socket_timeout                                  = 120000,
  Boolean $plan_runner_active                              = false,
  String $host_action_collector_host                       = $puppet_enterprise::host_action_collector_host,
  Integer $host_action_collector_port                      = $puppet_enterprise::host_action_collector_port,
  Integer $license_check_interval_seconds                  = $puppet_enterprise::license_check_interval_seconds,
  Integer $license_check_retries                           = $puppet_enterprise::license_check_retries
) {
  # NOTE: this inclusion of the master profile is here because the plan_executor profile
  # needs to reference the puppet_enterprise::profile::master::code_manager_auto_configure value.
  # The intention is to ensure that no matter what order the classifier declares the classes,
  # the master profile is compiled first so that we can reference the parameter.
  # Since we only ever include the master and orchestrator profiles on the same node, it
  # shouldn't have any other impact, although it is ugly to look at.
  include puppet_enterprise::profile::master

  include puppet_enterprise::packages

  File {
    mode => '0644',
  }

  $container = 'orchestration-services'
  Package <| tag == 'pe-orchestrator-packages' |> {
    before => [Puppet_enterprise::Trapperkeeper::Orchestrator[$container],
      Puppet_enterprise::Trapperkeeper::Pcp_broker[$container],
    Class[puppet_enterprise::trapperkeeper::inventory]]
  }

  $client_certname = $puppet_enterprise::puppet_master_host

  # For PuppetDB HA, a user may pass in an Array to specify their PuppetDBs
  $first_puppetdb_host = $puppetdb_host[0]
  $first_puppetdb_port = pe_any2array($puppetdb_port)[0]

  $orch_ssl_dir = "/etc/puppetlabs/${container}/ssl"
  $client_pk8_key  =  "${orch_ssl_dir}/${certname}.private_key.pk8"
  $client_pem_key  =  "${orch_ssl_dir}/${certname}.private_key.pem"
  $client_cert     =  "${orch_ssl_dir}/${certname}.cert.pem"
  $puppetdb_url    =  "https://${first_puppetdb_host}:${first_puppetdb_port}"

  if $puppet_enterprise::database_ssl and $puppet_enterprise::database_cert_auth {
    $ssl_database_properties = "${database_properties}&sslkey=${client_pk8_key}&sslcert=${client_cert}"
  } else {
    $ssl_database_properties = $database_properties
  }

  if $settings::storeconfigs {
    $pcp_brokers = puppetdb_query("resources[certname] { type = 'Class' and title = 'Puppet_enterprise::Master::Pcp_broker' }").map |$node| { $node['certname'] }
  } else {
    $pcp_brokers = []
  }

  puppet_enterprise::trapperkeeper::orchestrator { $container:
    database_host                             => $database_host,
    database_port                             => $database_port,
    database_name                             => $database_name,
    database_user                             => $database_user,
    database_migration_user                   => $database_migration_user,
    database_password                         => $database_password,
    database_properties                       => $ssl_database_properties,
    master_url                                => "https://${master_host}:${master_port}",
    puppetdb_url                              => $puppetdb_url,
    classifier_url                            => "https://${classifier_host}:${classifier_port}${classifier_prefix}",
    console_services_url                      => "https://${console_services_host}:${console_services_port}",
    rbac_url                                  => "https://${rbac_host}:${rbac_port}${rbac_prefix}",
    activity_url                              => "https://${activity_host}:${activity_port}${activity_prefix}",
    console_url                               => "https://${console_host}",
    inventory_service_url                     => "https://${master_host}:${inventory_port}${inventory_prefix}",
    pcp_brokers                               => $pcp_brokers,
    notify                                    => Service["pe-${container}"],
    pcp_timeout                               => $pcp_timeout,
    allowed_pcp_status_timeouts               => $allowed_pcp_status_timeouts,
    global_concurrent_compiles                => $global_concurrent_compiles,
    task_concurrency                          => $task_concurrency,
    bolt_server_max_connections               => $bolt_server_max_connections,
    job_prune_threshold                       => $job_prune_threshold,
    app_management                            => $use_application_services,
    allow_unauthenticated_status              => $allow_unauthenticated_status,
    ace_server_max_connections                => $ace_server_max_connections,
    jruby_borrow_timeout                      => $jruby_borrow_timeout,
    jruby_max_active_instances                => $jruby_max_active_instances,
    jruby_max_requests_per_instance           => $jruby_max_requests_per_instance,
    jruby_compile_mode                        => $jruby_compile_mode,
    jruby_multithreaded                       => $jruby_multithreaded,
    default_task_node_timeout                 => $default_task_node_timeout,
    default_deploy_node_timeout               => $default_deploy_node_timeout,
    default_plan_timeout                      => $default_plan_timeout,
    max_connections_per_route_authenticated   => $max_connections_per_route_authenticated,
    max_connections_total_authenticated       => $max_connections_total_authenticated,
    max_connections_per_route_unauthenticated => $max_connections_per_route_unauthenticated,
    max_connections_total_unauthenticated     => $max_connections_total_unauthenticated,
    socket_timeout                            => $socket_timeout,
    plan_runner_active                        => $plan_runner_active,
  }

  puppet_enterprise::trapperkeeper::pcp_broker { $container:
    accept_consumers   => $pcp_accept_consumers,
    delivery_consumers => $pcp_delivery_consumers,
    controller_uris    => ["wss://${certname}:${ssl_listen_port}/server"],
    notify             => Service["pe-${container}"],
  }

  class { 'puppet_enterprise::trapperkeeper::inventory':
    database_host           => $inventory_database_host,
    database_port           => $inventory_database_port,
    database_name           => $inventory_database_name,
    database_user           => $inventory_database_user,
    database_migration_user => $inventory_database_migration_user,
    database_password       => $inventory_database_password,
    database_properties     => $ssl_database_properties,
    puppetdb_url            => $puppetdb_url,
    client_pem_key          => $client_pem_key,
    client_cert             => $client_cert,
  }
  contain puppet_enterprise::trapperkeeper::inventory

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container} webrouting-service":
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.webrouting.webrouting-service',
    service   => 'webrouting-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:pcp-broker metrics-service":
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    service   => 'metrics-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:pcp-broker metrics-webservice":
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    service   => 'metrics-webservice',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:pcp-broker scheduler-service" :
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.scheduler.scheduler-service',
    service   => 'scheduler-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:pcp-broker status-service" :
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.status.status-service',
    service   => 'status-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { '${container}:host-action-collector-service' :
    container => $container,
    namespace => 'puppetlabs.host-action-collector.service',
    service   => 'host-action-collector-service',
  }

  $confdir = "/etc/puppetlabs/${container}/conf.d"

  file { "${confdir}/auth.conf":
    ensure => file,
    owner  => "pe-${container}",
    group  => "pe-${container}",
    mode   => '0640',
  }

  # Remove the old authorization.conf file laid down by previous
  # orchestration-services packages
  file { "${confdir}/authorization.conf":
    ensure => absent,
  }

  pe_hocon_setting { "${container}.authorization.version":
    path    => "${confdir}/auth.conf",
    setting => 'authorization.version',
    value   => 1,
  }

  file { "${confdir}/webserver.conf":
    ensure => file,
    owner  => "pe-${container}",
    group  => "pe-${container}",
    mode   => '0640',
  }

  $access_log_config = "/etc/puppetlabs/${container}/request-logging.xml"

  Pe_hocon_setting {
    ensure => present,
    path   => "${confdir}/webserver.conf",
    notify => Service["pe-${container}"],
  }

  puppet_enterprise::trapperkeeper::global_settings { $container :
    certname    => $client_certname,
    ssl_cert    => "/etc/puppetlabs/${container}/ssl/${client_certname}.cert.pem",
    ssl_key     => "/etc/puppetlabs/${container}/ssl/${client_certname}.private_key.pem",
    localcacert => $puppet_enterprise::params::localcacert,
  }

  puppet_enterprise::trapperkeeper::webserver_settings { 'pcp-broker' :
    container          => $container,
    ssl_listen_address => $ssl_listen_address,
    ssl_listen_port    => Integer($pcp_listen_port),
    ssl_cert           => "/etc/puppetlabs/${container}/ssl/${certname}.cert.pem",
    ssl_key            => "/etc/puppetlabs/${container}/ssl/${certname}.private_key.pem",
    ssl_protocols      => $ssl_protocols,
    ssl_cipher_suites  => $ssl_cipher_suites,
  }

  pe_hocon_setting { "${container}.web-router-service.broker-service":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.pcp.broker.service/broker-service"',
    value   => {
      'v1'      => {
        'route'  => '/pcp',
        'server' => 'pcp-broker',
      },
      'v2'      => {
        'route'  => '/pcp2',
        'server' => 'pcp-broker',
      },
      'metrics' => {
        'route'  => '/',
        'server' => 'pcp-broker',
      },
    },
  }

  puppet_enterprise::trapperkeeper::webserver_settings { 'orchestrator' :
    container          => $container,
    ssl_listen_address => $ssl_listen_address,
    ssl_listen_port    => Integer($ssl_listen_port),
    ssl_cert           => "/etc/puppetlabs/${container}/ssl/${certname}.cert.pem",
    ssl_key            => "/etc/puppetlabs/${container}/ssl/${certname}.private_key.pem",
    ssl_protocols      => $ssl_protocols,
    ssl_cipher_suites  => $ssl_cipher_suites,
    access_log_config  => $access_log_config,
    default_server     => true,
  }

  pe_hocon_setting { "${container}.web-router-service.orchestrator-service":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.orchestrator.service/orchestrator-service"',
    value   => { route => '/orchestrator/v1', server => 'orchestrator' },
  }

  pe_hocon_setting { "${container}.web-router-service.orchestrator-dispatch-service":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.orchestrator.dispatch/orchestrator-dispatch-service"',
    value   => { route => '/server', server => 'orchestrator' },
  }

  pe_hocon_setting { "${container}.web-router-service.status-service":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.status.status-service/status-service"',
    value   => { route => '/status', server => 'orchestrator' },
  }

  pe_hocon_setting { "${container}.web-router-service.metrics-webservice":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.metrics.metrics-service/metrics-webservice"',
    value   => { 'route' => '/metrics', 'server' => 'orchestrator' },
  }

  puppet_enterprise::certs { $container:
    certname      => $client_certname,
    cert_dir      => $orch_ssl_dir,
    container     => $container,
    make_pk8_cert => true,
  }

  $_djava_io_tmpdir = pe_pick($djava_io_tmpdir, '/opt/puppetlabs/server/apps/orchestration-services/tmp')
  $_tmpdir_java_args = { 'Djava.io.tmpdir=' => $_djava_io_tmpdir }
  if $flight_recorder_enabled {
    $_jfr_java_args = puppet_enterprise::jfr_java_args(
      $flight_recorder_repository,
      $flight_recorder_filename,
      $flight_recorder_settings,
      $flight_recorder_stackdepth,
      $flight_recorder_maxage,
      $flight_recorder_maxsize
    )
  } else {
    $_jfr_java_args = {}
  }

  $_java_args = pe_merge($java_args, $_tmpdir_java_args, puppet_enterprise::calculate_reserved_code_cache($reserved_code_cache), $_jfr_java_args)

  puppet_enterprise::trapperkeeper::java_args { $container :
    java_args                    => $_java_args,
    disable_string_deduplication => $disable_string_deduplication,
    enable_gc_logging            => $enable_gc_logging,
  }

  puppet_enterprise::trapperkeeper::init_defaults { $container :
    service_stop_retries => $service_stop_retries,
    start_timeout        => $start_timeout,
  }

  $_orchestrator_service_ensure = $run_service ? { true => 'running', false => 'stopped' }
  puppet_enterprise::trapperkeeper::pe_service { $container :
    ensure => $_orchestrator_service_ensure,
    enable => $run_service,
  }

  include puppet_enterprise::profile::bolt_server
  include puppet_enterprise::profile::ace_server

  $orchestrator_conf_path = "${confdir}/orchestrator.conf"
  # NOTE: Plan Runner is going to succeed the JRuby-based Plan Executor
  pe_hocon_setting { "${container}.orchestrator.v2-api":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.v2-api',
    value   => $plan_runner_active,
  }

  class { 'puppet_enterprise::profile::plan_runner':
    active => $plan_runner_active,
  }

  if $plan_runner_active {
    pe_hocon_setting { "${container}.orchestrator.plan-executor-enable":
      path    => $orchestrator_conf_path,
      setting => 'orchestrator.plan-executor-enable',
      value   => false,
    }
  } else {
    Package <| tag == 'pe-orchestrator-packages' |> {
      before => [Class[puppet_enterprise::profile::plan_executor]]
    }
    include puppet_enterprise::profile::plan_executor
  }

  #
  # File Sync configuration for plan execution (either plan_executor or plan_runner)
  #

  $_enable_plan_executor = lookup('puppet_enterprise::profile::plan_executor::enable',
  { 'default_value' => undef })

  if $_enable_plan_executor != undef {
    puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::enable': }
  }

  # In the plan_executor profile if this param is not explicitly set to false, it will default to true.
  if $_enable_plan_executor == false {
    $enable_plan_executor = false
  } else {
    $enable_plan_executor = true
  }

  $cm_auto_config = $puppet_enterprise::profile::master::code_manager_auto_configure
  $executor_needs_file_sync = ($enable_plan_executor and $cm_auto_config)
  $enable_file_sync_client = ($executor_needs_file_sync or $plan_runner_active)

  pe_hocon_setting { "${container}.orchestrator.enable-file-sync-client":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.enable-file-sync-client',
    value   => $enable_file_sync_client,
  }

  if $enable_file_sync_client {
    $deprecated_versioned_deploys = lookup('puppet_enterprise::profile::plan_executor::versioned_deploys',
    { 'default_value' => false })
    if $deprecated_versioned_deploys != false {
      puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::versioned_deploys': }
    }
    if ($puppet_enterprise::master::code_management::versioned_deploys_enabled == false and $plan_runner_active == true) {
      fail('The plan runner feature of Orchestrator requires the versioned deploys feature of code manager. In order to configure plan_runner_active you must have versioned deploys enabled')
    }

    $file_sync_data_dir = lookup('puppet_enterprise::profile::plan_executor::file_sync_data_dir',
    { 'default_value' => '/opt/puppetlabs/server/data/orchestration-services/data-dir' })
    if $file_sync_data_dir != '/opt/puppetlabs/server/data/orchestration-services/data-dir' {
      puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::file_sync_data_dir': }
    }

    $file_sync_poll_interval = lookup('puppet_enterprise::profile::plan_executor::file_sync_poll_interval',
    { 'default_value' => 5 })
    if $file_sync_poll_interval != 5 {
      puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::file_sync_poll_interval': }
    }

    $file_sync_stream_file_threshold = lookup('puppet_enterprise::profile::plan_executor::file_sync_stream_file_threshold',
    { 'default_value' => 1024 })
    if $file_sync_stream_file_threshold != 1024 {
      puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::file_sync_stream_file_threshold': }
    }

    $fs_livedir = lookup('puppet_enterprise::profile::plan_executor::environments_codedir',
    { 'default_value' => '/opt/puppetlabs/server/data/orchestration-services/code' })
    if $fs_livedir != '/opt/puppetlabs/server/data/orchestration-services/code' {
      puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::environments_codedir': }
    }

    file { "${confdir}/file-sync.conf":
      ensure => file,
      owner  => "pe-${container}",
      group  => "pe-${container}",
      mode   => '0640',
    }

    pe_hocon_setting { "${container}.web-router-service.file-sync-web-service":
      path    => "${confdir}/web-routes.conf",
      setting => 'web-router-service."puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service/file-sync-web-service"',
      value   => {
        'route'  => '/file-sync',
        'server' => 'orchestrator',
      },
    }

    pe_hocon_setting { "${container}.file-sync.data-dir":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.data-dir',
      value   => $file_sync_data_dir,
    }

    pe_hocon_setting { "${container}.file-sync.client.client-identifier":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.client-identifier',
      value   => "${client_certname}-orchestrator",
    }

    pe_hocon_setting { "${container}.file-sync.client.poll-interval":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.poll-interval',
      value   => $file_sync_poll_interval,
    }

    pe_hocon_setting { "${container}.file-sync.client.server-api-url":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.server-api-url',
      value   => "https://${master_host}:${master_port}/file-sync/v1",
    }

    pe_hocon_setting { "${container}.file-sync.client.server-repo-url":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.server-repo-url',
      value   => "https://${master_host}:${master_port}/file-sync-git",
    }

    pe_hocon_setting { "${container}.file-sync.client.ssl-ca-cert":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.ssl-ca-cert',
      value   => $puppet_enterprise::params::localcacert,
    }

    pe_hocon_setting { "${container}.file-sync.client.ssl-cert":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.ssl-cert',
      value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.cert.pem",
    }

    pe_hocon_setting { "${container}.file-sync.client.ssl-key":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.ssl-key',
      value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.private_key.pem",
    }

    pe_hocon_setting { "${container}.file-sync.client.stream-file-threshold":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.stream-file-threshold',
      value   => $file_sync_stream_file_threshold,
    }

    pe_hocon_setting { "${container}.file-sync.client.copy-method":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.copy-method',
      value   => $puppet_enterprise::master::code_management::file_sync_copy_method,
    }

    pe_hocon_setting { "${container}.file-sync.client.versioned-sync-pool":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.client.versioned-sync-pool',
      value   => $puppet_enterprise::master::code_management::file_sync_versioned_sync_pool,
    }

    pe_hocon_setting { "${container}.file-sync.repos.puppet-code.live-dir":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.repos.puppet-code.live-dir',
      value   => $fs_livedir,
    }
    # The value of submodules-dir isn't configurable, since it also isn't
    # configurable in orchestrator, bolt-server, or the plan runner. Eventually
    # we should make that value configurable but for now just do what all the
    # orchestration services expect
    pe_hocon_setting { "${container}.file-sync.repos.puppet-code.submodules-dir":
      path    => "${confdir}/file-sync.conf",
      setting => 'file-sync.repos.puppet-code.submodules-dir',
      value   => 'environments',
    }

    if($puppet_enterprise::master::code_management::versioned_deploys_enabled) {
      pe_hocon_setting { "${container}.file-sync.repos.puppet-code.versioned-dirs":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.puppet-code.versioned-dirs',
        value   => true,
      }
      pe_hocon_setting { "${container}.orchestrator.enable-file-sync-locking":
        path    => $orchestrator_conf_path,
        setting => 'orchestrator.enable-file-sync-locking',
        value   => false,
      }
    } else {
      pe_hocon_setting { "${container}.file-sync.repos.puppet-code.versioned-dirs":
        ensure  => absent,
        setting => 'file-sync.repos.puppet-code.versioned-dirs',
        path    => "${confdir}/file-sync.conf",
      }
      pe_hocon_setting { "${container}.orchestrator.enable-file-sync-locking":
        path    => $orchestrator_conf_path,
        setting => 'orchestrator.enable-file-sync-locking',
        value   => true,
      }
    }

    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:file-sync web-service" :
      container => $container,
      namespace => 'puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service',
      service   => 'file-sync-web-service',
    }

    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:file-sync client-service" :
      container => $container,
      namespace => 'puppetlabs.enterprise.services.file-sync-client.file-sync-client-service',
      service   => 'file-sync-client-service',
    }
  } else {
    file { "${confdir}/file-sync.conf":
      ensure => absent,
    }

    pe_hocon_setting { "${container}.web-router-service.file-sync-web-service":
      ensure  => absent,
      path    => "${confdir}/web-routes.conf",
      setting => 'web-router-service."puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service/file-sync-web-service"',
    }
  }

  # host action collector
  file { "${confdir}/host-action-collector.conf":
    ensure  => file,
    owner   => "pe-${container}",
    group   => "pe-${container}",
    mode    => '0640',
    require => Package['pe-orchestration-services'],
  }

  pe_hocon_setting { "${container}.host-action-collector.certs.ssl-key":
    ensure  => present,
    path    => "${confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-key',
    value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.private_key.pem",
    require => Package['pe-orchestration-services'],
  }

  pe_hocon_setting { "${container}.host-action-collector.certs.ssl-ca-cert":
    ensure  => present,
    path    => "${confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-ca-cert',
    value   => $puppet_enterprise::params::localcacert,
    require => Package['pe-orchestration-services'],
  }

  pe_hocon_setting { "${container}.host-action-collector.certs.ssl-cert":
    ensure  => present,
    path    => "${confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-cert',
    value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.cert.pem",
    require => Package['pe-orchestration-services'],
  }

  pe_hocon_setting { "${container}.host-action-collector.storage-location":
    ensure  => present,
    path    => "${confdir}/host-action-collector.conf",
    setting => 'host-action-collector.storage-location',
    value   => '/var/log/puppetlabs/orchestration-services/data',
    require => Package['pe-orchestration-services'],
  }

  pe_hocon_setting { "${container}.host-action-collector.collector-host-url":
    ensure  => present,
    path    => "${confdir}/host-action-collector.conf",
    setting => 'host-action-collector.collector-host-url',
    value   => "https://${host_action_collector_host}:${host_action_collector_port}",
    require => Package['pe-orchestration-services'],
  }

  puppet_enterprise::license_check_config { $container:
    conf_file_path => $orchestrator_conf_path,
    license_api    => "https://${host_action_collector_host}:${host_action_collector_port}/license/v1/current",
    check_interval => $license_check_interval_seconds,
    retries        => $license_check_retries,
    notify         => Service["pe-${container}"],
  }

  # The orchestration-services user needs to be in the pe-puppet group so that it can read from
  # codedirs and the 'hidden' modulepaths for entitlement content
  user { 'pe-orchestration-services':
    groups  => ['pe-orchestration-services', 'pe-puppet'],
    require => Package['pe-orchestration-services'],
  }
}
