# This is run via Bolt over SSH.  Currently, this will also regenerate the cert
# on a standalone postgres node as well, if one exists.  When regenerating a
# cert that contains subject alt names (defined in puppet.conf), those subject
# alt names MUST be passed in to the dns_alt_names parameter.  This is a security
# measure to ensure that unexpected alt names do not appear in the cert after
# regeneration.
# Use --force option if the certs on the primary are damaged
# to bypass node verification failure.
# Set offline=true to generate certs in puppetserver offline mode with
# 'puppetserver ca generate --ca-client --force'. Use this option only if your
#  server certs are expired
#
# Steps this plan takes:
# 1. Disable the agent on the primary (and standalone postgres if it exists) with
#    puppet agent --disable.
# 2. Backup certs on primary (see the backup step in run regenerate_agent_certificate)
# 3. Remove cached catalog at
#    /opt/puppetlabs/puppet/cache/client_data/catalog/<primary_certname>.json
# 4. Cleanup certs
#    Run puppetserver ca clean --certname=<primary_certname>
#    OR
#    Stop puppet service and delete cert files if offline mode is set
# 5. Run find <dir> -name <primary_certname>.* -delete with the following directories
#      a. /etc/puppetlabs/puppet/ssl
#      b. /etc/puppetlabs/puppetdb/ssl
#      c. /etc/puppetlabs/ace-server/ssl
#      d. /etc/puppetlabs/bolt-server/ssl
#      e. /etc/puppetlabs/orchestration-services/ssl
#      f. /opt/puppetlabs/server/data/console-services/certs
# 6. If extension_requests or custom_attributes are defined, add them to
#    csr_attributes.yaml.
# 7. Stop the pe-puppetserver service.
# 8. If dns_alt_names is defined, add pe_install::puppet_master_dnsaltnames to
#    pe.conf with this value (converted into a string array).
# 9. If offline mode is set, run 'puppetserver ca generate --ca-client --force' on primary
# 10. Run puppet infra configure --no-recover
# 11. If a standalone postgres node exists
#       a. Run puppet agent -t on the node
#       b. Remove the cached catalog at
#          /opt/puppetlabs/puppet/cache/client_data/catalog/<postgres_certname>.json
#       c. Run puppetserver ca clean --certname=<postgres_certname> on the primary.
#       d. Delete the certs noted in previously, using the postgres node certname,
#          on the postgres node.
#       e. Run puppet infra configure --no-recover on the postgres node.
# 12. Run puppet on the primary, and standalone postgres node if it exists.
# 13. If dns_alt_names are specified, run puppetserver ca list to verify the
#     alt names are in the cert.
# 14. If manage_pxp_service is true, restart the pxp-agent service.
# 15. Re-enable the agent on the primary (and standalone postgres if it exists)
#     with puppet agent --enable.
#
# @api private
plan enterprise_tasks::primary_cert_regen(
  Optional[TargetSpec] $primary         = 'localhost',
  Optional[Hash] $extension_requests    = undef,
  Optional[Hash] $custom_attributes     = undef,
  Optional[Boolean] $manage_pxp_service = true,
  Optional[String] $dns_alt_names       = undef,
  Optional[Boolean] $force              = false,
  Optional[Boolean] $offline            = false,
) {
  $constants = constants()
  enterprise_tasks::test_connection($primary)
  enterprise_tasks::verify_node($primary, 'primary', $force or $offline)

  $db_host_or_error = catch_errors() || {
    enterprise_tasks::get_nodes_with_profile('database')[0]
    }
  if $db_host_or_error =~ Error {
    enterprise_tasks::message('primary_cert_regen', 'Unable to identify a database host. If you are using a standalone PostgreSQL node, you must regenerate its certificate as well with puppet infra run regenerate_standalone_db_certificate')
    $database_host = undef
  } else {
    $database_host = $db_host_or_error
  }

  $primary_certname = run_command("${constants['puppet_bin']} config print certname", $primary).first['stdout'].strip
  # If there is no defined database_host, it's likely because the primary node was purged and has the database on it,
  # so we should assume that the primary node is the db node in that case
  $is_external_postgres = $database_host and ($database_host != $primary_certname) and !enterprise_tasks::node_has_profile($database_host, 'primary_master_replica')
  $infra_nodes = $is_external_postgres ? {
    true  => [$primary, $database_host],
    false => [$primary],
  }

  if $is_external_postgres {
    enterprise_tasks::message('primary_cert_regen','We noticed that you have a standalone PE-PostgreSQL node. Please be aware that this plan will regenerate certs on this node as well.')
  }
  if $dns_alt_names {
    $dns_alt_names_array = split($dns_alt_names, ',').strip
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force => $force or $offline)
  }

  $status_hash = run_plan(enterprise_tasks::get_service_status, target => $primary,
    service    => 'puppet',
  )

  $result_or_error = catch_errors() || {
    enterprise_tasks::with_agent_disabled($infra_nodes) || {
      $primary_node_facts = run_plan('facts', $primary, '_catch_errors' => true)
      if $primary_node_facts =~ Error {
        if $primary_node_facts.details()['result_set'].first.value['pe_postgresql_info'] {
          $pe_postgresql_info = $primary_node_facts.details()['result_set'].first.value['pe_postgresql_info']
        } else {
          enterprise_tasks::message('primary_cert_regen', "${primary_node_facts.kind()}:${primary_node_facts.msg()}")
          fail_plan('primary_cert_regen', "Failed to retrieve os facts from ${node}.")
        }
      } else {
        $pe_postgresql_info = $primary_node_facts.first.value['pe_postgresql_info']
      }
      $pg_version = $pe_postgresql_info['installed_server_version']

      enterprise_tasks::message('primary_cert_regen', 'Regenerating certs on primary node...')
      run_task(enterprise_tasks::pe_services, $primary, state => 'stopped') # Only 'puppet' since we pass in no role

      $backups = run_task(enterprise_tasks::backup_certs, $primary,
        pg_version => $pg_version,
      ).first().value()['backups']
      enterprise_tasks::message('primary_cert_regen',"Certificate backups saved to ${backups}")

      run_command("rm -f /opt/puppetlabs/puppet/cache/client_data/catalog/${primary_certname}.json", $primary)
      if ($offline) {
        $cadir = run_command("${constants['puppet_bin']} config print cadir", $primary).first['stdout'].strip
        run_command("find ${cadir} -name ${primary_certname}.* -delete", $primary)
      } else {
        run_task(enterprise_tasks::puppetserver_ca_clean, $primary,
          certname    => $primary,
        )
      }
      run_task(enterprise_tasks::delete_cert, $primary,
        certname => $primary,
      )
      if $extension_requests or $custom_attributes {
        run_task(enterprise_tasks::set_csr_attributes, $primary,
          extension_requests => $extension_requests,
          custom_attributes  => $custom_attributes,
        )
      }
      run_task(enterprise_tasks::pe_services, $primary,
        role  => 'primary',
        state => 'stopped',
      )
      if $dns_alt_names and !empty($dns_alt_names_array) {
        enterprise_tasks::message('primary_cert_regen', 'Updating pe.conf with dns_alt_names...')
        $conf_key = 'pe_install::puppet_master_dnsaltnames'
        run_task(enterprise_tasks::add_modify_conf_keys,
          $primary,
          file    => $constants['pe_conf'],
          hash    => { $conf_key => $dns_alt_names_array },
        )
      }
      if ($offline) {
        run_command("/opt/puppetlabs/bin/puppetserver ca  generate --certname ${primary_certname} --ca-client --force", $primary)
      }
      run_task(enterprise_tasks::puppet_infra_configure, $primary)

      if $is_external_postgres {
        run_plan(enterprise_tasks::standalone_pg_cert_regen, primary => $primary, external_pg => $database_host)
      }
      $infra_nodes.each |$node| {
        run_task(enterprise_tasks::run_puppet, $node)
      }

      if $dns_alt_names and !empty($dns_alt_names_array) {
        $result = run_command("/opt/puppetlabs/bin/puppetserver ca list --all | grep ${primary_certname}", $primary).first().value()['stdout']
        $dns_alt_names_array.each |String $dns_name| {
          unless "DNS:${dns_name}" in $result {
            enterprise_tasks::message('primary_cert_regen', 'Cert was not generated with dns_alt_names')
            fail_plan('ERROR', 'Cert was not generated with dns_alt_names')
          }
        }
      }
    }

    if $manage_pxp_service {
      $targets = get_targets($primary)
      $targets.each |$target| {
        enterprise_tasks::set_feature($target, 'puppet-agent', true)
      }
      run_task(service, [$primary],
        name => 'pxp-agent',
        action => 'restart',
        _catch_errors => true,
      )
      wait_until_available([$primary], wait_time => 30)
    }
  }
  enterprise_tasks::message('primary_cert_regen', 'Applying original puppet service state...')
  run_command("${constants['puppet_bin']} resource service puppet ensure=${status_hash[status]} enable=${status_hash[enabled]}", $primary)
  if $result_or_error =~ Error {
    enterprise_tasks::message('primary_cert_regen', "${result_or_error.details()}")
    return fail_plan($result_or_error)
  }
}
