# This plan is run via Bolt over SSH.
#
# Steps this plan takes:
# 1. Disable the agent on primary infrastructure nodes with puppet agent --disable.
# 2. Stop all PE services on primary and standalone postgres node, if it exists.
# 3. Back up certs on primary and standalone postgres node, if it exists (see the
#    backup step in run regenerate_agent_certificate)
# 4. Run rm -rf /etc/puppetlabs/puppet/ssl/* on the primary.
# 5. Remove cached catalog at
#    /opt/puppetlabs/puppet/cache/client_data/catalog/<primary_certname>.json
# 6. Run puppet infra configure --no-recover on the primary. If a standalone
#    postgres node exists, this will fail at a certain point. That’s expected.
# 7. If a standalone postgres node exists, repeat steps 4 through 6 on the
#    postgres node, then repeat the previous step on the primary node.
# 8. Run puppet on the primary, and the standalone postgres node, if it exists.
# 9. If manage_pxp_service is true, restart pxp-agent.
# 10. Re-enable the agent on primary infrastructure nodes with
#     puppet agent --enable.
#
# @api private
plan enterprise_tasks::rebuild_ca(
  Optional[TargetSpec] $primary         = 'localhost',
  Optional[Boolean] $manage_pxp_service = true,
  Optional[Boolean] $force              = false,
) {
  enterprise_tasks::verify_node($primary, 'primary', $force)

  $database_host = enterprise_tasks::get_nodes_with_profile('database')[0]
  $primary_certname = run_command('/opt/puppetlabs/bin/puppet config print certname', $primary).first['stdout'].strip
  # If there is no defined database_host, it's likely because the primary node was purged and has the database on it,
  # so we should assume that the primary node is the db node in that case
  $is_external_postgres = $database_host and ($database_host != $primary_certname) and !enterprise_tasks::node_has_profile($database_host, 'primary_master_replica')
  $infra_nodes = $is_external_postgres ? {
    true  => [$primary, $database_host],
    false => [$primary],
  }

  $status_hash = run_plan(enterprise_tasks::get_service_status, target => $primary,
    service => 'puppet',
  )

  enterprise_tasks::verify_node($primary, 'ca', $force)
  enterprise_tasks::test_connection($infra_nodes)

  $result_or_error = catch_errors() || {
    enterprise_tasks::with_agent_disabled($infra_nodes) || {
      $primary_node_facts = run_plan('facts', $primary, '_catch_errors' => true)
      if $primary_node_facts =~ Error {
        if $primary_node_facts.details()['result_set'].first.value['pe_postgresql_info'] {
          $pe_postgresql_info = $primary_node_facts.details()['result_set'].first.value['pe_postgresql_info']
        } else {
          enterprise_tasks::message('rebuild_ca', "${primary_node_facts.kind()}:${primary_node_facts.msg()}")
          fail_plan('rebuild_ca', "Failed to retrieve os facts from ${node}.")
        }
      } else {
        $pe_postgresql_info = $primary_node_facts.first.value['pe_postgresql_info']
      }
      $pg_version = $pe_postgresql_info['installed_server_version']

      enterprise_tasks::message('rebuild_ca', 'Stopping PE services...')
      run_task(enterprise_tasks::pe_services, $primary,
        role  => 'primary',
        state => 'stopped',
      )

      if $is_external_postgres {
        run_task(enterprise_tasks::pe_services, $database_host,
          role  => 'external_postgres',
          state => 'stopped',
        )
      }

      enterprise_tasks::message('rebuild_ca', 'Backing up certificates...')
      $backups = run_task(enterprise_tasks::backup_certs, $infra_nodes, pg_version => $pg_version)
      $backups.each |$r| { enterprise_tasks::message('rebuild_ca', "Certificate backups saved to ${r.value()['backups']} on ${r.target}") }

      enterprise_tasks::message('rebuild_ca', 'Deleting ca files...')
      run_task(enterprise_tasks::delete_ca_files, $primary)

      enterprise_tasks::message('rebuild_ca', 'Regenerating ca...')
      # If we have an external postgres, we expect infra configure failures
      run_task(enterprise_tasks::puppet_infra_configure, $primary, _catch_errors => $is_external_postgres)

      enterprise_tasks::message('rebuild_ca', 'Reconfiguring infrastructure nodes...')
      if $is_external_postgres {
        run_task(enterprise_tasks::delete_ca_files, $database_host)
        run_task(enterprise_tasks::puppet_infra_configure, $database_host)
        run_task(enterprise_tasks::puppet_infra_configure, $primary)
      }

      $infra_nodes.each |$hostname| {
        run_task(enterprise_tasks::run_puppet, $hostname)
      }

      if $manage_pxp_service {
        enterprise_tasks::message('rebuild_ca', 'Restarting pxp-agent...')
        run_task(service, [$primary],
          name          => 'pxp-agent',
          action        => 'restart',
          _catch_errors => true,
        )
        wait_until_available([$primary], wait_time => 30)
      }
    }
  }
  enterprise_tasks::message('rebuild_ca', 'Applying original puppet service state...')
  run_command("${constants()['puppet_bin']} resource service puppet ensure=${status_hash[status]} enable=${status_hash[enabled]}", $primary)
  if $result_or_error =~ Error {
    return fail_plan($result_or_error)
  }
  else {
    enterprise_tasks::message('rebuild_ca', "Successfully rebuilt the certificate authority. Make sure to clean up the local CA and CRL bundle on all of your agent nodes by running 'puppet infra run regenerate_agent_certificate' with 'crl_clean=true' setting.")
    enterprise_tasks::message('rebuild_ca', "You will also need to run the 'puppet infra run regenerate_replica_certificate' and/or 'puppet infra run regenerate_compiler_certificate' commands if these exist in your infrastructure.")
    enterprise_tasks::message('rebuild_ca', 'Note that you will need to use the --use-ssh or --use-winrm flags with these plans as the PXP agent will be unable to connect until the certificates are regenerated.')
  }
}
