# frozen_string_literal: true

require 'hocon'
require 'bolt/error'
require 'logging'

module PEBoltServer
  class BaseConfig
    def config_keys
      %w[host port ssl-cert ssl-key ssl-ca-cert
         ssl-cipher-suites ssl-crl-path loglevel logfile allowlist
         environments-codedir
         environmentpath basemodulepath]
    end

    def env_keys
      %w[ssl-cert ssl-key ssl-ca-cert ssl-crl-path loglevel]
    end

    def defaults
      { 'host' => '127.0.0.1',
        'loglevel' => 'warn',
        'ssl-cipher-suites' => %w[ECDHE-ECDSA-AES256-GCM-SHA384
                                  ECDHE-RSA-AES256-GCM-SHA384
                                  ECDHE-ECDSA-CHACHA20-POLY1305
                                  ECDHE-RSA-CHACHA20-POLY1305
                                  ECDHE-ECDSA-AES128-GCM-SHA256
                                  ECDHE-RSA-AES128-GCM-SHA256
                                  ECDHE-ECDSA-AES256-SHA384
                                  ECDHE-RSA-AES256-SHA384
                                  ECDHE-ECDSA-AES128-SHA256
                                  ECDHE-RSA-AES128-SHA256] }
    end

    def ssl_keys
      %w[ssl-cert ssl-key ssl-ca-cert ssl-crl-path]
    end

    def license_check_keys
      %w[check-interval-seconds license-api retries]
    end

    def required_keys
      ssl_keys
    end

    def service_name
      raise "Method service_name must be defined in the service class"
    end

    def initialize(config = nil)
      @data = defaults
      if config
        config_override = config.slice(*config_keys)
        config_override['license-check'] = config['license-check'].slice(*license_check_keys) if config['license-check']
        @data = @data.merge(config_override)
      end
      @config_path = nil
    end

    def load_logger
      @logger = Bolt::Logger.logger(self)
    end

    def load_file_config(path)
      @config_path = path
      begin
        # This lets us get the actual config values without needing to
        # know the service name
        parsed_hocon_service = Hocon.load(path)[service_name]
        parsed_hocon_license_check = Hocon.load(path)['license-check']
      rescue Hocon::ConfigError => e
        raise "Hocon data in '#{path}' failed to load.\n Error: '#{e.message}'"
      rescue Errno::EACCES
        raise "Your user doesn't have permission to read #{path}"
      end

      raise "Could not find service config at #{path}" if parsed_hocon_service.nil?
      raise "Could not find license-check config at #{path}" if parsed_hocon_service.nil?

      parsed_hocon = parsed_hocon_service.slice(*config_keys)
      parsed_hocon['license-check'] = parsed_hocon_license_check.slice(*license_check_keys)

      @data = @data.merge(parsed_hocon)
    end

    def load_env_config
      raise "load_env_config should be defined in the service class"
    end

    def natural?(num)
      num.is_a?(Integer) && num.positive?
    end

    def validate
      required_keys.each do |k|
        # Handled nested config
        if k.is_a?(Array)
          next unless @data.dig(*k).nil?
        else
          next unless @data[k].nil?
        end
        raise Bolt::ValidationError, "You must configure #{k} in #{@config_path}"
      end

      unless natural?(@data['port'])
        raise Bolt::ValidationError, "Configured 'port' must be a valid integer greater than 0"
      end

      if @data['license-check']
        unless natural?(@data['license-check']['check-interval-seconds'])
          raise Bolt::ValidationError,
                "Configured 'license-check.check-interval-seconds' must be a valid integer greater than 0"
        end

        unless @data['license-check']['retries'].is_a?(Integer)
          raise Bolt::ValidationError, "Configured 'license-check.retries' must be a valid integer"
        end
      end

      ssl_keys.each do |sk|
        unless File.file?(@data[sk]) && File.readable?(@data[sk])
          raise Bolt::ValidationError, "Configured #{sk} must be a valid filepath"
        end
      end

      unless @data['ssl-cipher-suites'].is_a?(Array)
        raise Bolt::ValidationError, "Configured 'ssl-cipher-suites' must be an array of cipher suite names"
      end

      unless @data['allowlist'].nil? || @data['allowlist'].is_a?(Array)
        raise Bolt::ValidationError, "Configured 'allowlist' must be an array of names"
      end
    end

    def load_ssl_data
      @logger.debug("Loading SSL Files")
      @data[:ca_cert] = File.read(@data['ssl-ca-cert'])
                            .scan(/-----BEGIN CERTIFICATE-----.*?-----END CERTIFICATE-----/m).map do |cert|
        OpenSSL::X509::Certificate.new(cert)
      end
      @data[:crl] = File.read(@data['ssl-crl-path'])
                        .scan(/-----BEGIN X509 CRL-----.*?-----END X509 CRL-----/m).map do |crl|
        OpenSSL::X509::CRL.new(crl)
      end
      @data[:cert] = OpenSSL::X509::Certificate.new(File.read(@data['ssl-cert']))
      @data[:private_key] = OpenSSL::PKey::RSA.new(File.read(@data['ssl-key']))
    end

    def [](key)
      @data[key]
    end
  end
end
