# frozen_string_literal: true

require 'bolt/transport/base'

module Bolt
  module Transport
    class Netconf < Base
      def self.options
        %w[
          host port user password
          timeout connect-timeout disconnect-timeout command-timeout
          hello-timeout idle-timeout keepalive-interval
          auto-reconnect reconnect-attempts reconnect-delay
          host-key-check private-key
        ]
      end

      def self.default_options
        {
          'port' => 830,
          'connect-timeout' => 10,
          'disconnect-timeout' => 5,
          'host-key-check' => true
        }
      end

      def provided_features
        ['netconf']
      end

      def initialize
        super
        require 'bolt/transport/local'
        @local_transport = Bolt::Transport::Local.new
      end

      def run_command(target, _command, _options = {}, _position = [])
        unsupported_operation(:run_command, target)
      end

      def run_script(target, _script, _arguments, _options = {}, _position = [])
        unsupported_operation(:run_script, target)
      end

      def run_task(target, task, arguments, options = {}, position = [])
        # Check if puppetlabs-edgeops module is available
        check_netconf_module_availability

        # Build connection parameters from target
        # Keep hyphenated format to match what the Session class expects
        connection_params = {
          'host' => target.host,
          'port' => target.port || 830,
          'user' => target.user,
          'password' => target.password,
          'private-key' => target.options['private-key'],
          'host-key-check' => target.options.fetch('host-key-check', true),
          'connect-timeout' => target.options.fetch('connect-timeout', 10),
          'disconnect-timeout' => target.options.fetch('disconnect-timeout', 5)
        }

        # Add global timeout if specified
        if target.options['timeout']
          connection_params['timeout'] = target.options['timeout']
        end

        # Add optional timeout parameters (keep hyphenated format)
        %w[command-timeout hello-timeout idle-timeout].each do |opt|
          connection_params[opt] = target.options[opt] if target.options[opt]
        end

        # Add keepalive parameter (keep hyphenated format)
        if target.options['keepalive-interval']
          connection_params['keepalive-interval'] = target.options['keepalive-interval']
        end

        # Add reconnection parameters (keep hyphenated format)
        if target.options['auto-reconnect']
          connection_params['auto-reconnect'] = target.options['auto-reconnect']
        end

        if target.options['reconnect-attempts']
          connection_params['reconnect-attempts'] = target.options['reconnect-attempts']
        end

        if target.options['reconnect-delay']
          connection_params['reconnect-delay'] = target.options['reconnect-delay']
        end

        # Remove nil values
        connection_params.compact!

        # Merge task arguments with connection parameters
        # The connection info is passed under '_target' key
        task_args = arguments.merge('_target' => connection_params)

        # Create a local target to run the task on
        require 'bolt/target'
        # Use the target's inventory to preserve module paths
        local_target = Bolt::Target.from_hash({ 'uri' => 'localhost', 'config' => { 'transport' => 'local' } },
                                              target.inventory)

        # Run the task locally with the merged arguments
        result = @local_transport.run_task(local_target, task, task_args, options, position)

        # Wrap the result to use the original target instead of localhost
        if result.error_hash
          Bolt::Result.new(target, error: result.error_hash, action: result.action, object: result.object)
        else
          Bolt::Result.new(target, value: result.value, action: result.action, object: result.object)
        end
      end

      def upload(target, _source, _destination, _options = {})
        unsupported_operation(:upload, target)
      end

      def download(target, _source, _destination, _options = {})
        unsupported_operation(:download, target)
      end

      def connected?(target)
        # Perform a simple TCP connectivity check
        require 'socket'
        require 'timeout'

        host = target.host
        port = target.port || 830
        timeout = target.options.fetch('connect-timeout', 10)

        begin
          Timeout.timeout(timeout) do
            socket = TCPSocket.new(host, port)
            socket.close
            true
          end
        rescue Timeout::Error
          logger.debug("Connection timeout to #{host}:#{port}")
          false
        rescue Errno::ECONNREFUSED
          logger.debug("Connection refused to #{host}:#{port}")
          false
        rescue SocketError => e
          logger.debug("Socket error connecting to #{host}:#{port}: #{e.message}")
          false
        rescue StandardError => e
          logger.debug("Error checking connection to #{host}:#{port}: #{e.message}")
          false
        end
      end

      def with_connection(_target)
        # NETCONF transport doesn't maintain persistent connections
        # Tasks handle their own connections via NetconfSession
        yield(:noop)
      end

      private

      def check_netconf_module_availability
        # Only check once per transport instance
        return if @module_check_done
        @module_check_done = true

        # Try to load the puppetlabs-edgeops module (netconf client)
        begin
          require 'puppet_x/puppetlabs/netconf/client'
          # Module is available, no action needed
        rescue LoadError => e
          # Module is not available, log a helpful message
          logger = Bolt::Logger.logger(self)
          logger.debug("LoadError when trying to load netconf module: #{e.message}")
          logger.debug("Current LOAD_PATH: #{$LOAD_PATH.grep(/modules/).join(', ')}")
          logger.warn("The puppetlabs-edgeops module is not available. To use Puppet's NETCONF client " \
                      "implementation, ensure you have the appropriate license entitlement and the " \
                      "puppetlabs-edgeops module is installed.")
        end
      end

      def unsupported_operation(operation, target)
        # Return a failed result instead of raising an exception
        Bolt::Result.new(
          target,
          error: {
            'msg' => "The NETCONF transport does not support #{operation.to_s.tr('_', ' ')} operations. " \
                     "Only 'run_task' is supported for NETCONF devices.",
            'kind' => 'bolt/transport-unsupported'
          }
        )
      end
    end
  end
end
