#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'open3'
require 'fileutils'

class GetOldPostgresInfo < EnterpriseTaskHelper
  def get_dirs(postgres_version_info, current_version_dirs, current_installed_version)
    dirs_to_remove = []
    postgres_dirs = postgres_version_info['tablespaces'] << postgres_version_info['data_dir'] << postgres_version_info['app_dir']
    version_strs = [current_installed_version, "PG_#{current_installed_version}"]
    postgres_dirs.select { |dir| !dir.to_s.empty? }.each do |dir|
      next if !Dir.exist?(dir)
      # Extra precaution to make sure we are not deleting current version
      basename = File.basename(dir)
      if !current_version_dirs.include?(dir) && !version_strs.include?(basename)
        dirs_to_remove << dir
      end
    end
  end

  def sort_packages(packages)
    # Sort packages according to the dependency order, otherwise uninstall will fail
    sort_order = ['pglogical', 'pgrepack', 'contrib', 'server']
    sorted_packages = []
    sort_order.each do |item|
      packages.delete_if do |pkg|
        if pkg.match?(/#{item}/)
          sorted_packages << pkg
          true
        end
      end
    end
    sorted_packages += packages
  end

  def get_packages(packages, current_postgres_version)
    pkgs_to_remove = []
    sorted_pkgs = sort_packages(packages.reject { |pkg| pkg.to_s.empty? || pkg.include?(current_postgres_version) })
    sorted_pkgs.each do |pkg|
      # Doing a double check to be extra careful
      pkgs_to_remove <<  pkg if !pkg.include?(current_postgres_version)
    end
    pkgs_to_remove
  end

  def get_old_postgresql_info(certname)
    old_packages = []
    old_dirs = []
    old_versions = []
    facter_pg_json, _stderr, _status = Open3.capture3('/opt/puppetlabs/bin/facter -pj "pe_postgresql_info"')[0].strip
    pe_postgresql_info = JSON.parse(facter_pg_json)['pe_postgresql_info']
    raise EnterpriseTaskHelper::Error.new("facter command to get pe_postgresql_info failed on #{certname}", 'puppetlabs.get-old-postgres-info/facter-pe_postgresql_info-failed', 'output' => facter_pg_json) if pe_postgresql_info.nil? || pe_postgresql_info.empty?
    postgres_versions = pe_postgresql_info['versions'].keys
    postgres_versions_data = pe_postgresql_info['versions']
    current_postgres_version = pe_postgresql_info['installed_server_version']
    installed_postgres_packages = pe_postgresql_info['installed_packages'].keys
    current_postgres_info = postgres_versions_data[current_postgres_version]
    current_postgres_dirs = current_postgres_info['tablespaces'] << current_postgres_info['data_dir'] << current_postgres_info['app_dir']
    if current_postgres_version.nil? || current_postgres_version.empty? || current_postgres_dirs.nil? || current_postgres_dirs.empty?
      raise EnterpriseTaskHelper::Error.new("Unable to get current postgresql information  on #{certname}", 'puppetlabs.get-old-postgres-info/facter-pe_postgresql_info-failed', 'output' => facter_pg_json)
    end

    if postgres_versions && !postgres_versions.empty?
      postgres_versions.each do |pg_version|
        next if pg_version == current_postgres_version
        postgres_version_info = postgres_versions_data[pg_version]
        packages = []
        # pe-postgres 9.4 package names does not include version number, so add them directly here
        if pg_version == '9.4'
          packages += ['pe-postgresql-pglogical', 'pe-postgresql-contrib', 'pe-postgresql-server', 'pe-postgresql']
        else
          version = pg_version.gsub('.', '')
          pkg_str = 'pe-postgresql' + version
          installed_postgres_packages.each do |pkg|
            if pkg.match?(/#{pkg_str}/)
              packages.push(pkg.to_s.strip.chomp)
            end
          end
        end
        old_packages.concat(get_packages(packages, current_postgres_version)) if !packages.empty?
        if postgres_version_info && !postgres_version_info.empty?
          old_dirs.concat(get_dirs(postgres_version_info, current_postgres_dirs, current_postgres_version))
        end
        old_versions << pg_version
      end
    end
    { 'versions' => old_versions, 'packages' => old_packages, 'dirs' => old_dirs }
  end

  def task(_)
    certname, _stderr, _status = Open3.capture3('/opt/puppetlabs/bin/puppet config print certname --section agent')[0].strip
    info = get_old_postgresql_info(certname)
    result = { info: info }
    result.to_json
  end
end

GetOldPostgresInfo.run if __FILE__ == $PROGRAM_NAME
