# frozen_string_literal: true

require_relative '../../../bolt/error'
require_relative '../../../bolt/config/transport/base'
require_relative '../../../bolt/util'

module Bolt
  class Config
    module Transport
      class Netconf < Base
        OPTIONS = %w[
          host
          port
          user
          password
          timeout
          connect-timeout
          disconnect-timeout
          command-timeout
          hello-timeout
          idle-timeout
          keepalive-interval
          auto-reconnect
          reconnect-attempts
          reconnect-delay
          host-key-check
          private-key
        ].freeze

        DEFAULTS = {
          'port' => 830,
          'connect-timeout' => 10,
          'disconnect-timeout' => 5,
          'host-key-check' => true
        }.freeze

        private def defaults
          DEFAULTS
        end

        private def filter(unfiltered)
          unfiltered.slice(*OPTIONS)
        end

        private def validate
          super

          if @config['private-key'] && @config['password']
            raise Bolt::ValidationError,
                  "Only one of 'password' or 'private-key' can be specified for NETCONF transport"
          end

          if (key_opt = @config['private-key'])
            if key_opt.instance_of?(String)
              @config['private-key'] = File.expand_path(key_opt, @project)
              Bolt::Util.validate_file('ssh key', @config['private-key'])
            elsif key_opt.instance_of?(Hash)
              unless key_opt['key-data']
                raise Bolt::ValidationError, "private-key hash must include 'key-data'"
              end
            else
              raise Bolt::ValidationError, "private-key must be a string or hash, not #{key_opt.class}"
            end
          end
        end
      end
    end
  end
end
