# frozen_string_literal: true

require 'hocon'
require 'concurrent/atomic/read_write_lock'
require 'concurrent/timer_task'
require 'pe_bolt_server/base_config'
require 'bolt/error'
require 'bolt/logger'
require 'pe_bolt_server/service/http_client'
require 'logging'

module PEBoltServer
  # It would be lovely if we didnt' need to do this, but nothing in
  # concurrent-ruby appears to use a readwritelock kind of locking
  # mechanism on data. Which means all the read threads wait on each
  # other which would be bonkers for data that only updates every
  # few minutes.
  #
  # So instead create our own atomic object for reading/writing
  # entitlements data
  class EntitlementsData
    def initialize
      @entitlements = []
      @lock = Concurrent::ReadWriteLock.new
      @timer_task = nil
    end

    def load_logger(logger)
      # We don't need the user to understand that there's an entitlements
      # object that's keeping things from them, so use the logger from
      # the config class instead
      @logger = logger
    end

    # Start a Concurrent::TimerTask to periodically check the license
    # using the license API and perform a thread safe update to the
    # entitlements based on the result.
    #
    # Interval between executions defined by [`license-check']['check-interval-seconds`]
    # and the number of retries for each execution defined by
    # ['license-check']['retries']
    def start_entitlements_check(client, config)
      @timer_task = Concurrent::TimerTask.new(execution_interval: config['license-check']['check-interval-seconds'],
                                              run_now: true,
                                              interval_type: :fixed_rate) {
        @logger.debug("Loading entitlements data")
        i = 0
        success = false
        while i <= config['license-check']['retries']
          i += 1
          begin
            license_data, http_code = client.get_with_cert(config['license-check']['license-api'])
            if http_code == 200
              entitlements_found = update(license_data)
              @logger.debug("Entitlements detected: #{entitlements_found.join(' ')}")
              success = true
              break
            else
              @logger.debug("License check returned #{http_code}, not 200, retrying...")
            end
          rescue StandardError => e
            @logger.debug("License check threw exception: #{e.message}, retrying...")
          end
        end
        unless success
          @logger.warn("Failed to load entitlements data, advanced PE content will be unavailable.")
        end
      }
      @timer_task.execute
    end

    def stop_entitlements_check
      @timer_task&.shutdown
    end

    # Thread safe read of the entitlements data
    def read
      @lock.acquire_read_lock
      @entitlements
    ensure
      @lock.release_read_lock
    end

    private

    def update(license_data)
      @lock.acquire_write_lock
      updated_entitlements = []
      if license_data['entitlements'].include?("EDGE")
        updated_entitlements << 'edge'
      end
      @entitlements = updated_entitlements
      updated_entitlements
    ensure
      @lock.release_write_lock
    end
  end

  class Config < PEBoltServer::BaseConfig
    def config_keys
      super + %w[concurrency cache-dir file-server-conn-timeout
                 file-server-uri environments-codedir
                 environmentpath basemodulepath builtin-content-dir]
    end

    def env_keys
      super + %w[concurrency file-server-conn-timeout file-server-uri]
    end

    def int_keys
      %w[concurrency file-server-conn-timeout]
    end

    def defaults
      super.merge(
        'port' => 62658,
        'concurrency' => 100,
        'cache-dir' => "/opt/puppetlabs/server/data/bolt-server/cache",
        'file-server-conn-timeout' => 120,
        'entitlements' => EntitlementsData.new
      )
    end

    def required_keys
      super + %w[file-server-uri]
    end

    def service_name
      'bolt-server'
    end

    def load_env_config
      env_keys.each do |key|
        transformed_key = "BOLT_#{key.tr('-', '_').upcase}"
        next unless ENV.key?(transformed_key)
        @data[key] = if int_keys.include?(key)
                       ENV[transformed_key].to_i
                     else
                       ENV.fetch(transformed_key, nil)
                     end
      end
    end

    def load_entitlements
      client = PEBoltServer::Service::HttpClient.new(@data)
      @data['entitlements'].load_logger(@logger)
      @data['entitlements'].start_entitlements_check(client, @data)
    rescue StandardError => e
      @logger.debug("Starting entitlements check failed with #{e.message}")
    end

    def validate
      super

      unless natural?(@data['concurrency'])
        raise Bolt::ValidationError, "Configured 'concurrency' must be a positive integer"
      end

      unless natural?(@data['file-server-conn-timeout'])
        raise Bolt::ValidationError, "Configured 'file-server-conn-timeout' must be a positive integer"
      end
    end
  end
end
