# frozen_string_literal: true

# Repeat the block until it returns a truthy value. Returns the value.
Puppet::Functions.create_function(:'ctrl::do_until') do
  # @param options A hash of additional options.
  # @param block The code block to repeat.
  # @option options [Numeric] limit The number of times to repeat the block.
  # @option options [Numeric] interval The number of seconds to wait before repeating the block.
  # @return [nil]
  # @example Run a task until it succeeds
  #   ctrl::do_until() || {
  #     run_task('test', $target, '_catch_errors' => true).ok()
  #   }
  # @example Run a task until it succeeds or fails 10 times
  #   ctrl::do_until('limit' => 10) || {
  #     run_task('test', $target, '_catch_errors' => true).ok()
  #   }
  # @example Run a task and wait 10 seconds before running it again
  #   ctrl::do_until('interval' => 10) || {
  #     run_task('test', $target, '_catch_errors' => true).ok()
  #   }
  dispatch :do_until do
    optional_param 'Hash[String[1], Any]', :options
    block_param
  end

  def do_until(options = {})
    # Send Analytics Report
    Puppet.lookup(:bolt_executor) {}&.report_function_call(self.class.name)

    limit = options['limit'] || 0
    interval = options['interval']
    # The reference to the active thread needs to be created *before* the
    # call to sleep (and any subsequent changes like interrupts that the
    # sleep makes on the thread). Otherwise if you try to pull the active
    # thread after the sleep has completed JRuby appears to give you a
    # reference to a thread not connected to what just happened.
    #
    # I have suspicions this is due to https://github.com/jruby/jruby/blob/163d0aa13f9299925e754d8b0b07581e8bdac832/core/src/main/java/org/jruby/javasupport/util/ObjectProxyCache.java#L13L31
    # but I'm not 100% positive about that. Regardless, as long as the
    # thread reference is created before the sleep everything appears to
    # work well, so I'm not digging any more.
    #
    #                             - Sean McDonald 10/12/2022
    java_thread = nil
    if defined?(JRuby)
      java_thread = JRuby.reference(Thread.current).native_thread
    end

    i = 0
    until (x = yield)
      i += 1
      break if limit != 0 && i >= limit
      Kernel.sleep(interval) if interval

      if java_thread&.interrupted?
        raise "do_until interrupted, stopping"
      end
    end
    x
  end
end
