require 'hocon'
require 'date'
require 'puppet'
require 'puppet/application'
require 'puppet/network/http_pool'
require 'pe_license'
require 'pe_license/status'
require 'pe_license/request_license'

class Puppet::Application::License < Puppet::Application
  run_mode :master

  attr_accessor :license, :active_nodes
  attr_reader :log_level, :error, :to, :nodes, :start_date, :end_date, :id, :type

  def summary
    _("Display licensing information")
  end

  def help
      <<-HELP
puppet-license(8) -- #{summary}
SYNOPSIS
--------
Prints active node count, license count, and license status.
USAGE
-----
puppet license
COPYRIGHT
---------
(c)2024 Puppet, Inc., a Perforce company. All rights reserved.
      HELP
  end

  def setup
    Puppet::Util::Log.newdestination :console
    @buffer = []
    @log_level = :notice

    @url   = "https://buy.puppet.com/"
    @email = "sales@puppet.com"
    @phone = "+1 (877) 575-9775"
  end

  def license_key
    '/etc/puppetlabs/license.key'
  end

  def services_conf_path
    '/etc/puppetlabs/client-tools/services.conf'
  end

  def get_hac_url
    if File.exist?(services_conf_path)
      begin
        config = Hocon.load(services_conf_path)
        hac = config['services'].detect { |s| s['status_key'] === 'pe-host-action-collector' }

        hac['url']
      rescue Errno::ENOENT => e
        #
        exit 2
      end
    else
      # config file missing, throw
      Puppet.err(_("Unable to load services.conf"));exit 1
    end
  end

  def hac_url
    @hac_url ||= get_hac_url
  end

  def days_remaining(date=Date.today)
    if @type == 'expired'
      0
    elsif @end_date
      (Date.parse(end_date) - date).to_i
    else
      # If there's no end date, it's valid indefinitely
      Float::INFINITY
    end
  end

  def days_exceeded(date=Date.today)
    if @type == 'expired'
      (date - Date.parse(end_date)).to_i
    else
      0
    end
  end

  def retreive_license
    uri = URI(hac_url)
    http = Puppet.runtime[:http]
    result = PELicense::RequestLicense.get_current(http, uri)

    @to = result['customer']
    @nodes = result['node_count']
    @start_date = result['start_date']
    @end_date = result['end_date']
    @uuid = result['id']
    @type = result['type']
  end

  def retrieve_status
    uri = URI(hac_url)
    http = Puppet.runtime[:http]
    result = PELicense::Status.retrieve(http, uri)
    @active_nodes = result.nodes
  end

  def collect_status_messages
    messages = []

    messages << :node_count

    if @start_date or @end_date
      messages << :start_date if @start_date
      messages << :end_date if @end_date
      messages << :blank
    end

    messages << :licensee if @to

    messages << :complimentary if @type == 'none'

    if @active_nodes > @nodes
      set_error
      messages << :exceeded_node_count
    end


    if @type == 'expired'
      set_error
      messages << :expired
    elsif days_remaining <= 30
      set_warning
      messages << :expiring_soon
    end

    messages << :contact_details

    messages
  end

  def main
    retreive_license

    retrieve_status

    messages = collect_status_messages

    messages.each do |msg|
      send "append_#{msg}"
    end

    # (#12660) To avoid sending a massive string to the logging system, we send
    # the message line by line.  Sending a large string is a problem because it
    # gets truncated by syslog.
    @buffer.each do |line|
      Puppet.send @log_level, line
    end

    exit 0
  end

  def set_error
    @log_level = :alert
    @error = true
  end

  def set_warning
    @log_level = :warning unless @log_level == :alert
  end

  private

  def pluralize(noun, count)
    count == 1 ? noun : "#{noun}s"
  end

  def append(msg)
    @buffer << msg
  end

  def complain_invalid_license(reason)
    Puppet.crit <<-INVALID_LICENSE.gsub('  '*3, '')
      Your License is incorrectly formatted or corrupted!
      #{reason}

      Please contact Puppet to correct this problem: email #{@email}
      or visit our website at:
      #{@url}
      INVALID_LICENSE
  end

  def append_node_count
    append "You have #{@active_nodes == 0 ? 'no' : @active_nodes} active #{pluralize 'node', @active_nodes}."
    append "You are currently licensed for #{@nodes} active #{pluralize 'node', @nodes}."
  end

  def append_start_date
    append "Your support and maintenance agreement starts on #{@start_date}"
  end

  def append_end_date
    append "Your support and maintenance agreement ends on #{@end_date}"
  end

  def append_blank
    append ""
  end

  def append_licensee
    append "This Puppet Enterprise distribution is licensed to:"
    append @to
    append ""
  end

  def append_complimentary
    append "You are using a complimentary ten node license provided free by Puppet."
    append ""
    append "Your complimentary license does not include Support & Maintenance. If you"
    append "would like to obtain official Support & Maintenance, please contact us."
    append ""
  end

  def append_exceeded_node_count
    error = true

    over = @active_nodes - @nodes

    append "You have exceeded the node count in your license by #{over} active #{pluralize 'node', over}!"
    append ""
    append "You should make sure that all inactive nodes are deactivated in PuppetDB."
    append "You can use the command `puppet node status <node>` to check the status of a"
    append "node, and `puppet node deactivate <node>` to deactivate unused nodes."
    append ""
    append "Please contact Puppet to obtain additional licenses to bring your network"
    append "back in compliance."
    append ""
  end

  def append_expired
    append "Your Support & Maintenance agreement expired on #{@end_date}!"
    append "You have run for #{days_exceeded} #{pluralize 'day', days_exceeded} without a support agreement; please contact"
    append "Puppet urgently to renew your Support & Maintenance agreement."
    append ""
  end

  def append_expiring_soon
    append "Your Support & Maintenance term expires on #{@end_date}."
    append "You have #{days_remaining} #{pluralize 'day', days_remaining} remaining under that agreement; please contact"
    append "Puppet to renew your Support & Maintenance agreement:"
    append ""
  end

  def append_contact_details
    append "You can reach Puppet for sales, support, or maintenance agreements"
    append "by email to #{@email}, on #{@phone}, or visit us on"
    append "the web at #{@url} for more information."
  end
end
