require 'json'
require 'fileutils'
require 'logger'
require 'tmpdir'
require 'pe_backup_tools/backup'
require 'pe_backup_tools/utils'
require 'pe_backup_tools/utils/analytics_client'
require 'pe_backup_tools/utils/conf_files'
require 'pe_backup_tools/utils/classifier'
require 'pe_backup_tools/utils/mixins'
require 'pe_backup_tools/utils/puppetdb'
require 'puppet'
require 'hocon'
require 'hocon/parser/config_document_factory'
require 'hocon/config_value_factory'
require 'hocon/config_error'

module PeBackupTools
  # This module provides the restore functionality, allowing users
  # to restore from both local and s3 backups.
  # restore: Restores puppet server from backup and configures puppet server w/ current server config
  module Restore
    extend PeBackupTools::Utils::Mixins

    def self.gpg_decrypt(backup_path)
      archive_path = File.basename(backup_path, '.gpg')
      command = "gpg --yes -o #{archive_path} -d #{backup_path}"
      output, status = run_command(command)
      if status.exitstatus.zero? && File.exist?(archive_path)
        return archive_path
      else
        error(_('Failed to decrypt backup file %{backup_path}. Make sure gpg is installed and your gpg private key is imported on this host') % { backup_path: backup_path }, command, output)
      end
    end

    def self.get_backup_metadata(archive_path)
      command = "tar --to-stdout -xf #{archive_path} backup_metadata.json"
      output, status = run_command(command)
      if !status.exitstatus.zero? || output.strip.empty?
        error(_('Failed to find backup metadata'), output, status)
      end
      JSON.parse(output.strip, symbolize_names: true)
    end

    def self.unpack_tarfile(archive_path, possible_scope, scope_config, tempdir)
      included_paths_args = ''
      unless possible_scope.sort == scope_config[:scope].sort
        # If we're not using the whole scope of the backup, we only want to grab
        # the relevant files from the archive
        included_paths = scope_config[:included_paths] + scope_config[:included_dbs].map { |db| "#{db}.bin" }
        included_paths_args = included_paths.map(&:to_s).join(' ')
      end
      command = "tar -P -C '#{tempdir}' -xvf #{archive_path} #{included_paths_args}"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Failed to unpack PE backup'), command, output)
      end
    end

    def self.remove_puppet_ssl_dir
      puppet_ssl_dir = '/etc/puppetlabs/puppet/ssl'
      FileUtils.remove_entry(puppet_ssl_dir) if File.directory?(puppet_ssl_dir)
    end

    def self.remove_puppetserver_ca_dir
      puppetserver_ca_dir = '/etc/puppetlabs/puppetserver/ca'
      FileUtils.remove_entry(puppetserver_ca_dir) if File.directory?(puppetserver_ca_dir)
    end

    def self.remove_old_certs(hostname)
      signed = "/etc/puppetlabs/puppet/ssl/ca/signed/#{hostname}.pem"
      request = "/etc/puppetlabs/puppet/ssl/ca/requests/#{hostname}.pem"
      FileUtils.rm_f(signed) if File.exist?(signed)
      FileUtils.rm_f(request) if File.exist?(request)
    end

    def self.stop_pe_services(options)
      %w[
        puppet
        pe-puppetdb
        pe-puppetserver
        pe-orchestration-services
        pe-console-services
        pe-host-action-collector
        pe-patching-service
        pe-infra-assistant
        pe-workflow
      ].each do |svc|
        debugflag = options[:debug] ? ' --debug' : ''
        command = "/opt/puppetlabs/bin/puppet resource service #{svc} ensure=stopped#{debugflag}"
        output, status = run_command(command)
        unless status.exitstatus.zero?
          error(_('Failed to stop service %{svc}') % { svc: svc }, command, output)
        end
      end
    end

    def self.migrate_puppet_conf(certname, options)
      puppet_cmd = '/opt/puppetlabs/puppet/bin/puppet'
      debugflag = options[:debug] ? ' --debug' : ''

      commands = [
        "#{puppet_cmd} config set --section=main certname '#{certname}'#{debugflag}",
        "#{puppet_cmd} config set --section=main server '#{certname}'#{debugflag}",
        "#{puppet_cmd} config set --section=master certname '#{certname}'#{debugflag}",
        "#{puppet_cmd} config set --section=agent server_list '#{certname}:8140'#{debugflag}",
      ]
      commands.each do |c|
        output, status = run_command(c)
        unless status.exitstatus.zero?
          error(_('Failed to migrate puppet.conf to new certname value'), c, output)
        end
      end
    end

    def self.migrate_user_data_conf(certname)
      migrate_enterprise_conf_file('/etc/puppetlabs/enterprise/conf.d/user_data.conf', certname)
    end

    def self.migrate_pe_conf(certname)
      migrate_enterprise_conf_file('/etc/puppetlabs/enterprise/conf.d/pe.conf', certname)
    end

    def self.migrate_enterprise_conf_file(pe_conf_path, certname)
      return if !File.exist?(pe_conf_path)

      pe_conf = Hocon::Parser::ConfigDocumentFactory.parse_file(pe_conf_path)
      new_conf_values = {
        'puppet_enterprise::profile::agent::primary_uris' => ["https://#{certname}:8140"]
      }
      pe_conf = PeBackupTools::Utils::ConfFiles.hocon_merge(pe_conf, new_conf_values)
      # PE-37369: We should really change how this works instead of playing whack-a-mole
      # with various parameters. But when this was fixed for classifier_host, we couldn't
      # think of a good way to do this without introducing more edge cases (e.g. gsubbing
      # the whole file, trying to inspect every different kind of data structure where
      # the certname might exist, etc.)
      %w[
        puppet_enterprise::puppet_master_host
        puppet_enterprise::puppetdb_host
        puppet_enterprise::console_host
        puppet_enterprise::database_host
        puppet_enterprise::certificate_authority_host
        puppet_enterprise::pcp_broker_host
        pe_repo::master
        puppet_enterprise::profile::master::classifier_host
      ].each do |setting|
        pe_conf = PeBackupTools::Utils::ConfFiles.hocon_assoc(pe_conf, setting, certname.to_s)
      end
      File.write(pe_conf_path, pe_conf.render)
    rescue Hocon::ConfigError::ConfigParseError => e
      error(_('Failed to migrate pe.conf to new certname value: %{error}.') % { error: e.message })
    end

    # MEEP uses the pe_version file to determine whether or not to run certain
    # upgrade logic but the pe_version file is managed by a PE package so we
    # don't want to overwrite it completely. The two functions below backup the
    # package version of the file before the restore, let the restore happen
    # (running any upgrade logic), and restores the package's version of the
    # file.
    def self.backup_pe_version_file(dir)
      FileUtils.cp('/opt/puppetlabs/server/pe_version', "#{dir}/pe_version")
    end

    def self.restore_pe_version_file(dir)
      FileUtils.cp("#{dir}/pe_version", '/opt/puppetlabs/server/pe_version') if File.exist?("#{dir}/pe_version")
    end

    def self.restore_database(db, tempdir)
      # Drop pglogical schema if present
      # When restoring onto a replica that has the pglogical schema already, not dropping it here will cause failures
      tmp_db_dump = "#{tempdir}/#{db}.bin"
      commands = [
        [
          "su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP SCHEMA IF EXISTS pglogical CASCADE;'\"",
          _('Failed to drop pglogical schema for %{db}.') % { db: db },
        ],
        [
          "su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/psql -d '#{db}' -c 'DROP SCHEMA public CASCADE; CREATE SCHEMA public;'\"",
          _('Failed to drop tables in PE database %{db}.') % { db: db },
        ],
        [
          "su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/pg_restore #{tmp_db_dump} -Fd -j4 --dbname=#{db}\"",
          _('Failed to restore PE database %{db}.') % { db: db },
        ],
        [
          "su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP SCHEMA IF EXISTS pglogical CASCADE;'\"",
          _('Failed to drop pglogical schema for %{db}.') % { db: db },
        ],
        [
          "su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP EXTENSION IF EXISTS pglogical CASCADE;'\"",
          _('Failed to drop pglogical extension for %{db}.') % { db: db }
        ],
      ]
      commands.each do |cmd, err|
        output, status = run_command(cmd)
        error(err, cmd, output) unless status.exitstatus.zero?
      end
    ensure
      FileUtils.rm_f(tmp_db_dump)
    end

    def self.compute_total_steps(scope, new_certname)
      steps = 13
      steps -= 1 unless scope.include?('certs')
      steps -= 1 unless scope.include?('puppetdb')
      steps -= 2 unless scope.include?('config') && new_certname
      steps -= 5 unless scope.include?('config')
      steps -= 1 unless scope.include?('puppetdb') && new_certname
      steps
    end

    def self.update_pe_classification(backup_certname, restore_certname)
      cert_conf = PeBackupTools::Utils.get_cert_conf(restore_certname)
      backup_groups = PeBackupTools::Utils::Classifier.get_classification(backup_certname, restore_certname, cert_conf)['groups']
      restore_groups = PeBackupTools::Utils::Classifier.get_classification(restore_certname, restore_certname, cert_conf)['groups']
      backup_only_groups = backup_groups - restore_groups
      backup_only_groups.each do |id|
        PeBackupTools::Utils::Classifier.pin_nodes_to_group(id, [restore_certname], restore_certname, cert_conf)
      end
      PeBackupTools::Utils::Classifier.unpin_from_all([backup_certname], restore_certname, cert_conf)

      # Check for HA replica
      replica_hostnames = nil
      pe_ha_ng = PeBackupTools::Utils::Classifier.get_node_group('PE HA Master', restore_certname, cert_conf)
      if pe_ha_ng
        replica_hostnames = pe_ha_ng['classes']['puppet_enterprise::profile::database']['replica_hostnames'] - [restore_certname]
      end
      # Replace old primary with new primary and remove replica from parameter values
      PeBackupTools::Utils::Classifier.update_node_group_params(['PE Infrastructure Agent', 'PE Agent'],
                                                                %w[pcp_broker_list server_list primary_uris],
                                                                'puppet_enterprise::profile::agent',
                                                                backup_certname,
                                                                restore_certname,
                                                                replica_hostnames,
                                                                restore_certname,
                                                                cert_conf)
      PeBackupTools::Utils::Classifier.update_node_group_params(['PE Master'],
                                                                %w[puppetdb_host classifier_host classifier_client_certname],
                                                                'puppet_enterprise::profile::master',
                                                                backup_certname,
                                                                restore_certname,
                                                                replica_hostnames,
                                                                restore_certname,
                                                                cert_conf)
      # Remove extraneous port value from puppetdb_port and classifier_port
      pe_primary_ng = PeBackupTools::Utils::Classifier.get_node_group('PE Master', restore_certname, cert_conf)
      if pe_primary_ng
        %w[puppetdb_port classifier_port].each do |parameter|
          if pe_primary_ng['classes']['puppet_enterprise::profile::master'][parameter]
            pe_primary_ng['classes']['puppet_enterprise::profile::master'][parameter].uniq!
          end
        end
        PeBackupTools::Utils::Classifier.update_node_group(pe_primary_ng, restore_certname, cert_conf)
      end
      PeBackupTools::Utils::Classifier.set_node_group_param('PE Infrastructure', 'replicating', 'puppet_enterprise', nil, restore_certname, cert_conf)
      # returns hostname if replicas are present in backup config
      return replica_hostnames if replica_hostnames && !replica_hostnames.empty?
    end

    def self.pre_restore_interaction
      puts _('Continue with restore? Y/n')
      attempts = 0
      answer = nil
      while attempts < 3
        answer = STDIN.gets.chomp.downcase
        return if ['y', 'yes', ''].include? answer
        error(_('Restore halted by confirmation check.')) if %w[n no].include? answer
        attempts += 1
      end
      error(_('Unknown response received: %{answer}.') % { answer: answer })
    end

    def self.disk_space_interaction(space_needed, free_space, mount_point)
      if free_space && free_space > space_needed
        Puppet.info(_('Disk space available.'))
        return
      end
      humanized_space_needed = PeBackupTools::Utils.humanize_size(space_needed)
      if free_space
        Puppet.warning(_('There is not enough space to restore. Estimated restore size is %{space_needed}, and %{mount_point} has %{free_space} of space available. Use the --force option to proceed without a disk space check.') % { space_needed: humanized_space_needed, mount_point: mount_point, free_space: PeBackupTools::Utils.humanize_size(free_space) })
        puts _('Would you like to proceed with estimated insufficient space? Y/n')
      else
        Puppet.warning(_('Unable to determine if there is enough space to restore. Estimated restore size is %{space_needed}. Use the --force option to proceed without a disk space check.') % { space_needed: humanized_space_needed })
        puts _('Would you like to proceed without checking the available disk space? Y/n')
      end
      attempts = 0
      answer = nil
      while attempts < 3
        answer = STDIN.gets.chomp.downcase
        return if ['y', 'yes', ''].include? answer
        error(_('Restore halted by disk space check.')) if %w[n no].include? answer
        attempts += 1
      end
      error(_('Unknown response received: %{answer}.') % { answer: answer })
    end

    def self.estimate_restore_size(backup_size, scopes)
      scopes.inject(0) { |sum, scope| sum + backup_size[scope.to_sym][:total_size] }
    end

    def self.get_tempdir(options)
      File.expand_path(options[:tempdir] || ENV['TMPDIR'] || '/tmp')
    end

    def self.get_compilers(backup_certname, restore_certname, replica_certname, options = {})
      cert_conf = PeBackupTools::Utils.get_cert_conf(restore_certname)
      puppetdb_port = options[:puppetdb_port] || 8081
      begin
        nodes = PeBackupTools::Utils::PuppetDB.get_nodes_with_class('Puppet_enterprise::Profile::Master', restore_certname, cert_conf, puppetdb_port)
        nodes.reject { |x| [backup_certname, restore_certname, replica_certname].flatten.compact.include?(x) }
      rescue PeBackupTools::Utils::PuppetDB::PuppetDBQueryError => e
        Puppet.err("Could not query PuppetDB to determine if any compilers are present in the infrastructure. Error: #{e.message}")
        []
      end
    end

    def self.puppet_infrastructure_configure(environment, options)
      debugflag = options[:debug] ? ' --debug' : ''
      command = "/opt/puppetlabs/bin/puppet-infrastructure configure --detailed-exitcodes --no-noop --no-recover --pe-environment #{environment}#{debugflag}"
      output, status = run_command(command)
      case status.exitstatus
      when 0, 2
        return
      when 1
        error(_('Failed to configure PE.'), command, output)
      when 4, 6
        error(_('Configuring PE completed but with failures.'), command, output)
      else
        error(_('Received unknown exit code while configuring PE.'), command, output)
      end
    end

    def self.puppet_node_deactivate(certname, options)
      debugflag = options[:debug] ? ' --debug' : ''
      command = "/opt/puppetlabs/puppet/bin/puppet node deactivate #{certname}#{debugflag}"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Could not deactivate the old primary node.'), command, output)
      end
    end

    def self.restore(options)
      timestamp = Time.now.utc.strftime('pe_restore-%Y-%m-%d_%H.%M.%S_UTC')
      logdir = File.expand_path(options[:logdir] || '/var/log/puppetlabs/pe-backup-tools')
      logfilename = timestamp + '.log'
      logfile = File.join(logdir, logfilename)
      PeBackupTools::Utils.configure_logging(logfile, options)

      base_tempdir = get_tempdir(options)
      FileUtils.mkdir_p(base_tempdir) unless Dir.exist?(base_tempdir)
      tempdir = Dir.mktmpdir('pe_restore', base_tempdir)
      # pe-postgres needs to be able to access tempdir, and since we're
      # running as root, we've already got access.
      FileUtils.chown('pe-postgres', 'pe-postgres', tempdir)
      archive_path = options[:backup]

      error(_('Specified local backup file does not exist.')) unless File.readable?(archive_path)

      if File.extname(archive_path) == '.gpg'
        decrypted_file = gpg_decrypt(archive_path)
        archive_path = decrypted_file
      end

      Puppet.info(_('Restoring backup file %{archive_path}') % { archive_path: archive_path })

      Puppet.info(_('Retrieving metadata from backup file'))
      backup_metadata = get_backup_metadata(archive_path)
      backup_scope = backup_metadata[:scope_config][:scope]
      if options[:scope] == 'all'
        options[:scope] = backup_scope
      else
        invalid = options[:scope].reject { |scope| backup_scope.include?(scope) }
        unless invalid.empty?
          error(
            n_('Backup only contains scope: %{valid}', 'Backup only contains scopes: %{valid}.', backup_scope.length) % { valid: backup_scope.join(', ').chomp } + "\n" + n_('Invalid scope option: %{invalid}.', 'Invalid scope options: %{invalid}.', invalid.length) % { invalid: invalid.join(', ').chomp }
          )
        end
      end

      scope_config = PeBackupTools::Backup.generate_scope_config(options)
      timings = {}
      restore_certname = Puppet[:certname]
      backup_certname = backup_metadata[:certname]
      step_printer = PeBackupTools::Utils::StepPrinter.new(compute_total_steps(scope_config[:scope], backup_certname != restore_certname))

      backup_pe_version_file(tempdir)

      space_needed = estimate_restore_size(backup_metadata[:uncompressed_size], scope_config[:scope])
      humanized_space_needed = PeBackupTools::Utils.humanize_size(space_needed)
      unless options[:force]
        timings['disk_space_check'] = PeBackupTools::Utils.benchmark do
          Puppet.info(_('Checking for disk space to restore.'))
          free_space, mount_point = PeBackupTools::Utils.extract_available_space_and_mount_point(tempdir)
          disk_space_interaction(space_needed, free_space, mount_point)
        end
      end

      pre_restore_summary = _(<<-SUMMARY
Restoring backup file %{path}
  Scope: %{scope}
  Backup size: %{space_needed}
  Backup PE version: %{backup_pe_version}

Restoring to:
  Primary: %{certname}
  PE version: %{restore_pe_version}

Temp dir set to:
  %{tempdir}

This typically takes about 20 minutes, but in some cases could take several hours.
SUMMARY
      ) % {
        path: archive_path,
        scope: scope_config[:scope].join(', ').chomp,
        space_needed: humanized_space_needed,
        backup_pe_version: backup_metadata[:pe_version],
        certname: restore_certname,
        restore_pe_version: PeBackupTools::Utils.get_pe_version,
        tempdir: tempdir.to_s,
      }

      Puppet.notice(pre_restore_summary)
      pre_restore_interaction unless options[:force]

      Puppet.info(_("Log messages will be saved to %{logfile}\n") % { logfile: logfile })

      r10k_remote = nil
      preexisting_replica = nil
      compilers = nil

      with_agent_disabled('puppet backup restore in progress') do
        wait_for_agent_lock

        timings['stop_services'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Stopping PE related services'))
          stop_pe_services(options)
        end

        if scope_config[:scope].include?('certs')
          # The previous PE install has agent certificates hanging around and we
          # need to remove them if we're restoring certs
          step_printer.step(_('Cleaning the agent certificates from previous PE install'))
          remove_puppet_ssl_dir
          # Always remove this dir. If the backup contains an unmigrated CA, then we don't want
          # the new CA dir to still stick around. The user will need to migrate it afterwards.
          remove_puppetserver_ca_dir
        end

        pe_databases = scope_config[:included_dbs]
        begin
          timings['file_system_restore'] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Restoring PE file system components'))
            unpack_tarfile(archive_path, backup_scope, scope_config, tempdir)
            # When versioned deploys is enabled, this symlink will exist, but if the backup
            # tarball does not contain the directory it points to, it will be broken
            # and cause problems during the restore. Delete the symlink so it can get recreated
            # if the symlink is broken.
            codedir_symlink = '/etc/puppetlabs/puppetserver/code'
            FileUtils.rm_f(codedir_symlink) if File.symlink?(codedir_symlink) && !File.exist?(codedir_symlink)
          end
          pe_databases.each do |db|
            timings["#{db.tr('-', '_')}_restore"] = PeBackupTools::Utils.benchmark do
              step_printer.step(_('Restoring the %{db} database') % { db: db })
              restore_database(db, tempdir)
            end
          end
        ensure
          # This is a fail safe in case we fail to run restore_database on
          # all the databases which should clean up each binary individually
          tmp_db_dumps = pe_databases.map { |db| "#{tempdir}/#{db}.bin" }
          FileUtils.rm_f(tmp_db_dumps)
        end

        if scope_config[:scope].include?('config') && restore_certname != backup_certname
          timings['config_migration'] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Migrating PE configuration for new server'))
            migrate_puppet_conf(restore_certname, options)
            migrate_pe_conf(restore_certname)
            migrate_user_data_conf(restore_certname)
          end

          if scope_config[:scope].include?('certs')
            remove_old_certs(restore_certname)
          end
        end

        timings['puppet_infrastructure_configure'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Configuring PE on newly restored primary'))
          environment = options[:pe_environment] || Puppet[:environment]
          puppet_infrastructure_configure(environment, options)
        end

        if scope_config[:scope].include?('config') && restore_certname != backup_certname
          timings['classification_updates'] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Updating PE Classification for new server'))
            preexisting_replica = update_pe_classification(backup_certname, restore_certname)
            r10k_remote = `/opt/puppetlabs/bin/puppet lookup puppet_enterprise::profile::master::r10k_remote`
            compilers = get_compilers(backup_certname, restore_certname, preexisting_replica, options)
            migrate_puppet_conf(restore_certname, options)
          end
        end

        if scope_config[:scope].include?('puppetdb') && restore_certname != backup_certname
          timings['deactivate_old_master'] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Deactivating the old primary node'))
            puppet_node_deactivate(backup_certname, options)
          end
        end
      end

      timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))

      cert_conf = PeBackupTools::Utils.get_agent_cert_conf(restore_certname)
      analytics_url = "https://#{restore_certname}:8140/analytics/v1"
      PeBackupTools::Utils::AnalyticsClient.store_event(analytics_url, cert_conf, 'puppet-backup.restore',
                                                        scope: scope_config[:scope],
                                                        force: options[:force] ? true : false,
                                                        timings: timings)

      Puppet.notice(_('Restore succeeded (size: %{space_needed}, scope: %{scope}, time: %{timings})') % { space_needed: humanized_space_needed, scope: scope_config[:scope], timings: PeBackupTools::Utils.humanize_time(timings[:total]) })
      {
        command: 'restore',
        scope: scope_config[:scope],
        size: space_needed,
        runtime: timings,
        preexisting_replica: preexisting_replica,
        r10k_remote: r10k_remote,
        compilers: compilers,
        restore_certname: restore_certname,
      }
    rescue RuntimeError => e
      error(_('Restore failed with: %{error}') % { error: e.message })
    ensure
      restore_pe_version_file(tempdir)
      FileUtils.remove_entry(tempdir)
    end
  end
end
